#!/usr/bin/env perl
use strict;
use warnings;

use Test::More;
use Math::Prime::Util qw/stirling/;

my $extra = defined $ENV{EXTENDED_TESTING} && $ENV{EXTENDED_TESTING};

plan tests => 4;

subtest 'input validations', sub {
  # We are using UV prototypes, so don't catch this.
  #ok(!defined eval { stirling(-4, -3) }, "Expected fail: stirling with negative args");
  ok(!defined eval { stirling(4,3,4) }, "Expected fail: stirling type 4");
};

subtest 'stirling numbers of the first kind', sub {
  my @stirling1 = (
    [qw/1 0/],
    [qw/0 1 0/],
    [qw/0 -1 1 0/],
    [qw/0 2 -3 1 0/],
    [qw/0 -6 11 -6 1 0/],
    [qw/0 24 -50 35 -10 1 0/],
    [qw/0 -120 274 -225 85 -15 1 0/],
    [qw/0 720 -1764 1624 -735 175 -21 1 0/],
    [qw/0 -5040 13068 -13132 6769 -1960 322 -28 1 0/],
    [qw/0 40320 -109584 118124 -67284 22449 -4536 546 -36 1 0/],
    [qw/0 -362880 1026576 -1172700 723680 -269325 63273 -9450 870 -45 1 0/],
    [qw/0 3628800 -10628640 12753576 -8409500 3416930 -902055 157773 -18150 1320 -55 1 0/],
    [qw/0 -39916800 120543840 -150917976 105258076 -45995730 13339535 -2637558 357423 -32670 1925 -66 1 0/],
    [qw/0 479001600 -1486442880 1931559552 -1414014888 657206836 -206070150 44990231 -6926634 749463 -55770 2717 -78 1 0/],
    [qw/0 -6227020800 19802759040 -26596717056 20313753096 -9957703756 3336118786 -790943153 135036473 -16669653 1474473 -91091 3731 -91 1 0/],
    [qw/0 87178291200 -283465647360 392156797824 -310989260400 159721605680 -56663366760 14409322928 -2681453775 368411615 -37312275 2749747 -143325 5005 -105 1 0/],
    [qw/0 -1307674368000 4339163001600 -6165817614720 5056995703824 -2706813345600 1009672107080 -272803210680 54631129553 -8207628000 928095740 -78558480 4899622 -218400 6580 -120 1 0/],
    [qw/0 20922789888000 -70734282393600 102992244837120 -87077748875904 48366009233424 -18861567058880 5374523477960 -1146901283528 185953177553 -23057159840 2185031420 -156952432 8394022 -323680 8500 -136 1 0/],
    [qw/0 -355687428096000 1223405590579200 -1821602444624640 1583313975727488 -909299905844112 369012649234384 -110228466184200 24871845297936 -4308105301929 577924894833 -60202693980 4853222764 -299650806 13896582 -468180 10812 -153 1 0/],
    [qw/0 6402373705728000 -22376988058521600 34012249593822720 -30321254007719424 17950712280921504 -7551527592063024 2353125040549984 -557921681547048 102417740732658 -14710753408923 1661573386473 -147560703732 10246937272 -549789282 22323822 -662796 13566 -171 1 0/],
    [qw/0 -121645100408832000 431565146817638400 -668609730341153280 610116075740491776 -371384787345228000 161429736530118960 -52260903362512720 12953636989943896 -2503858755467550 381922055502195 -46280647751910 4465226757381 -342252511900 20692933630 -973941900 34916946 -920550 16815 -190 1 0/],
  );
  $#stirling1 = 12 unless $extra;
  my $n = 0;
  foreach my $narr (@stirling1) {
    my @s1 = map { stirling($n,$_,1) } 0..$n+1;
    is_deeply( \@s1, $narr, "Stirling 1: s($n,0..". ($n+1) .")" );
    $n++;
  }

  SKIP: {
    skip "stirling(132,67) only with EXTENDED_TESTING", 1 unless $extra;
    is( "".stirling(132,67,1), '-6132458966070920781607687809239433538883836871765225500351514785120957322534135782514155513931693375104995311496306605620444680401484569675682191339176710', "s(132,67)" );
  }
};

subtest 'stirling numbers of the second kind', sub {
  # Generated by gp 2.8.0: for(n=0,20,printf("[qw/");for(m=0,n+1,printf("%d ",stirling(n,m,2)));printf("/],\n"))
  my @stirling2 = (
    [qw/1 0/],
    [qw/0 1 0/],
    [qw/0 1 1 0/],
    [qw/0 1 3 1 0/],
    [qw/0 1 7 6 1 0/],
    [qw/0 1 15 25 10 1 0/],
    [qw/0 1 31 90 65 15 1 0/],
    [qw/0 1 63 301 350 140 21 1 0/],
    [qw/0 1 127 966 1701 1050 266 28 1 0/],
    [qw/0 1 255 3025 7770 6951 2646 462 36 1 0/],
    [qw/0 1 511 9330 34105 42525 22827 5880 750 45 1 0/],
    [qw/0 1 1023 28501 145750 246730 179487 63987 11880 1155 55 1 0/],
    [qw/0 1 2047 86526 611501 1379400 1323652 627396 159027 22275 1705 66 1 0/],
    [qw/0 1 4095 261625 2532530 7508501 9321312 5715424 1899612 359502 39325 2431 78 1 0/],
    [qw/0 1 8191 788970 10391745 40075035 63436373 49329280 20912320 5135130 752752 66066 3367 91 1 0/],
    [qw/0 1 16383 2375101 42355950 210766920 420693273 408741333 216627840 67128490 12662650 1479478 106470 4550 105 1 0/],
    [qw/0 1 32767 7141686 171798901 1096190550 2734926558 3281882604 2141764053 820784250 193754990 28936908 2757118 165620 6020 120 1 0/],
    [qw/0 1 65535 21457825 694337290 5652751651 17505749898 25708104786 20415995028 9528822303 2758334150 512060978 62022324 4910178 249900 7820 136 1 0/],
    [qw/0 1 131071 64439010 2798806985 28958095545 110687251039 197462483400 189036065010 106175395755 37112163803 8391004908 1256328866 125854638 8408778 367200 9996 153 1 0/],
    [qw/0 1 262143 193448101 11259666950 147589284710 693081601779 1492924634839 1709751003480 1144614626805 477297033785 129413217791 23466951300 2892439160 243577530 13916778 527136 12597 171 1 0/],
    [qw/0 1 524287 580606446 45232115901 749206090500 4306078895384 11143554045652 15170932662679 12011282644725 5917584964655 1900842429486 411016633391 61068660380 6302524580 452329200 22350954 741285 15675 190 1 0/],
  );
  $#stirling2 = 12 unless $extra;
  my $n = 0;
  foreach my $narr (@stirling2) {
    my @s2 = map { stirling($n,$_,2) } 0..$n+1;
    is_deeply( \@s2, $narr, "Stirling 2: S($n,0..". ($n+1) .")" );
    $n++;
  }

  SKIP: {
    skip "large stirling tests only with EXTENDED_TESTING", 1 unless $extra;
    is( "".stirling(114,85,2), '722095587897382907118640452680242028195738761915144254970925658656935934040', "S(114,85)" );
    my $sum = 0;
    $sum += $_ for split(//, stirling(234,96));
    is($sum, 1504, "sumdigits(stirling(234,96) = 1504");
  }
};

subtest 'stirling numbers of the third kind', sub {
  # Generated by gp 2.8.0:
  #   lah(n,k)={n==0&&k==0&&return(1);(n==0||m==0||k>n)&&return(0);binomial(n,k)*binomial(n-1,k-1)*(n-k)!}
  #   for(n=0,20,printf("[qw/");for(m=0,n+1,printf("%d ",lah(n,m)));printf("/],\n"))
  my @stirling3 = (
    [qw/1 0 /],
    [qw/0 1 0 /],
    [qw/0 2 1 0 /],
    [qw/0 6 6 1 0 /],
    [qw/0 24 36 12 1 0 /],
    [qw/0 120 240 120 20 1 0 /],
    [qw/0 720 1800 1200 300 30 1 0 /],
    [qw/0 5040 15120 12600 4200 630 42 1 0 /],
    [qw/0 40320 141120 141120 58800 11760 1176 56 1 0 /],
    [qw/0 362880 1451520 1693440 846720 211680 28224 2016 72 1 0 /],
    [qw/0 3628800 16329600 21772800 12700800 3810240 635040 60480 3240 90 1 0 /],
    [qw/0 39916800 199584000 299376000 199584000 69854400 13970880 1663200 118800 4950 110 1 0 /],
    [qw/0 479001600 2634508800 4390848000 3293136000 1317254400 307359360 43908480 3920400 217800 7260 132 1 0 /],
    [qw/0 6227020800 37362124800 68497228800 57081024000 25686460800 6849722880 1141620480 122316480 8494200 377520 10296 156 1 0 /],
    [qw/0 87178291200 566658892800 1133317785600 1038874636800 519437318400 155831195520 29682132480 3710266560 309188880 17177160 624624 14196 182 1 0 /],
    [qw/0 1307674368000 9153720576000 19833061248000 19833061248000 10908183686400 3636061228800 779155977600 111307996800 10821610800 721440720 32792760 993720 19110 210 1 0 /],
    [qw/0 20922789888000 156920924160000 366148823040000 396661224960000 237996734976000 87265469491200 20777492736000 3339239904000 371026656000 28857628800 1574052480 59623200 1528800 25200 240 1 0 /],
    [qw/0 355687428096000 2845499424768000 7113748561920000 8299373322240000 5394592659456000 2157837063782400 565147802419200 100919250432000 12614906304000 1121325004800 71357045760 3243502080 103958400 2284800 32640 272 1 0 /],
    [qw/0 6402373705728000 54420176498688000 145120470663168000 181400588328960000 126980411830272000 55024845126451200 15721384321843200 3088129063219200 428906814336000 42890681433600 3119322286080 165418606080 6362254080 174787200 3329280 41616 306 1 0 /],
    [qw/0 121645100408832000 1094805903679488000 3101950060425216000 4135933413900288000 3101950060425216000 1447576694865100800 448059453172531200 96012739965542400 14668613050291200 1629845894476800 133351027729920 8081880468480 362648482560 11955444480 284653440 4744224 52326 342 1 0 /],
    [qw/0 2432902008176640000 23112569077678080000 69337707233034240000 98228418580131840000 78582734864105472000 39291367432052736000 13097122477350912000 3040403432242176000 506733905373696000 61934143990118400 5630376726374400 383889322252800 19686631910400 757178150400 21633661440 450701280 6627960 64980 380 1 0 /],
  );
  my $n = 0;
  foreach my $narr (@stirling3) {
    my @s3 = map { "".stirling($n,$_,3) } 0..$n+1;
    is_deeply( \@s3, $narr, "Stirling 3: L($n,0..". ($n+1) .")" );
    $n++;
  }
};
