from __future__ import annotations
from typing import Any, Dict, List
from PySide6.QtWidgets import QTreeWidgetItem
from PySide6.QtGui import QIcon
from src.utils.tool_card.tool_widgets import ToolCard
from src.path_resolver import PathResolver
from src.controller.docker_controller import DockerController
from src.utils.tool_card.helpers.timer_setup import setup_timer


class ToolListManager:
    """
    It controls how the Docker tool list is displayed in the UI.
    """
    def __init__(self, ui: Any, tools: List[Dict[str, str]]) -> None:
        self.ui = ui
        self.tools = tools
        self.tool_items: Dict[str, QTreeWidgetItem] = {}
        self.setup_tool_list()

    def setup_tool_list(self) -> None:
        """
        Configure the Docker tools tree view and set the stacked widget to display details.
        """
        docker_tools_item = QTreeWidgetItem(["Docker Tools"])
        docker_icon_path = PathResolver.resource_path('resources/assets/docker.png')
        docker_tools_item.setIcon(0, QIcon(docker_icon_path))
        self.ui.tools_list.addTopLevelItem(docker_tools_item)

        # Refresh docker cache once at startup to avoid per-tool blocking calls
        DockerController.refresh_cache(force=True)

        # Add the tools to the tree by iterating on them
        # and creating the associated detail widget.
        for tool in self.tools:
            tool_name = tool["name"]
            tool_item = QTreeWidgetItem([tool_name])
            docker_tools_item.addChild(tool_item)
            self.tool_items[tool_name] = tool_item

            card = ToolCard(tool, tool_item, self)
            card.run_button.clicked.connect(card.run_container)
            card.stop_button.clicked.connect(card.stop_container)

            # Check the status of the container and update the interface.
            DockerController.check_container_status(
                tool, tool_item, card.run_button
            )
            card.tool_item = tool_item

            self.ui.tool_cards.append(card)
            self.ui.tool_details.addWidget(card)

            # Configure the timer for periodic updates (e.g. container status)
            setup_timer(card)

        self.ui.tools_list.setFixedWidth(170)
        self.ui.tools_list.currentItemChanged.connect(self.on_tool_selection_changed)
        self.ui.tools_list.expandAll()

    def on_tool_selection_changed(self, current: QTreeWidgetItem | None) -> None:
        """
        It handles the change of element selection in the tools tree
        and updates the display widget accordingly.
        """
        if current is not None and current.parent() is not None:
            index = current.parent().indexOfChild(current)
            self.ui.tool_details.setCurrentIndex(index)
