"""
PathResolver ensures that resources (such as configuration files, images, data files, etc.)
are correctly located both during development and after the Python application is deployed with PyInstaller.
"""

import sys
import os


class PathResolver:
    @staticmethod
    def resource_path(relative_path: str) -> str:
        """
        Get absolute path to resource, works for dev and for PyInstaller.
        """
        try:
            """
            If sys._MEIPASS is available, the method sets base_path to sys._MEIPASS, 
            which represents the absolute path to the temporary directory containing 
            the executable files generated by PyInstaller.
            """
            base_path = sys._MEIPASS
        except AttributeError:
            """
            If sys._MEIPASS is not available (e.g., during development), the method sets base_path 
            to the absolute path of the current directory of the running Python file 
            """
            base_path = os.path.abspath(os.path.dirname(__file__))
        return os.path.join(base_path, relative_path)
