from PySide6.QtWidgets import QWidget, QVBoxLayout, QLineEdit
from src.utils.tool_card.helpers.layout_components import (
    build_layout,
    usage_examples_section,
    container_buttons_section,
)
from src.controller.docker_controller import DockerController


class ToolCard(QWidget):
    """
    This widget displays detailed information about a specific tool,
    Displays a tool's details, usage examples, and Docker controls.
    Periodically updates the tool's image download status.
    """
    def __init__(self, tool, tool_item, context):
        """
        Initializes UI components and sets up the image update timer.
        """
        super().__init__()
        self.tool = tool
        self.tool_item = tool_item
        self.context = context
        self.params = None
        self.run_button = None
        self.stop_button = None
        self.download_status_label = None
        layout = QVBoxLayout()
        details_layout = build_layout(self)
        layout.addLayout(details_layout)
        usage_examples_section(self, layout)
        self.input_params(layout)
        container_buttons_section(self, layout)
        
        self.run_button.clicked.connect(self.run_container)
        self.stop_button.clicked.connect(self.stop_container)

        self.setLayout(layout)

    def update_download_status(self):
        DockerController.check_container_status(self.tool, self.tool_item, self.run_button)

    def stop_container(self):
        DockerController.stop_container(self.tool, self)

    def run_container(self):
        DockerController.run_container(self.context)

    @staticmethod
    def get_placeholder_text(tool_name: str) -> str:
        """
        Set the placeholder text for the parameter input field based on the tool name.
        """
        placeholders = {
            "nmap": "Enter arguments (e.g., -p 80 192.168.0.1)",
            "bettercap": "Enter options (e.g., --proxy)",
            "sqlmap": "Enter arguments (e.g., -u 'http://example.com')",
            "netexec": "Enter options (e.g., --help)",
            "evil-winrm": "Enter options (e.g., --help)",
            "wpscan": "Enter options (e.g., --help)",
            "katana": "Enter options (e.g., --help)",
            "subfinder": "Enter options (e.g., --help",
            "tshark": "Enter arguments (e.g., -h)",
        }
        return placeholders.get(tool_name, "Enter options for the tool")

    def input_params(self, layout) -> None:
        """
        Create and add the parameters input field to the provided layout.
        """
        if self.tool["name"].lower() in ["nmap", "bettercap", "sqlmap", "netexec", "evil-winrm", "wpscan", "katana", "subfinder", "tshark"]:
            self.params = QLineEdit(self)
            self.params.setPlaceholderText(self.get_placeholder_text(self.tool["name"].lower()))
            layout.addWidget(self.params)

    def get_params(self) -> str:
        return self.params.text() if self.params else ""
