﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/location/LocationService_EXPORTS.h>
#include <aws/location/model/ApiKeyRestrictions.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Json {
class JsonValue;
}  // namespace Json
}  // namespace Utils
namespace LocationService {
namespace Model {
class DescribeKeyResult {
 public:
  AWS_LOCATIONSERVICE_API DescribeKeyResult() = default;
  AWS_LOCATIONSERVICE_API DescribeKeyResult(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);
  AWS_LOCATIONSERVICE_API DescribeKeyResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);

  ///@{
  /**
   * <p>The key value/string of an API key.</p>
   */
  inline const Aws::String& GetKey() const { return m_key; }
  template <typename KeyT = Aws::String>
  void SetKey(KeyT&& value) {
    m_keyHasBeenSet = true;
    m_key = std::forward<KeyT>(value);
  }
  template <typename KeyT = Aws::String>
  DescribeKeyResult& WithKey(KeyT&& value) {
    SetKey(std::forward<KeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) for the API key resource. Used when you need
   * to specify a resource across all Amazon Web Services.</p> <ul> <li> <p>Format
   * example: <code>arn:aws:geo:region:account-id:key/ExampleKey</code> </p> </li>
   * </ul>
   */
  inline const Aws::String& GetKeyArn() const { return m_keyArn; }
  template <typename KeyArnT = Aws::String>
  void SetKeyArn(KeyArnT&& value) {
    m_keyArnHasBeenSet = true;
    m_keyArn = std::forward<KeyArnT>(value);
  }
  template <typename KeyArnT = Aws::String>
  DescribeKeyResult& WithKeyArn(KeyArnT&& value) {
    SetKeyArn(std::forward<KeyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the API key resource.</p>
   */
  inline const Aws::String& GetKeyName() const { return m_keyName; }
  template <typename KeyNameT = Aws::String>
  void SetKeyName(KeyNameT&& value) {
    m_keyNameHasBeenSet = true;
    m_keyName = std::forward<KeyNameT>(value);
  }
  template <typename KeyNameT = Aws::String>
  DescribeKeyResult& WithKeyName(KeyNameT&& value) {
    SetKeyName(std::forward<KeyNameT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const ApiKeyRestrictions& GetRestrictions() const { return m_restrictions; }
  template <typename RestrictionsT = ApiKeyRestrictions>
  void SetRestrictions(RestrictionsT&& value) {
    m_restrictionsHasBeenSet = true;
    m_restrictions = std::forward<RestrictionsT>(value);
  }
  template <typename RestrictionsT = ApiKeyRestrictions>
  DescribeKeyResult& WithRestrictions(RestrictionsT&& value) {
    SetRestrictions(std::forward<RestrictionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp for when the API key resource was created in <a
   * href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a>
   * format: <code>YYYY-MM-DDThh:mm:ss.sssZ</code>. </p>
   */
  inline const Aws::Utils::DateTime& GetCreateTime() const { return m_createTime; }
  template <typename CreateTimeT = Aws::Utils::DateTime>
  void SetCreateTime(CreateTimeT&& value) {
    m_createTimeHasBeenSet = true;
    m_createTime = std::forward<CreateTimeT>(value);
  }
  template <typename CreateTimeT = Aws::Utils::DateTime>
  DescribeKeyResult& WithCreateTime(CreateTimeT&& value) {
    SetCreateTime(std::forward<CreateTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp for when the API key resource will expire in <a
   * href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a>
   * format: <code>YYYY-MM-DDThh:mm:ss.sssZ</code>. </p>
   */
  inline const Aws::Utils::DateTime& GetExpireTime() const { return m_expireTime; }
  template <typename ExpireTimeT = Aws::Utils::DateTime>
  void SetExpireTime(ExpireTimeT&& value) {
    m_expireTimeHasBeenSet = true;
    m_expireTime = std::forward<ExpireTimeT>(value);
  }
  template <typename ExpireTimeT = Aws::Utils::DateTime>
  DescribeKeyResult& WithExpireTime(ExpireTimeT&& value) {
    SetExpireTime(std::forward<ExpireTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp for when the API key resource was last updated in <a
   * href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a>
   * format: <code>YYYY-MM-DDThh:mm:ss.sssZ</code>. </p>
   */
  inline const Aws::Utils::DateTime& GetUpdateTime() const { return m_updateTime; }
  template <typename UpdateTimeT = Aws::Utils::DateTime>
  void SetUpdateTime(UpdateTimeT&& value) {
    m_updateTimeHasBeenSet = true;
    m_updateTime = std::forward<UpdateTimeT>(value);
  }
  template <typename UpdateTimeT = Aws::Utils::DateTime>
  DescribeKeyResult& WithUpdateTime(UpdateTimeT&& value) {
    SetUpdateTime(std::forward<UpdateTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The optional description for the API key resource.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  DescribeKeyResult& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags associated with the API key resource.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  DescribeKeyResult& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  DescribeKeyResult& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  DescribeKeyResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_key;

  Aws::String m_keyArn;

  Aws::String m_keyName;

  ApiKeyRestrictions m_restrictions;

  Aws::Utils::DateTime m_createTime{};

  Aws::Utils::DateTime m_expireTime{};

  Aws::Utils::DateTime m_updateTime{};

  Aws::String m_description;

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::String m_requestId;
  bool m_keyHasBeenSet = false;
  bool m_keyArnHasBeenSet = false;
  bool m_keyNameHasBeenSet = false;
  bool m_restrictionsHasBeenSet = false;
  bool m_createTimeHasBeenSet = false;
  bool m_expireTimeHasBeenSet = false;
  bool m_updateTimeHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace LocationService
}  // namespace Aws
