// SPDX-License-Identifier: GPL-2.0-or-later
// Copyright The Music Player Daemon Project

#ifndef MPD_NEIGHBOR_EXPLORER_HXX
#define MPD_NEIGHBOR_EXPLORER_HXX

#include <forward_list>

class NeighborListener;
struct NeighborInfo;

/**
 * An object that explores the neighborhood for music servers.
 *
 * As soon as this object is opened, it will start exploring, and
 * notify the #NeighborListener when it found or lost something.
 *
 * The implementation is supposed to be non-blocking.  This can be
 * implemented either using the #EventLoop instance that was passed to
 * the NeighborPlugin or by moving the blocking parts in a dedicated
 * thread.
 */
class NeighborExplorer {
protected:
	NeighborListener &listener;

	explicit NeighborExplorer(NeighborListener &_listener) noexcept
		:listener(_listener) {}

public:
	typedef std::forward_list<NeighborInfo> List;

	/**
	 * Free instance data.
         */
	virtual ~NeighborExplorer() noexcept = default;

	/**
         * Start exploring the neighborhood.
	 *
	 * Throws std::runtime_error on error.
	 */
	virtual void Open() = 0;

	/**
         * Stop exploring.
	 */
	virtual void Close() noexcept = 0;

	/**
	 * Obtain a list of currently known neighbors.
	 */
	virtual List GetList() const noexcept = 0;
};

#endif
