/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * a_file.h - include file for clients of a_file.c
 */

#ifndef AUDIO_FILE_H
#define AUDIO_FILE_H 1

#if HAVE_LIBMPG123
#include <mpg123.h>
#endif
#if HAVE_LIBSNDFILE
#include <sndfile.h>
#endif
#if HAVE_LIBAV
#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>
#endif
#if HAVE_LIBSOX
#include <sox.h>
#endif

enum af_lib {
    /* Set to positive values so that, if none are available, fail gracefully */
#if HAVE_LIBMPG123
    libmpg123 = 1,
#endif
#if HAVE_LIBSNDFILE
    libsndfile = 2,
#endif
#if HAVE_LIBAV
    libav = 3,
#endif
#if HAVE_LIBSOX
    libsox = 4,
#endif
};

typedef struct audio_file {
	char *filename;
	freq_t sample_rate;
	frames_t frames;	/* The file has (frames*channels) samples */
	int channels;
	enum af_lib opened_with; /* Which library was this file opened by? */
#if HAVE_LIBSNDFILE
	/* libsndfile handle, NULL if not using libsndfile */
	SNDFILE *sndfile;
#endif
#if HAVE_LIBMPG123
	/* libmpg123 handle, NULL if not using libmpg123 */
	mpg123_handle *mh;
	/* What mpg123_getformat returned, in case it changes mid-stream */
	long mpg123_rate;
	int mpg123_channels;
	int mpg123_encoding;
#endif
#if HAVE_LIBSOX
	sox_format_t *sox_format;
#endif
} audio_file_t;

typedef enum {
	af_float,   /* mono floats */
	af_signed,  /* 16-bit native endian, same number of channels as the input file */
} af_format_t;

extern audio_file_t *current_audio_file(void);

/* Return a handle for the audio file, NULL on failure */
extern audio_file_t *open_audio_file(char *filename);

extern frames_t read_audio_file(audio_file_t *af, char *data,
				af_format_t format, int channels,
				frames_t start,    /* Offset from time 0 */
				frames_t nframes);

extern void close_audio_file(audio_file_t *audio_file);

/* Utility functions */

/* The length of an audio file in seconds */
extern secs_t audio_file_length(void);

/* What's the sample rate of the audio file? */
extern freq_t sr;

/* Dump from the left to the right bar marker */
extern bool dump_to_audio_file(char *filename, frames_t start, frames_t nframes);

#endif
