/*
 * diffpng - Compare two PNG files.
 *
 * Usage: diffpng [-f fuzz] file1.png file2.png
 * -f: Allow a difference of "fuzz" in color values (default: exact)
 *
 * Like cmp and friends, exit 0 if they are identical, non-zero otherwise.
 *
 *	Martin Guy <martinwguy@gmail.com>, January 2024.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <png.h>

static int open_png(char *file, png_structp *ptrp, png_infop *infop,
		     png_bytepp *rowsp, png_uint_32 *widthp, png_uint_32 *heightp,
		     int *depthp, int *colorp);

void
main(int argc, char **argv)
{
    char *file1, *file2;
    png_structp ptr1, ptr2;
    png_infop info1, info2;
    png_bytepp rows1, rows2;
    png_uint_32 width1, width2;
    png_uint_32 height1, height2;
    int depth1, depth2;
    int color1, color2;
    png_uint_32 x, y;
    int fuzz = 0;

    if (argc >= 3 && strcmp(argv[1], "-f") == 0) {
	fuzz = atoi(argv[2]);
	argc -= 2;
	argv += 2;
    }
    if (argc != 3) {
	fprintf(stderr, "Usage: diffpng file1.png file2.png\n");
	exit(1);
    }
    file1 = argv[1];
    file2 = argv[2];
    if (!open_png(file1, &ptr1, &info1, &rows1, &width1, &height1, &depth1, &color1))
	exit(1);
    if (!open_png(file2, &ptr2, &info2, &rows2, &width2, &height2, &depth2, &color2))
	exit(1);

    if (width1 != width2 ||
        height1 != height2 ||
	depth1 != depth2 ||
	color1 != color2) {
	    fprintf(stderr, "%s%s%s%sdiffer\n",
		width1 != width2 ? "width " : "",
		height1 != height2 ? "height " : "",
		depth1 != depth2 ? "depth " : "",
		color1 != color2 ? "color-space " : "");
	    exit(1);
    }

    for (y=0; y < height1; y++)
	for (x=0; x < width1; x++)
	    if (abs((rows1[y][x] & 0xFF) - (rows2[y][x] & 0xFF)) > fuzz ||
		abs(((rows1[y][x] >> 8) & 0xFF) - ((rows2[y][x] >> 8) & 0xFF)) > fuzz ||
		abs(((rows1[y][x] >> 16) & 0xFF) - ((rows2[y][x] >> 16) & 0xFF)) > fuzz)
		exit(1);

    exit(0);
}

/* Common code to open the two PNG files.
 * Returns 1 of sucess, 0 on failure */
static int
open_png(char *file, png_structp *ptrp, png_infop *infop,
	 png_bytepp *rowsp, png_uint_32 *widthp, png_uint_32 *heightp,
	 int *depthp, int *colorp)
{
    char header[8];
    FILE *fp;

    /* Read file */
    if (strcmp(file, "-") == 0) fp = stdin;
    else fp = fopen(file, "rb");
    if (fp == NULL) {
	    fprintf(stderr, "Cannot fopen ");
	    perror(file);
	    return 0;
    }
    if (fread(header, 1, 8, fp) != 8) {
	    fprintf(stderr, "Cannot fread %s\n", file);
	    return 0;
    }
    if (png_sig_cmp(header, 0, 8) != 0) {
	    fprintf(stderr, "Cannot png_sig_cmp %s\n", file);
	    return 0;
    }
    if (!(*ptrp = png_create_read_struct(PNG_LIBPNG_VER_STRING, 0, 0, 0))) {
	    fprintf(stderr, "Cannot png_create_read_struct %s\n", file);
	    return 0;
    }
    if (!(*infop = png_create_info_struct(*ptrp))) {
	    fprintf(stderr, "Cannot png_create_info_struct %s\n", file);
	    return 0;
    }
    if (setjmp(png_jmpbuf(*ptrp))) {
	    fprintf(stderr, "Cannot png_jmpbuf %s\n", file);
	    return 0;
    }
    png_init_io(*ptrp, fp);
    png_set_sig_bytes(*ptrp, 8);
    png_read_png(*ptrp, *infop, 0, NULL);
    *rowsp = png_get_rows(*ptrp, *infop);
    png_get_IHDR(*ptrp, *infop, widthp, heightp, depthp, colorp,
		 NULL, NULL, NULL);
    return 1;
}
