/*
 * Decompiled with CFR 0.152.
 */
package org.mapdb.volume;

import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedByInterruptException;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.HashSet;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.mapdb.DBException;
import org.mapdb.DataIO;
import org.mapdb.DataInput2;
import org.mapdb.volume.Volume;
import org.mapdb.volume.VolumeFactory;

public final class FileChannelVol
extends Volume {
    public static final VolumeFactory FACTORY = new VolumeFactory(){

        @Override
        public Volume makeVolume(String file, boolean readOnly, long fileLockWait, int sliceShift, long initSize, boolean fixedSize) {
            return new FileChannelVol(new File(file), readOnly, fileLockWait, sliceShift, initSize);
        }

        @Override
        @NotNull
        public boolean exists(@Nullable String file) {
            return new File(file).exists();
        }

        @Override
        public boolean handlesReadonly() {
            return true;
        }
    };
    protected final File file;
    protected final int sliceSize;
    protected FileChannel channel;
    protected final boolean readOnly;
    protected final FileLock fileLock;
    protected volatile long size;
    protected final Lock growLock = new ReentrantLock();

    public FileChannelVol(File file, boolean readOnly, long fileLockWait, int sliceShift, long initSize) {
        this.file = file;
        this.readOnly = readOnly;
        this.sliceSize = 1 << sliceShift;
        HashSet<StandardOpenOption> options = new HashSet<StandardOpenOption>();
        options.add(StandardOpenOption.READ);
        if (!readOnly) {
            options.add(StandardOpenOption.WRITE);
            options.add(StandardOpenOption.CREATE);
        }
        try {
            long oldSize;
            FileChannelVol.checkFolder(file, readOnly);
            if (readOnly && !file.exists()) {
                this.channel = null;
                this.size = 0L;
            } else {
                this.channel = FileChannel.open(file.toPath(), options, new FileAttribute[0]);
                this.size = this.channel.size();
            }
            this.fileLock = Volume.lockFile(file, this.channel, readOnly, fileLockWait);
            if (initSize != 0L && !readOnly && initSize > (oldSize = this.channel.size())) {
                this.clear(oldSize, initSize);
            }
        }
        catch (ClosedByInterruptException e) {
            throw new DBException.VolumeClosedByInterrupt(e);
        }
        catch (ClosedChannelException e) {
            throw new DBException.VolumeClosed(e);
        }
        catch (IOException e) {
            throw new DBException.VolumeIOError(e);
        }
    }

    public FileChannelVol(File file) {
        this(file, false, 0L, 20, 0L);
    }

    protected static void checkFolder(File file, boolean readOnly) throws IOException {
        File parent = file.getParentFile();
        if (parent == null) {
            parent = file.getCanonicalFile().getParentFile();
        }
        if (parent == null) {
            throw new IOException("Parent folder could not be determined for: " + file);
        }
        if (!parent.exists() || !parent.isDirectory()) {
            throw new IOException("Parent folder does not exist: " + file);
        }
        if (!parent.canRead()) {
            throw new IOException("Parent folder is not readable: " + file);
        }
        if (!readOnly && !parent.canWrite()) {
            throw new IOException("Parent folder is not writable: " + file);
        }
    }

    @Override
    public void ensureAvailable(long offset) {
        if ((offset = DataIO.roundUp(offset, (long)this.sliceSize)) > this.size) {
            this.growLock.lock();
            try {
                this.channel.position(offset - 1L);
                this.channel.write(ByteBuffer.allocate(1));
                this.size = offset;
            }
            catch (IOException e) {
                throw new DBException.VolumeIOError(e);
            }
            finally {
                this.growLock.unlock();
            }
        }
    }

    @Override
    public void truncate(long size) {
        this.growLock.lock();
        try {
            this.size = size;
            this.channel.truncate(size);
        }
        catch (ClosedByInterruptException e) {
            throw new DBException.VolumeClosedByInterrupt(e);
        }
        catch (ClosedChannelException e) {
            throw new DBException.VolumeClosed(e);
        }
        catch (IOException e) {
            throw new DBException.VolumeIOError(e);
        }
        finally {
            this.growLock.unlock();
        }
    }

    protected void writeFully(long offset, ByteBuffer buf) {
        try {
            int write;
            for (int remaining = buf.limit() - buf.position(); remaining > 0; remaining -= write) {
                write = this.channel.write(buf, offset);
                if (write >= 0) continue;
                throw new EOFException();
            }
        }
        catch (ClosedByInterruptException e) {
            throw new DBException.VolumeClosedByInterrupt(e);
        }
        catch (ClosedChannelException e) {
            throw new DBException.VolumeClosed(e);
        }
        catch (IOException e) {
            throw new DBException.VolumeIOError(e);
        }
    }

    @Override
    public void putLong(long offset, long value) {
        ByteBuffer buf = ByteBuffer.allocate(8);
        buf.putLong(0, value);
        this.writeFully(offset, buf);
    }

    @Override
    public void putInt(long offset, int value) {
        ByteBuffer buf = ByteBuffer.allocate(4);
        buf.putInt(0, value);
        this.writeFully(offset, buf);
    }

    @Override
    public void putByte(long offset, byte value) {
        ByteBuffer buf = ByteBuffer.allocate(1);
        buf.put(0, value);
        this.writeFully(offset, buf);
    }

    @Override
    public void putData(long offset, byte[] src, int srcPos, int srcSize) {
        ByteBuffer buf = ByteBuffer.wrap(src, srcPos, srcSize);
        this.writeFully(offset, buf);
    }

    @Override
    public void putData(long offset, ByteBuffer buf) {
        this.writeFully(offset, buf);
    }

    protected void readFully(long offset, ByteBuffer buf) {
        try {
            int read;
            for (int remaining = buf.limit() - buf.position(); remaining > 0; remaining -= read) {
                read = this.channel.read(buf, offset);
                if (read >= 0) continue;
                throw new EOFException();
            }
        }
        catch (ClosedByInterruptException e) {
            throw new DBException.VolumeClosedByInterrupt(e);
        }
        catch (ClosedChannelException e) {
            throw new DBException.VolumeClosed(e);
        }
        catch (IOException e) {
            throw new DBException.VolumeIOError(e);
        }
    }

    @Override
    public long getLong(long offset) {
        ByteBuffer buf = ByteBuffer.allocate(8);
        this.readFully(offset, buf);
        return buf.getLong(0);
    }

    @Override
    public int getInt(long offset) {
        ByteBuffer buf = ByteBuffer.allocate(4);
        this.readFully(offset, buf);
        return buf.getInt(0);
    }

    @Override
    public byte getByte(long offset) {
        ByteBuffer buf = ByteBuffer.allocate(1);
        this.readFully(offset, buf);
        return buf.get(0);
    }

    @Override
    public DataInput2.ByteBuffer getDataInput(long offset, int size) {
        ByteBuffer buf = ByteBuffer.allocate(size);
        this.readFully(offset, buf);
        return new DataInput2.ByteBuffer(buf, 0);
    }

    @Override
    public void getData(long offset, byte[] bytes, int bytesPos, int size) {
        ByteBuffer buf = ByteBuffer.wrap(bytes, bytesPos, size);
        this.readFully(offset, buf);
    }

    @Override
    public synchronized void close() {
        try {
            if (!this.closed.compareAndSet(false, true)) {
                return;
            }
            if (this.fileLock != null && this.fileLock.isValid()) {
                this.fileLock.release();
            }
            if (this.channel != null) {
                this.channel.close();
            }
            this.channel = null;
        }
        catch (ClosedByInterruptException e) {
            throw new DBException.VolumeClosedByInterrupt(e);
        }
        catch (ClosedChannelException e) {
            throw new DBException.VolumeClosed(e);
        }
        catch (IOException e) {
            throw new DBException.VolumeIOError(e);
        }
    }

    @Override
    public void sync() {
        try {
            this.channel.force(true);
        }
        catch (ClosedByInterruptException e) {
            throw new DBException.VolumeClosedByInterrupt(e);
        }
        catch (ClosedChannelException e) {
            throw new DBException.VolumeClosed(e);
        }
        catch (IOException e) {
            throw new DBException.VolumeIOError(e);
        }
    }

    @Override
    public int sliceSize() {
        return -1;
    }

    @Override
    public boolean isSliced() {
        return false;
    }

    @Override
    public long length() {
        try {
            return this.channel.size();
        }
        catch (IOException e) {
            throw new DBException.VolumeIOError(e);
        }
    }

    @Override
    public boolean isReadOnly() {
        return this.readOnly;
    }

    @Override
    public File getFile() {
        return this.file;
    }

    @Override
    public boolean getFileLocked() {
        return this.fileLock != null && this.fileLock.isValid();
    }

    @Override
    public void clear(long startOffset, long endOffset) {
        FileChannelVol.clear(this.channel, startOffset, endOffset);
    }

    public static void clear(FileChannel channel, long startOffset, long endOffset) {
        try {
            ByteBuffer b = ByteBuffer.wrap(CLEAR);
            while (startOffset < endOffset) {
                b.rewind();
                b.limit((int)Math.min((long)CLEAR.length, endOffset - startOffset));
                channel.write(b, startOffset);
                startOffset += (long)CLEAR.length;
            }
        }
        catch (IOException e) {
            throw new DBException.VolumeIOError(e);
        }
    }
}

