"use strict";
/*********************************************************************
 * Copyright (c) 2018, 2022 Ericsson and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const path = require("path");
const common_1 = require("../gdb/common");
const utils_1 = require("./utils");
describe('Memory Test Suite', function () {
    let dc;
    let frame;
    const memProgram = path.join(utils_1.testProgramsDir, 'mem');
    const memSrc = path.join(utils_1.testProgramsDir, 'mem.c');
    beforeEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            dc = yield (0, utils_1.standardBeforeEach)();
            yield dc.hitBreakpoint((0, utils_1.fillDefaults)(this.currentTest, {
                program: memProgram,
            }), {
                path: memSrc,
                line: 12,
            });
            const threads = yield dc.threadsRequest();
            // On windows additional threads can exist to handle signals, therefore find
            // the real thread & frame running the user code. The other thread will
            // normally be running code from ntdll or similar.
            loop_threads: for (const thread of threads.body.threads) {
                const stack = yield dc.stackTraceRequest({ threadId: thread.id });
                if (stack.body.stackFrames.length >= 1) {
                    for (const f of stack.body.stackFrames) {
                        if (f.source && f.source.name === 'mem.c') {
                            frame = f;
                            break loop_threads;
                        }
                    }
                }
            }
            // Make sure we found the expected frame
            (0, chai_1.expect)(frame).not.eq(undefined);
        });
    });
    afterEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            yield dc.stop();
        });
    });
    /**
     * Verify that `resp` contains the bytes `expectedBytes` and the
     * `expectedAddress` start address.
     *
     * `expectedAddress` should be an hexadecimal string, with the leading 0x.
     */
    function verifyReadMemoryResponse(resp, expectedBytes, expectedAddress) {
        var _a, _b, _c, _d;
        (0, chai_1.expect)((_a = resp.body) === null || _a === void 0 ? void 0 : _a.data).eq((0, common_1.hexToBase64)(expectedBytes));
        (0, chai_1.expect)((_b = resp.body) === null || _b === void 0 ? void 0 : _b.address).match(/^0x[0-9a-fA-F]+$/);
        if ((_c = resp.body) === null || _c === void 0 ? void 0 : _c.address) {
            const actualAddress = parseInt((_d = resp.body) === null || _d === void 0 ? void 0 : _d.address);
            (0, chai_1.expect)(actualAddress).eq(expectedAddress);
        }
    }
    // Test reading memory
    it('can read memory', function () {
        return __awaiter(this, void 0, void 0, function* () {
            // Get the address of the array.
            const addrOfArrayResp = yield dc.evaluateRequest({
                expression: '&array',
                frameId: frame.id,
            });
            const addrOfArray = parseInt(addrOfArrayResp.body.result, 16);
            let mem = yield dc.readMemoryRequest({
                memoryReference: '0x' + addrOfArray.toString(16),
                count: 10,
            });
            verifyReadMemoryResponse(mem, 'f1efd4fd7248450c2d13', addrOfArray);
            mem = yield dc.readMemoryRequest({
                memoryReference: '&array[3 + 2]',
                count: 10,
            });
            verifyReadMemoryResponse(mem, '48450c2d1374d6f612dc', addrOfArray + 5);
            mem = yield dc.readMemoryRequest({
                memoryReference: 'parray',
                count: 10,
            });
            verifyReadMemoryResponse(mem, 'f1efd4fd7248450c2d13', addrOfArray);
            mem = yield dc.readMemoryRequest({
                memoryReference: 'parray',
                count: 10,
                offset: 5,
            });
            verifyReadMemoryResponse(mem, '48450c2d1374d6f612dc', addrOfArray + 5);
            mem = yield dc.readMemoryRequest({
                memoryReference: 'parray',
                count: 0,
            });
            // the spec isn't clear on what exactly can be retruned if count == 0
            // so the following works with VSCode - simply having no body
            (0, chai_1.expect)(mem.body).is.undefined;
            mem = yield dc.readMemoryRequest({
                memoryReference: 'parray',
                count: 0,
                offset: 5,
            });
            (0, chai_1.expect)(mem.body).is.undefined;
        });
    });
    it('handles unable to read memory', function () {
        return __awaiter(this, void 0, void 0, function* () {
            // This test will only work for targets for which address 0 is not readable, which is good enough for now.
            const err = yield (0, utils_1.expectRejection)(dc.readMemoryRequest({
                memoryReference: '0',
                count: 10,
            }));
            (0, chai_1.expect)(err.message).contains('Unable to read memory');
        });
    });
    it('can read memory with offset', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const addrOfArrayResp = yield dc.evaluateRequest({
                expression: '&array',
                frameId: frame.id,
            });
            const addrOfArray = parseInt(addrOfArrayResp.body.result, 16);
            // Test positive offset
            let offset = 5;
            let mem = yield dc.readMemoryRequest({
                memoryReference: '&array',
                count: 5,
                offset,
            });
            verifyReadMemoryResponse(mem, '48450c2d13', addrOfArray + offset);
            // Test negative offset
            offset = -5;
            mem = yield dc.readMemoryRequest({
                memoryReference: `array + ${-offset}`,
                count: 10,
                offset,
            });
            verifyReadMemoryResponse(mem, 'f1efd4fd7248450c2d13', addrOfArray);
        });
    });
    const newValue = '123456789abcdef01234';
    const writeArguments = {
        data: (0, common_1.hexToBase64)(newValue),
        memoryReference: '&array',
    };
    it('can write memory', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const addrOfArray = parseInt((yield dc.evaluateRequest({
                expression: '&array',
                frameId: frame.id,
            })).body.result);
            yield dc.writeMemoryRequest(writeArguments);
            const memory = yield dc.readMemoryRequest({
                memoryReference: '&array',
                count: 10,
                offset: 0,
            });
            verifyReadMemoryResponse(memory, newValue, addrOfArray);
        });
    });
    it('fails when trying to write to read-only memory', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const addrOfArray = parseInt((yield dc.evaluateRequest({
                expression: '&array',
                frameId: frame.id,
            })).body.result);
            yield dc.send('cdt-gdb-tests/executeCommand', {
                command: `-interpreter-exec console "mem ${addrOfArray} ${addrOfArray + 10} ro"`,
            });
            const error = yield (0, utils_1.expectRejection)(dc.writeMemoryRequest(writeArguments));
            (0, chai_1.expect)(error.message).contains('Cannot access memory');
        });
    });
    it('Converts between hex and base64 without loss', () => {
        const normalize = (original) => original.toLowerCase();
        const hexToBase64TestCases = [
            'fe',
            '00',
            'fedc',
            '00FE',
            '29348798237abfeCCD',
        ];
        const base64ToHexTestCases = [
            'bGlnaHQgd29yay4=',
            'bGlnaHQgd29yaw==',
            'abc',
            'abcd',
        ];
        for (const test of hexToBase64TestCases) {
            (0, chai_1.expect)(normalize((0, common_1.base64ToHex)((0, common_1.hexToBase64)(test)))).equal(normalize(test));
        }
        for (const test of base64ToHexTestCases) {
            (0, chai_1.expect)((0, common_1.hexToBase64)((0, common_1.base64ToHex)(test))).equal(test + '='.repeat((4 - (test.length % 4)) % 4));
        }
    });
    it('Throws an error if it detects ill-formed input', () => {
        const hexToBase64TextCases = ['f', 'fED', '0fedc', 'zyxd'];
        const base64ToHexTestCases = [
            'ab',
            'a',
            'a=',
            'abcde',
            '!A==',
            '#$*@^',
            '234bGeuTHEUDReuhr',
        ];
        for (const test of hexToBase64TextCases) {
            (0, chai_1.expect)(() => (0, common_1.hexToBase64)(test)).throws();
        }
        for (const test of base64ToHexTestCases) {
            (0, chai_1.expect)(() => (0, common_1.base64ToHex)(test)).throws();
        }
    });
});
//# sourceMappingURL=mem.spec.js.map