/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.fineract.accounting.glaccount.domain;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.List;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;

public interface TrialBalanceRepository extends JpaRepository<TrialBalance, Long>, JpaSpecificationExecutor<TrialBalance> {

    @Query(value = "select * from m_trial_balance where office_id=:officeId and account_id=:accountId and closing_balance is null order by created_date, entry_date", nativeQuery = true)
    List<TrialBalance> findNewByOfficeAndAccount(@Param("officeId") Long officeId, @Param("accountId") Long accountId);

    @Query("SELECT MAX(tb.transactionDate) FROM TrialBalance tb")
    LocalDate findMaxCreatedDate();

    @Query("""
                    SELECT tb.closingBalance
                    FROM TrialBalance tb
                    WHERE tb.officeId = :officeId
                      AND tb.glAccountId = :accountId
                      AND tb.closingBalance IS NOT NULL
                    ORDER BY tb.transactionDate DESC, tb.entryDate DESC
            """)
    List<BigDecimal> findLastClosingBalance(Long officeId, Long accountId);

    @Query("""
                SELECT DISTINCT tb.officeId
                FROM TrialBalance tb
                WHERE tb.closingBalance IS NULL
            """)
    List<Long> findDistinctOfficeIdsWithNullClosingBalance();

    @Query("""
                SELECT DISTINCT tb.glAccountId
                FROM TrialBalance tb
                WHERE tb.officeId = :officeId
                  AND tb.closingBalance IS NULL
            """)
    List<Long> findDistinctAccountIdsWithNullClosingBalanceByOfficeId(Long officeId);

}
