/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.types;

import org.apache.flink.annotation.Public;
import org.apache.flink.core.memory.DataInputView;
import org.apache.flink.core.memory.DataOutputView;
import org.apache.flink.core.memory.MemorySegment;

import java.io.IOException;

/**
 * Boxed serializable and comparable short integer type, representing the primitive type {@code
 * short}.
 */
@Public
public class ShortValue
        implements NormalizableKey<ShortValue>,
                ResettableValue<ShortValue>,
                CopyableValue<ShortValue> {
    private static final long serialVersionUID = 1L;

    private short value;

    /** Initializes the encapsulated short with 0. */
    public ShortValue() {
        this.value = 0;
    }

    /**
     * Initializes the encapsulated short with the provided value.
     *
     * @param value Initial value of the encapsulated short.
     */
    public ShortValue(short value) {
        this.value = value;
    }

    /**
     * Returns the value of the encapsulated short.
     *
     * @return the value of the encapsulated short.
     */
    public short getValue() {
        return this.value;
    }

    /**
     * Sets the encapsulated short to the specified value.
     *
     * @param value the new value of the encapsulated short.
     */
    public void setValue(short value) {
        this.value = value;
    }

    @Override
    public void setValue(ShortValue value) {
        this.value = value.value;
    }

    @Override
    public String toString() {
        return String.valueOf(this.value);
    }

    // --------------------------------------------------------------------------------------------

    @Override
    public void read(DataInputView in) throws IOException {
        this.value = in.readShort();
    }

    @Override
    public void write(DataOutputView out) throws IOException {
        out.writeShort(this.value);
    }

    // --------------------------------------------------------------------------------------------

    @Override
    public int compareTo(ShortValue o) {
        final int other = o.value;
        return this.value < other ? -1 : this.value > other ? 1 : 0;
    }

    @Override
    public int hashCode() {
        return this.value;
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof ShortValue) {
            return ((ShortValue) obj).value == this.value;
        }
        return false;
    }

    // --------------------------------------------------------------------------------------------

    @Override
    public int getMaxNormalizedKeyLen() {
        return 2;
    }

    @Override
    public void copyNormalizedKey(MemorySegment target, int offset, int len) {
        if (len == 2) {
            // default case, full normalized key
            int highByte = ((value >>> 8) & 0xff);
            highByte -= Byte.MIN_VALUE;
            target.put(offset, (byte) highByte);
            target.put(offset + 1, (byte) ((value) & 0xff));
        } else if (len <= 0) {
        } else if (len == 1) {
            int highByte = ((value >>> 8) & 0xff);
            highByte -= Byte.MIN_VALUE;
            target.put(offset, (byte) highByte);
        } else {
            int highByte = ((value >>> 8) & 0xff);
            highByte -= Byte.MIN_VALUE;
            target.put(offset, (byte) highByte);
            target.put(offset + 1, (byte) ((value) & 0xff));
            for (int i = 2; i < len; i++) {
                target.put(offset + i, (byte) 0);
            }
        }
    }

    // --------------------------------------------------------------------------------------------

    @Override
    public int getBinaryLength() {
        return 2;
    }

    @Override
    public void copyTo(ShortValue target) {
        target.value = this.value;
    }

    @Override
    public ShortValue copy() {
        return new ShortValue(this.value);
    }

    @Override
    public void copy(DataInputView source, DataOutputView target) throws IOException {
        target.write(source, 2);
    }
}
