// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

use message_bus::MessageBus;

pub trait Project<T, C: Consensus> {
    fn project(self, consensus: &C) -> T;
}

pub trait Pipeline {
    type Message;
    type Entry;

    fn push_message(&mut self, message: Self::Message);

    fn pop_message(&mut self) -> Option<Self::Entry>;

    /// Extract and remove a message by op number.
    fn extract_by_op(&mut self, op: u64) -> Option<Self::Entry>;

    fn clear(&mut self);

    fn message_by_op(&self, op: u64) -> Option<&Self::Entry>;

    fn message_by_op_mut(&mut self, op: u64) -> Option<&mut Self::Entry>;

    fn message_by_op_and_checksum(&self, op: u64, checksum: u128) -> Option<&Self::Entry>;

    fn is_full(&self) -> bool;

    fn is_empty(&self) -> bool;

    fn verify(&self);
}

pub trait Consensus: Sized {
    type MessageBus: MessageBus;
    // I am wondering, whether we should create a dedicated trait for cloning, so it's explicit that we do ref counting.
    type RequestMessage: Project<Self::ReplicateMessage, Self> + Clone;
    type ReplicateMessage: Project<Self::AckMessage, Self> + Clone;
    type AckMessage;
    type Sequencer: Sequencer;
    type Pipeline: Pipeline<Message = Self::ReplicateMessage>;

    fn pipeline_message(&self, message: Self::ReplicateMessage);
    fn verify_pipeline(&self);

    // TODO: Figure out how we can achieve that without exposing such methods in the Consensus trait.
    fn post_replicate_verify(&self, message: &Self::ReplicateMessage);

    fn is_follower(&self) -> bool;
    fn is_normal(&self) -> bool;
    fn is_syncing(&self) -> bool;
}

/// Shared consensus lifecycle interface for control/data planes.
///
/// This abstracts the VSR message flow:
/// - request -> prepare
/// - replicate (prepare)
/// - ack (prepare_ok)
pub trait Plane<C>
where
    C: Consensus,
{
    fn on_request(&self, message: C::RequestMessage) -> impl Future<Output = ()>;
    fn on_replicate(&self, message: C::ReplicateMessage) -> impl Future<Output = ()>;
    fn on_ack(&self, message: C::AckMessage) -> impl Future<Output = ()>;
}

mod impls;
pub use impls::*;
mod plane_helpers;
pub use plane_helpers::*;

mod view_change_quorum;
pub use view_change_quorum::*;
mod vsr_timeout;
