/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.internals.KafkaFutureImpl;
import org.apache.kafka.common.metrics.MeasurableStat;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.metrics.stats.Avg;
import org.apache.kafka.common.metrics.stats.Rate;
import org.apache.kafka.common.metrics.stats.SampledStat;
import org.apache.kafka.common.metrics.stats.WindowedCount;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.streams.StreamsConfig;
import org.apache.kafka.streams.errors.StreamsException;
import org.apache.kafka.streams.errors.TaskCorruptedException;
import org.apache.kafka.streams.processor.TaskId;
import org.apache.kafka.streams.processor.internals.ChangelogReader;
import org.apache.kafka.streams.processor.internals.ReadOnlyTask;
import org.apache.kafka.streams.processor.internals.StandbyTask;
import org.apache.kafka.streams.processor.internals.StateUpdater;
import org.apache.kafka.streams.processor.internals.StreamTask;
import org.apache.kafka.streams.processor.internals.Task;
import org.apache.kafka.streams.processor.internals.TaskAndAction;
import org.apache.kafka.streams.processor.internals.TopologyMetadata;
import org.apache.kafka.streams.processor.internals.metrics.StreamsMetricsImpl;
import org.slf4j.Logger;

public class DefaultStateUpdater
implements StateUpdater {
    private static final String BUG_ERROR_MESSAGE = "This indicates a bug. Please report at https://issues.apache.org/jira/projects/KAFKA/issues or to the dev-mailing list (https://kafka.apache.org/contact).";
    private final Time time;
    private final Logger log;
    private final String name;
    private final StreamsMetricsImpl metrics;
    private final Consumer<byte[], byte[]> restoreConsumer;
    private final ChangelogReader changelogReader;
    private final TopologyMetadata topologyMetadata;
    private final Queue<TaskAndAction> tasksAndActions = new LinkedList<TaskAndAction>();
    private final Lock tasksAndActionsLock = new ReentrantLock();
    private final Condition tasksAndActionsCondition = this.tasksAndActionsLock.newCondition();
    private final Queue<StreamTask> restoredActiveTasks = new LinkedList<StreamTask>();
    private final Lock restoredActiveTasksLock = new ReentrantLock();
    private final Condition restoredActiveTasksCondition = this.restoredActiveTasksLock.newCondition();
    private final Lock exceptionsAndFailedTasksLock = new ReentrantLock();
    private final Queue<StateUpdater.ExceptionAndTask> exceptionsAndFailedTasks = new LinkedList<StateUpdater.ExceptionAndTask>();
    private final BlockingQueue<Task> removedTasks = new LinkedBlockingQueue<Task>();
    private final AtomicBoolean isTopologyResumed = new AtomicBoolean(false);
    private final long commitIntervalMs;
    private long lastCommitMs;
    private StateUpdaterThread stateUpdaterThread = null;

    public DefaultStateUpdater(String name, StreamsMetricsImpl metrics, StreamsConfig config, Consumer<byte[], byte[]> restoreConsumer, ChangelogReader changelogReader, TopologyMetadata topologyMetadata, Time time) {
        this.time = time;
        this.name = name;
        this.metrics = metrics;
        this.restoreConsumer = restoreConsumer;
        this.changelogReader = changelogReader;
        this.topologyMetadata = topologyMetadata;
        this.commitIntervalMs = config.getLong("commit.interval.ms");
        String logPrefix = String.format("state-updater [%s] ", name);
        LogContext logContext = new LogContext(logPrefix);
        this.log = logContext.logger(DefaultStateUpdater.class);
    }

    @Override
    public void start() {
        if (this.stateUpdaterThread == null) {
            if (!this.restoredActiveTasks.isEmpty() || !this.exceptionsAndFailedTasks.isEmpty()) {
                throw new IllegalStateException("State updater started with non-empty output queues. This indicates a bug. Please report at https://issues.apache.org/jira/projects/KAFKA/issues or to the dev-mailing list (https://kafka.apache.org/contact).");
            }
            this.stateUpdaterThread = new StateUpdaterThread(this.name, this.metrics, this.changelogReader);
            this.stateUpdaterThread.start();
            this.lastCommitMs = this.time.milliseconds();
        }
    }

    @Override
    public void shutdown(Duration timeout) {
        if (this.stateUpdaterThread != null) {
            this.log.info("Shutting down state updater thread");
            this.stateUpdaterThread.isRunning.set(false);
            this.tasksAndActionsLock.lock();
            try {
                this.tasksAndActionsCondition.signalAll();
            }
            finally {
                this.tasksAndActionsLock.unlock();
            }
            try {
                this.stateUpdaterThread.join(timeout.toMillis());
                if (this.stateUpdaterThread.isAlive()) {
                    throw new StreamsException("State updater thread did not shutdown within the timeout");
                }
                this.stateUpdaterThread = null;
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
    }

    @Override
    public void add(Task task) {
        this.verifyStateFor(task);
        this.tasksAndActionsLock.lock();
        try {
            this.tasksAndActions.add(TaskAndAction.createAddTask(task));
            this.tasksAndActionsCondition.signalAll();
        }
        finally {
            this.tasksAndActionsLock.unlock();
        }
    }

    private void verifyStateFor(Task task) {
        if (task.isActive() && task.state() != Task.State.RESTORING) {
            throw new IllegalStateException("Active task " + String.valueOf(task.id()) + " is not in state RESTORING. This indicates a bug. Please report at https://issues.apache.org/jira/projects/KAFKA/issues or to the dev-mailing list (https://kafka.apache.org/contact).");
        }
        if (!task.isActive() && task.state() != Task.State.RUNNING) {
            throw new IllegalStateException("Standby task " + String.valueOf(task.id()) + " is not in state RUNNING. This indicates a bug. Please report at https://issues.apache.org/jira/projects/KAFKA/issues or to the dev-mailing list (https://kafka.apache.org/contact).");
        }
    }

    @Override
    public CompletableFuture<StateUpdater.RemovedTaskResult> remove(TaskId taskId) {
        CompletableFuture<StateUpdater.RemovedTaskResult> future = new CompletableFuture<StateUpdater.RemovedTaskResult>();
        this.tasksAndActionsLock.lock();
        try {
            this.tasksAndActions.add(TaskAndAction.createRemoveTask(taskId, future));
            this.tasksAndActionsCondition.signalAll();
        }
        finally {
            this.tasksAndActionsLock.unlock();
        }
        return future;
    }

    @Override
    public void signalResume() {
        this.tasksAndActionsLock.lock();
        try {
            this.isTopologyResumed.set(true);
            this.tasksAndActionsCondition.signalAll();
        }
        finally {
            this.tasksAndActionsLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Set<StreamTask> drainRestoredActiveTasks(Duration timeout) {
        long timeoutMs = timeout.toMillis();
        long startTime = this.time.milliseconds();
        long deadline = startTime + timeoutMs;
        long now = startTime;
        HashSet<StreamTask> result = new HashSet<StreamTask>();
        try {
            while (now <= deadline && result.isEmpty()) {
                this.restoredActiveTasksLock.lock();
                try {
                    while (this.restoredActiveTasks.isEmpty() && now <= deadline) {
                        boolean ignored = this.restoredActiveTasksCondition.await(deadline - now, TimeUnit.MILLISECONDS);
                        now = this.time.milliseconds();
                    }
                    result.addAll(this.restoredActiveTasks);
                    this.restoredActiveTasks.clear();
                }
                finally {
                    this.restoredActiveTasksLock.unlock();
                }
                now = this.time.milliseconds();
            }
            return result;
        }
        catch (InterruptedException interruptedException) {
            return result;
        }
    }

    @Override
    public List<StateUpdater.ExceptionAndTask> drainExceptionsAndFailedTasks() {
        ArrayList<StateUpdater.ExceptionAndTask> result = new ArrayList<StateUpdater.ExceptionAndTask>();
        this.exceptionsAndFailedTasksLock.lock();
        try {
            result.addAll(this.exceptionsAndFailedTasks);
            this.exceptionsAndFailedTasks.clear();
        }
        finally {
            this.exceptionsAndFailedTasksLock.unlock();
        }
        return result;
    }

    @Override
    public boolean hasExceptionsAndFailedTasks() {
        this.exceptionsAndFailedTasksLock.lock();
        try {
            boolean bl = !this.exceptionsAndFailedTasks.isEmpty();
            return bl;
        }
        finally {
            this.exceptionsAndFailedTasksLock.unlock();
        }
    }

    public Set<StandbyTask> updatingStandbyTasks() {
        return this.stateUpdaterThread != null ? Set.copyOf(this.stateUpdaterThread.updatingStandbyTasks()) : Collections.emptySet();
    }

    @Override
    public Set<Task> updatingTasks() {
        return this.stateUpdaterThread != null ? Set.copyOf(this.stateUpdaterThread.updatingTasks()) : Collections.emptySet();
    }

    public Set<StreamTask> restoredActiveTasks() {
        this.restoredActiveTasksLock.lock();
        try {
            Set<StreamTask> set = Set.copyOf(this.restoredActiveTasks);
            return set;
        }
        finally {
            this.restoredActiveTasksLock.unlock();
        }
    }

    public List<StateUpdater.ExceptionAndTask> exceptionsAndFailedTasks() {
        this.exceptionsAndFailedTasksLock.lock();
        try {
            List<StateUpdater.ExceptionAndTask> list = List.copyOf(this.exceptionsAndFailedTasks);
            return list;
        }
        finally {
            this.exceptionsAndFailedTasksLock.unlock();
        }
    }

    public Set<Task> removedTasks() {
        return Set.copyOf(this.removedTasks);
    }

    public Set<Task> pausedTasks() {
        return this.stateUpdaterThread != null ? Set.copyOf(this.stateUpdaterThread.pausedTasks()) : Collections.emptySet();
    }

    @Override
    public Set<Task> tasks() {
        return this.executeWithQueuesLocked(() -> this.streamOfTasks().map(ReadOnlyTask::new).collect(Collectors.toSet()));
    }

    @Override
    public boolean restoresActiveTasks() {
        return !this.executeWithQueuesLocked(() -> this.streamOfTasks().filter(Task::isActive).collect(Collectors.toSet())).isEmpty();
    }

    public Set<StreamTask> activeTasks() {
        return this.executeWithQueuesLocked(() -> this.streamOfTasks().filter(Task::isActive).map(t -> (StreamTask)t).collect(Collectors.toSet()));
    }

    @Override
    public Set<StandbyTask> standbyTasks() {
        return this.executeWithQueuesLocked(() -> this.streamOfTasks().filter(t -> !t.isActive()).map(t -> (StandbyTask)t).collect(Collectors.toSet()));
    }

    @Override
    public KafkaFutureImpl<Uuid> restoreConsumerInstanceId(Duration timeout) {
        return this.stateUpdaterThread.restoreConsumerInstanceId(timeout);
    }

    public boolean isRunning() {
        return this.stateUpdaterThread != null && this.stateUpdaterThread.isRunning.get();
    }

    boolean isIdle() {
        if (this.stateUpdaterThread != null) {
            return this.stateUpdaterThread.isIdle.get();
        }
        return false;
    }

    private <T> Set<T> executeWithQueuesLocked(Supplier<Set<T>> action) {
        this.tasksAndActionsLock.lock();
        this.restoredActiveTasksLock.lock();
        this.exceptionsAndFailedTasksLock.lock();
        try {
            Set<T> set = action.get();
            return set;
        }
        finally {
            this.exceptionsAndFailedTasksLock.unlock();
            this.restoredActiveTasksLock.unlock();
            this.tasksAndActionsLock.unlock();
        }
    }

    private Stream<Task> streamOfTasks() {
        return Stream.concat(this.streamOfNonPausedTasks(), this.pausedTasks().stream());
    }

    private Stream<Task> streamOfNonPausedTasks() {
        return Stream.concat(this.tasksAndActions.stream().filter(taskAndAction -> taskAndAction.action() == TaskAndAction.Action.ADD).map(TaskAndAction::task), Stream.concat(this.updatingTasks().stream(), Stream.concat(this.restoredActiveTasks.stream(), Stream.concat(this.exceptionsAndFailedTasks.stream().map(StateUpdater.ExceptionAndTask::task), this.removedTasks.stream()))));
    }

    private class StateUpdaterThread
    extends Thread {
        private final ChangelogReader changelogReader;
        private final StateUpdaterMetrics updaterMetrics;
        private final AtomicBoolean isRunning;
        private final AtomicBoolean isIdle;
        private final Map<TaskId, Task> updatingTasks;
        private final Map<TaskId, Task> pausedTasks;
        private long totalCheckpointLatency;
        private volatile long fetchDeadlineClientInstanceId;
        private volatile KafkaFutureImpl<Uuid> clientInstanceIdFuture;

        public StateUpdaterThread(String name, StreamsMetricsImpl metrics, ChangelogReader changelogReader) {
            super(name);
            this.isRunning = new AtomicBoolean(true);
            this.isIdle = new AtomicBoolean(false);
            this.updatingTasks = new ConcurrentHashMap<TaskId, Task>();
            this.pausedTasks = new ConcurrentHashMap<TaskId, Task>();
            this.totalCheckpointLatency = 0L;
            this.fetchDeadlineClientInstanceId = -1L;
            this.clientInstanceIdFuture = new KafkaFutureImpl();
            this.changelogReader = changelogReader;
            this.updaterMetrics = new StateUpdaterMetrics(metrics, name);
        }

        public Collection<Task> updatingTasks() {
            return this.updatingTasks.values();
        }

        public Collection<StandbyTask> updatingStandbyTasks() {
            return this.updatingTasks.values().stream().filter(t -> !t.isActive()).map(t -> (StandbyTask)t).collect(Collectors.toList());
        }

        private boolean onlyStandbyTasksUpdating() {
            return !this.updatingTasks.isEmpty() && this.updatingTasks.values().stream().noneMatch(Task::isActive);
        }

        public Collection<Task> pausedTasks() {
            return this.pausedTasks.values();
        }

        public long numUpdatingStandbyTasks() {
            return this.updatingTasks.values().stream().filter(t -> !t.isActive()).count();
        }

        public long numRestoringActiveTasks() {
            return this.updatingTasks.values().stream().filter(Task::isActive).count();
        }

        public long numPausedStandbyTasks() {
            return this.pausedTasks.values().stream().filter(t -> !t.isActive()).count();
        }

        public long numPausedActiveTasks() {
            return this.pausedTasks.values().stream().filter(Task::isActive).count();
        }

        @Override
        public void run() {
            DefaultStateUpdater.this.log.info("State updater thread started");
            try {
                while (this.isRunning.get()) {
                    this.runOnce();
                }
            }
            catch (RuntimeException anyOtherException) {
                this.handleRuntimeException(anyOtherException);
            }
            finally {
                this.clearInputQueue();
                this.clearUpdatingAndPausedTasks();
                this.updaterMetrics.clear();
                DefaultStateUpdater.this.log.info("State updater thread stopped");
            }
        }

        private void clearInputQueue() {
            DefaultStateUpdater.this.tasksAndActionsLock.lock();
            try {
                DefaultStateUpdater.this.tasksAndActions.clear();
            }
            finally {
                DefaultStateUpdater.this.tasksAndActionsLock.unlock();
            }
        }

        private void runOnce() {
            long totalStartTimeMs = DefaultStateUpdater.this.time.milliseconds();
            this.performActionsOnTasks();
            this.resumeTasks();
            this.pauseTasks();
            this.restoreTasks(totalStartTimeMs);
            this.maybeGetClientInstanceIds();
            long checkpointStartTimeMs = DefaultStateUpdater.this.time.milliseconds();
            this.maybeCheckpointTasks(checkpointStartTimeMs);
            long waitStartTimeMs = DefaultStateUpdater.this.time.milliseconds();
            this.waitIfAllChangelogsCompletelyRead();
            long endTimeMs = DefaultStateUpdater.this.time.milliseconds();
            long totalWaitTime = Math.max(0L, endTimeMs - waitStartTimeMs);
            long totalTime = Math.max(0L, endTimeMs - totalStartTimeMs);
            this.recordMetrics(endTimeMs, totalTime, totalWaitTime);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void performActionsOnTasks() {
            DefaultStateUpdater.this.tasksAndActionsLock.lock();
            try {
                block7: for (TaskAndAction taskAndAction : this.tasksAndActions()) {
                    TaskAndAction.Action action = taskAndAction.action();
                    switch (action) {
                        case ADD: {
                            this.addTask(taskAndAction.task());
                            continue block7;
                        }
                        case REMOVE: {
                            this.removeTask(taskAndAction.taskId(), taskAndAction.futureForRemove());
                            continue block7;
                        }
                    }
                    throw new IllegalStateException("Unknown action type " + String.valueOf((Object)action));
                }
            }
            finally {
                DefaultStateUpdater.this.tasksAndActionsLock.unlock();
            }
        }

        private void resumeTasks() {
            if (DefaultStateUpdater.this.isTopologyResumed.compareAndSet(true, false)) {
                for (Task task : this.pausedTasks.values()) {
                    if (DefaultStateUpdater.this.topologyMetadata.isPaused(task.id().topologyName())) continue;
                    this.resumeTask(task);
                }
            }
        }

        private void pauseTasks() {
            for (Task task : this.updatingTasks.values()) {
                if (!DefaultStateUpdater.this.topologyMetadata.isPaused(task.id().topologyName())) continue;
                this.pauseTask(task);
            }
        }

        private void restoreTasks(long now) {
            try {
                long restored = this.changelogReader.restore(this.updatingTasks);
                this.updaterMetrics.restoreSensor.record((double)restored, now);
            }
            catch (TaskCorruptedException taskCorruptedException) {
                this.handleTaskCorruptedException(taskCorruptedException);
            }
            catch (StreamsException streamsException) {
                this.handleStreamsException(streamsException);
            }
            Set<TopicPartition> completedChangelogs = this.changelogReader.completedChangelogs();
            List activeTasks = this.updatingTasks.values().stream().filter(Task::isActive).collect(Collectors.toList());
            for (Task task : activeTasks) {
                this.maybeCompleteRestoration((StreamTask)task, completedChangelogs);
            }
        }

        private void maybeGetClientInstanceIds() {
            if (this.fetchDeadlineClientInstanceId != -1L && !this.clientInstanceIdFuture.isDone()) {
                if (this.fetchDeadlineClientInstanceId >= DefaultStateUpdater.this.time.milliseconds()) {
                    try {
                        this.clientInstanceIdFuture.complete((Object)DefaultStateUpdater.this.restoreConsumer.clientInstanceId(this.allWorkDone() ? Duration.ofMillis(100L) : Duration.ZERO));
                        this.fetchDeadlineClientInstanceId = -1L;
                    }
                    catch (IllegalStateException disabledError) {
                        this.clientInstanceIdFuture.complete(null);
                        this.fetchDeadlineClientInstanceId = -1L;
                    }
                    catch (TimeoutException disabledError) {
                    }
                    catch (Exception error) {
                        this.clientInstanceIdFuture.completeExceptionally((Throwable)error);
                        this.fetchDeadlineClientInstanceId = -1L;
                    }
                } else {
                    this.clientInstanceIdFuture.completeExceptionally((Throwable)new TimeoutException("Could not retrieve restore consumer client instance id."));
                    this.fetchDeadlineClientInstanceId = -1L;
                }
            }
        }

        private KafkaFutureImpl<Uuid> restoreConsumerInstanceId(Duration timeout) {
            boolean setDeadline = false;
            if (this.clientInstanceIdFuture.isDone()) {
                if (this.clientInstanceIdFuture.isCompletedExceptionally()) {
                    this.clientInstanceIdFuture = new KafkaFutureImpl();
                    setDeadline = true;
                }
            } else {
                setDeadline = true;
            }
            if (setDeadline) {
                this.fetchDeadlineClientInstanceId = DefaultStateUpdater.this.time.milliseconds() + timeout.toMillis();
                DefaultStateUpdater.this.tasksAndActionsLock.lock();
                try {
                    DefaultStateUpdater.this.tasksAndActionsCondition.signalAll();
                }
                finally {
                    DefaultStateUpdater.this.tasksAndActionsLock.unlock();
                }
            }
            return this.clientInstanceIdFuture;
        }

        private void handleRuntimeException(RuntimeException runtimeException) {
            DefaultStateUpdater.this.log.error("An unexpected error occurred within the state updater thread: {}", (Object)String.valueOf(runtimeException));
            this.addToExceptionsAndFailedTasksThenClearUpdatingAndPausedTasks(runtimeException);
            this.isRunning.set(false);
        }

        private void handleTaskCorruptedException(TaskCorruptedException taskCorruptedException) {
            DefaultStateUpdater.this.log.info("Encountered task corrupted exception: ", (Throwable)((Object)taskCorruptedException));
            Set<TaskId> corruptedTaskIds = taskCorruptedException.corruptedTasks();
            HashSet<Task> corruptedTasks = new HashSet<Task>();
            HashSet<TopicPartition> changelogsOfCorruptedTasks = new HashSet<TopicPartition>();
            for (TaskId taskId : corruptedTaskIds) {
                Task corruptedTask = this.updatingTasks.get(taskId);
                if (corruptedTask == null) {
                    throw new IllegalStateException("Task " + String.valueOf(taskId) + " is corrupted but is not updating. This indicates a bug. Please report at https://issues.apache.org/jira/projects/KAFKA/issues or to the dev-mailing list (https://kafka.apache.org/contact).");
                }
                corruptedTasks.add(corruptedTask);
                this.removeCheckpointForCorruptedTask(corruptedTask);
                changelogsOfCorruptedTasks.addAll(corruptedTask.changelogPartitions());
            }
            this.changelogReader.unregister(changelogsOfCorruptedTasks);
            corruptedTasks.forEach(task -> this.addToExceptionsAndFailedTasksThenRemoveFromUpdatingTasks(new StateUpdater.ExceptionAndTask((RuntimeException)((Object)taskCorruptedException), (Task)task)));
        }

        private void removeCheckpointForCorruptedTask(Task task) {
            try {
                task.markChangelogAsCorrupted(task.changelogPartitions());
                this.measureCheckpointLatency(() -> task.maybeCheckpoint(true));
            }
            catch (StreamsException swallow) {
                DefaultStateUpdater.this.log.warn("Checkpoint failed for corrupted task {}", (Object)task.id(), (Object)swallow);
            }
        }

        private void handleStreamsException(StreamsException streamsException) {
            DefaultStateUpdater.this.log.info("Encountered streams exception: ", (Throwable)((Object)streamsException));
            if (streamsException.taskId().isPresent()) {
                this.handleStreamsExceptionWithTask(streamsException, streamsException.taskId().get());
            } else {
                this.handleStreamsExceptionWithoutTask(streamsException);
            }
        }

        private void handleStreamsExceptionWithTask(StreamsException streamsException, TaskId failedTaskId) {
            if (this.updatingTasks.containsKey(failedTaskId)) {
                this.addToExceptionsAndFailedTasksThenRemoveFromUpdatingTasks(new StateUpdater.ExceptionAndTask((RuntimeException)((Object)streamsException), this.updatingTasks.get(failedTaskId)));
            } else if (this.pausedTasks.containsKey(failedTaskId)) {
                this.addToExceptionsAndFailedTasksThenRemoveFromPausedTasks(new StateUpdater.ExceptionAndTask((RuntimeException)((Object)streamsException), this.pausedTasks.get(failedTaskId)));
            } else {
                throw new IllegalStateException("Task " + String.valueOf(failedTaskId) + " failed but is not updating or paused. This indicates a bug. Please report at https://issues.apache.org/jira/projects/KAFKA/issues or to the dev-mailing list (https://kafka.apache.org/contact).");
            }
        }

        private void handleStreamsExceptionWithoutTask(StreamsException streamsException) {
            this.addToExceptionsAndFailedTasksThenClearUpdatingAndPausedTasks((RuntimeException)((Object)streamsException));
        }

        private void addToExceptionsAndFailedTasksThenRemoveFromUpdatingTasks(StateUpdater.ExceptionAndTask exceptionAndTask) {
            DefaultStateUpdater.this.exceptionsAndFailedTasksLock.lock();
            try {
                DefaultStateUpdater.this.exceptionsAndFailedTasks.add(exceptionAndTask);
                this.updatingTasks.remove(exceptionAndTask.task().id());
                if (exceptionAndTask.task().isActive()) {
                    this.transitToUpdateStandbysIfOnlyStandbysLeft();
                }
            }
            finally {
                DefaultStateUpdater.this.exceptionsAndFailedTasksLock.unlock();
            }
        }

        private void addToExceptionsAndFailedTasksThenRemoveFromPausedTasks(StateUpdater.ExceptionAndTask exceptionAndTask) {
            DefaultStateUpdater.this.exceptionsAndFailedTasksLock.lock();
            try {
                DefaultStateUpdater.this.exceptionsAndFailedTasks.add(exceptionAndTask);
                this.pausedTasks.remove(exceptionAndTask.task().id());
                if (exceptionAndTask.task().isActive()) {
                    this.transitToUpdateStandbysIfOnlyStandbysLeft();
                }
            }
            finally {
                DefaultStateUpdater.this.exceptionsAndFailedTasksLock.unlock();
            }
        }

        private void addToExceptionsAndFailedTasksThenClearUpdatingAndPausedTasks(RuntimeException runtimeException) {
            DefaultStateUpdater.this.exceptionsAndFailedTasksLock.lock();
            try {
                this.updatingTasks.values().forEach(task -> DefaultStateUpdater.this.exceptionsAndFailedTasks.add(new StateUpdater.ExceptionAndTask(runtimeException, (Task)task)));
                this.updatingTasks.clear();
                this.pausedTasks.values().forEach(task -> DefaultStateUpdater.this.exceptionsAndFailedTasks.add(new StateUpdater.ExceptionAndTask(runtimeException, (Task)task)));
                this.pausedTasks.clear();
            }
            finally {
                DefaultStateUpdater.this.exceptionsAndFailedTasksLock.unlock();
            }
        }

        private void waitIfAllChangelogsCompletelyRead() {
            DefaultStateUpdater.this.tasksAndActionsLock.lock();
            try {
                while (this.allWorkDone() && this.fetchDeadlineClientInstanceId == -1L) {
                    this.isIdle.set(true);
                    DefaultStateUpdater.this.tasksAndActionsCondition.await();
                }
            }
            catch (InterruptedException interruptedException) {
            }
            finally {
                DefaultStateUpdater.this.tasksAndActionsLock.unlock();
                this.isIdle.set(false);
            }
        }

        private boolean allWorkDone() {
            boolean noTasksToUpdate = this.changelogReader.allChangelogsCompleted() || this.updatingTasks.isEmpty();
            return this.isRunning.get() && noTasksToUpdate && DefaultStateUpdater.this.tasksAndActions.isEmpty() && !DefaultStateUpdater.this.isTopologyResumed.get();
        }

        private void clearUpdatingAndPausedTasks() {
            this.updatingTasks.clear();
            this.pausedTasks.clear();
            this.changelogReader.clear();
        }

        private List<TaskAndAction> tasksAndActions() {
            ArrayList<TaskAndAction> tasksAndActionsToProcess = new ArrayList<TaskAndAction>(DefaultStateUpdater.this.tasksAndActions);
            DefaultStateUpdater.this.tasksAndActions.clear();
            return tasksAndActionsToProcess;
        }

        private void addTask(Task task) {
            TaskId taskId = task.id();
            Task existingTask = this.pausedTasks.get(taskId);
            if (existingTask != null) {
                throw new IllegalStateException((existingTask.isActive() ? "Active" : "Standby") + " task " + String.valueOf(taskId) + " already exist in paused tasks, should not try to add another " + (task.isActive() ? "active" : "standby") + " task with the same id. This indicates a bug. Please report at https://issues.apache.org/jira/projects/KAFKA/issues or to the dev-mailing list (https://kafka.apache.org/contact).");
            }
            existingTask = this.updatingTasks.get(taskId);
            if (existingTask != null) {
                throw new IllegalStateException((existingTask.isActive() ? "Active" : "Standby") + " task " + String.valueOf(taskId) + " already exist in updating tasks, should not try to add another " + (task.isActive() ? "active" : "standby") + " task with the same id. This indicates a bug. Please report at https://issues.apache.org/jira/projects/KAFKA/issues or to the dev-mailing list (https://kafka.apache.org/contact).");
            }
            if (this.isStateless(task)) {
                this.addToRestoredTasks((StreamTask)task);
                DefaultStateUpdater.this.log.info("Stateless active task " + String.valueOf(taskId) + " was added to the restored tasks of the state updater");
            } else if (DefaultStateUpdater.this.topologyMetadata.isPaused(taskId.topologyName())) {
                this.pausedTasks.put(taskId, task);
                this.changelogReader.register(task.changelogPartitions(), task.stateManager());
                DefaultStateUpdater.this.log.debug((task.isActive() ? "Active" : "Standby") + " task " + String.valueOf(taskId) + " was directly added to the paused tasks.");
            } else {
                this.updatingTasks.put(taskId, task);
                this.changelogReader.register(task.changelogPartitions(), task.stateManager());
                if (task.isActive()) {
                    DefaultStateUpdater.this.log.info("Stateful active task " + String.valueOf(taskId) + " was added to the state updater");
                    this.changelogReader.enforceRestoreActive();
                } else {
                    DefaultStateUpdater.this.log.info("Standby task " + String.valueOf(taskId) + " was added to the state updater");
                    if (this.updatingTasks.size() == 1) {
                        this.changelogReader.transitToUpdateStandby();
                    }
                }
            }
        }

        private void removeTask(TaskId taskId, CompletableFuture<StateUpdater.RemovedTaskResult> future) {
            try {
                if (!(this.removeUpdatingTask(taskId, future) || this.removePausedTask(taskId, future) || this.removeRestoredTask(taskId, future) || this.removeFailedTask(taskId, future))) {
                    future.complete(null);
                    DefaultStateUpdater.this.log.warn("Task {} could not be removed from the state updater because the state updater does not own this task.", (Object)taskId);
                }
            }
            catch (StreamsException streamsException) {
                this.handleStreamsExceptionWithTask(streamsException, taskId);
                future.completeExceptionally((Throwable)((Object)streamsException));
            }
            catch (RuntimeException runtimeException) {
                this.handleRuntimeException(runtimeException);
                future.completeExceptionally(runtimeException);
            }
        }

        private boolean removeUpdatingTask(TaskId taskId, CompletableFuture<StateUpdater.RemovedTaskResult> future) {
            if (!this.updatingTasks.containsKey(taskId)) {
                return false;
            }
            Task task = this.updatingTasks.get(taskId);
            this.prepareUpdatingTaskForRemoval(task);
            this.updatingTasks.remove(taskId);
            if (task.isActive()) {
                this.transitToUpdateStandbysIfOnlyStandbysLeft();
            }
            DefaultStateUpdater.this.log.info((task.isActive() ? "Active" : "Standby") + " task " + String.valueOf(task.id()) + " was removed from the updating tasks.");
            future.complete(new StateUpdater.RemovedTaskResult(task));
            return true;
        }

        private void prepareUpdatingTaskForRemoval(Task task) {
            this.measureCheckpointLatency(() -> task.maybeCheckpoint(true));
            Set<TopicPartition> changelogPartitions = task.changelogPartitions();
            this.changelogReader.unregister(changelogPartitions);
        }

        private boolean removePausedTask(TaskId taskId, CompletableFuture<StateUpdater.RemovedTaskResult> future) {
            if (!this.pausedTasks.containsKey(taskId)) {
                return false;
            }
            Task task = this.pausedTasks.get(taskId);
            this.preparePausedTaskForRemoval(task);
            this.pausedTasks.remove(taskId);
            DefaultStateUpdater.this.log.info((task.isActive() ? "Active" : "Standby") + " task " + String.valueOf(task.id()) + " was removed from the paused tasks.");
            future.complete(new StateUpdater.RemovedTaskResult(task));
            return true;
        }

        private void preparePausedTaskForRemoval(Task task) {
            Set<TopicPartition> changelogPartitions = task.changelogPartitions();
            this.changelogReader.unregister(changelogPartitions);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private boolean removeRestoredTask(TaskId taskId, CompletableFuture<StateUpdater.RemovedTaskResult> future) {
            DefaultStateUpdater.this.restoredActiveTasksLock.lock();
            try {
                Iterator iterator = DefaultStateUpdater.this.restoredActiveTasks.iterator();
                while (iterator.hasNext()) {
                    StreamTask restoredTask = (StreamTask)iterator.next();
                    if (!restoredTask.id().equals(taskId)) continue;
                    iterator.remove();
                    DefaultStateUpdater.this.log.info((restoredTask.isActive() ? "Active" : "Standby") + " task " + String.valueOf(restoredTask.id()) + " was removed from the restored tasks.");
                    future.complete(new StateUpdater.RemovedTaskResult(restoredTask));
                    boolean bl = true;
                    return bl;
                }
                boolean bl = false;
                return bl;
            }
            finally {
                DefaultStateUpdater.this.restoredActiveTasksLock.unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private boolean removeFailedTask(TaskId taskId, CompletableFuture<StateUpdater.RemovedTaskResult> future) {
            DefaultStateUpdater.this.exceptionsAndFailedTasksLock.lock();
            try {
                Iterator iterator = DefaultStateUpdater.this.exceptionsAndFailedTasks.iterator();
                while (iterator.hasNext()) {
                    StateUpdater.ExceptionAndTask exceptionAndTask = (StateUpdater.ExceptionAndTask)iterator.next();
                    Task failedTask = exceptionAndTask.task();
                    if (!failedTask.id().equals(taskId)) continue;
                    iterator.remove();
                    DefaultStateUpdater.this.log.info((failedTask.isActive() ? "Active" : "Standby") + " task " + String.valueOf(failedTask.id()) + " was removed from the failed tasks.");
                    future.complete(new StateUpdater.RemovedTaskResult(failedTask, exceptionAndTask.exception()));
                    boolean bl = true;
                    return bl;
                }
                boolean bl = false;
                return bl;
            }
            finally {
                DefaultStateUpdater.this.exceptionsAndFailedTasksLock.unlock();
            }
        }

        private void pauseTask(Task task) {
            TaskId taskId = task.id();
            try {
                this.measureCheckpointLatency(() -> task.maybeCheckpoint(true));
                this.pausedTasks.put(taskId, task);
                this.updatingTasks.remove(taskId);
                if (task.isActive()) {
                    this.transitToUpdateStandbysIfOnlyStandbysLeft();
                }
                DefaultStateUpdater.this.log.info((task.isActive() ? "Active" : "Standby") + " task " + String.valueOf(task.id()) + " was paused from the updating tasks and added to the paused tasks.");
            }
            catch (StreamsException streamsException) {
                this.handleStreamsExceptionWithTask(streamsException, taskId);
            }
        }

        private void resumeTask(Task task) {
            TaskId taskId = task.id();
            this.updatingTasks.put(taskId, task);
            this.pausedTasks.remove(taskId);
            if (task.isActive()) {
                DefaultStateUpdater.this.log.info("Stateful active task " + String.valueOf(task.id()) + " was resumed to the updating tasks of the state updater");
                this.changelogReader.enforceRestoreActive();
            } else {
                DefaultStateUpdater.this.log.info("Standby task " + String.valueOf(task.id()) + " was resumed to the updating tasks of the state updater");
                if (this.updatingTasks.size() == 1) {
                    this.changelogReader.transitToUpdateStandby();
                }
            }
        }

        private boolean isStateless(Task task) {
            return task.changelogPartitions().isEmpty() && task.isActive();
        }

        private void maybeCompleteRestoration(StreamTask task, Set<TopicPartition> restoredChangelogs) {
            Set<TopicPartition> changelogPartitions = task.changelogPartitions();
            if (restoredChangelogs.containsAll(changelogPartitions)) {
                try {
                    this.measureCheckpointLatency(() -> task.maybeCheckpoint(true));
                    this.changelogReader.unregister(changelogPartitions);
                    this.addToRestoredTasks(task);
                    DefaultStateUpdater.this.log.info("Stateful active task " + String.valueOf(task.id()) + " completed restoration");
                    this.transitToUpdateStandbysIfOnlyStandbysLeft();
                }
                catch (StreamsException streamsException) {
                    this.handleStreamsExceptionWithTask(streamsException, task.id());
                }
            }
        }

        private void transitToUpdateStandbysIfOnlyStandbysLeft() {
            if (this.onlyStandbyTasksUpdating()) {
                this.changelogReader.transitToUpdateStandby();
            }
        }

        private void addToRestoredTasks(StreamTask task) {
            DefaultStateUpdater.this.restoredActiveTasksLock.lock();
            try {
                DefaultStateUpdater.this.restoredActiveTasks.add(task);
                this.updatingTasks.remove(task.id());
                DefaultStateUpdater.this.log.debug("Active task " + String.valueOf(task.id()) + " was added to the restored tasks");
                DefaultStateUpdater.this.restoredActiveTasksCondition.signalAll();
            }
            finally {
                DefaultStateUpdater.this.restoredActiveTasksLock.unlock();
            }
        }

        private void maybeCheckpointTasks(long now) {
            long elapsedMsSinceLastCommit = now - DefaultStateUpdater.this.lastCommitMs;
            if (elapsedMsSinceLastCommit > DefaultStateUpdater.this.commitIntervalMs) {
                if (DefaultStateUpdater.this.log.isDebugEnabled()) {
                    DefaultStateUpdater.this.log.debug("Checkpointing state of all restoring tasks since {}ms has elapsed (commit interval is {}ms)", (Object)elapsedMsSinceLastCommit, (Object)DefaultStateUpdater.this.commitIntervalMs);
                }
                this.measureCheckpointLatency(() -> {
                    for (Task task : this.updatingTasks.values()) {
                        try {
                            task.maybeCheckpoint(false);
                        }
                        catch (StreamsException streamsException) {
                            this.handleStreamsExceptionWithTask(streamsException, task.id());
                        }
                    }
                });
                DefaultStateUpdater.this.lastCommitMs = now;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void measureCheckpointLatency(Runnable actionToMeasure) {
            long startMs = DefaultStateUpdater.this.time.milliseconds();
            try {
                actionToMeasure.run();
            }
            finally {
                this.totalCheckpointLatency += Math.max(0L, DefaultStateUpdater.this.time.milliseconds() - startMs);
            }
        }

        private void recordMetrics(long now, long totalLatency, long totalWaitLatency) {
            long totalRestoreLatency = Math.max(0L, totalLatency - totalWaitLatency - this.totalCheckpointLatency);
            this.updaterMetrics.idleRatioSensor.record((double)totalWaitLatency / (double)totalLatency, now);
            this.updaterMetrics.checkpointRatioSensor.record((double)this.totalCheckpointLatency / (double)totalLatency, now);
            if (this.changelogReader.isRestoringActive()) {
                this.updaterMetrics.activeRestoreRatioSensor.record((double)totalRestoreLatency / (double)totalLatency, now);
                this.updaterMetrics.standbyRestoreRatioSensor.record(0.0, now);
            } else {
                this.updaterMetrics.standbyRestoreRatioSensor.record((double)totalRestoreLatency / (double)totalLatency, now);
                this.updaterMetrics.activeRestoreRatioSensor.record(0.0, now);
            }
            this.totalCheckpointLatency = 0L;
        }
    }

    private class StateUpdaterMetrics {
        private static final String STATE_LEVEL_GROUP = "stream-state-updater-metrics";
        private static final String IDLE_RATIO_DESCRIPTION = "The fraction of time the thread spent on being idle";
        private static final String RESTORE_RATIO_DESCRIPTION = "The fraction of time the thread spent on restoring active tasks";
        private static final String UPDATE_RATIO_DESCRIPTION = "The fraction of time the thread spent on updating standby tasks";
        private static final String CHECKPOINT_RATIO_DESCRIPTION = "The fraction of time the thread spent on checkpointing tasks restored progress";
        private static final String RESTORE_RECORDS_RATE_DESCRIPTION = "The average per-second number of records restored";
        private static final String RESTORE_RATE_DESCRIPTION = "The average per-second number of restore calls triggered";
        private final Sensor restoreSensor;
        private final Sensor idleRatioSensor;
        private final Sensor activeRestoreRatioSensor;
        private final Sensor standbyRestoreRatioSensor;
        private final Sensor checkpointRatioSensor;
        private final Deque<Sensor> allSensors = new LinkedList<Sensor>();
        private final Deque<MetricName> allMetricNames = new LinkedList<MetricName>();

        private StateUpdaterMetrics(StreamsMetricsImpl metrics, String threadId) {
            LinkedHashMap<String, String> threadLevelTags = new LinkedHashMap<String, String>();
            threadLevelTags.put("thread-id", threadId);
            Metrics metricsRegistry = metrics.metricsRegistry();
            MetricName metricName = metricsRegistry.metricName("active-restoring-tasks", STATE_LEVEL_GROUP, "The number of active tasks currently undergoing restoration", threadLevelTags);
            metricsRegistry.addMetric(metricName, (config, now) -> this$0.stateUpdaterThread != null ? (double)this$0.stateUpdaterThread.numRestoringActiveTasks() : 0.0);
            this.allMetricNames.push(metricName);
            metricName = metricsRegistry.metricName("standby-updating-tasks", STATE_LEVEL_GROUP, "The number of standby tasks currently undergoing state update", threadLevelTags);
            metricsRegistry.addMetric(metricName, (config, now) -> this$0.stateUpdaterThread != null ? (double)this$0.stateUpdaterThread.numUpdatingStandbyTasks() : 0.0);
            this.allMetricNames.push(metricName);
            metricName = metricsRegistry.metricName("active-paused-tasks", STATE_LEVEL_GROUP, "The number of active tasks paused restoring", threadLevelTags);
            metricsRegistry.addMetric(metricName, (config, now) -> this$0.stateUpdaterThread != null ? (double)this$0.stateUpdaterThread.numPausedActiveTasks() : 0.0);
            this.allMetricNames.push(metricName);
            metricName = metricsRegistry.metricName("standby-paused-tasks", STATE_LEVEL_GROUP, "The number of standby tasks paused state update", threadLevelTags);
            metricsRegistry.addMetric(metricName, (config, now) -> this$0.stateUpdaterThread != null ? (double)this$0.stateUpdaterThread.numPausedStandbyTasks() : 0.0);
            this.allMetricNames.push(metricName);
            this.idleRatioSensor = metrics.threadLevelSensor(threadId, "idle-ratio", Sensor.RecordingLevel.INFO, new Sensor[0]);
            this.idleRatioSensor.add(new MetricName("idle-ratio", STATE_LEVEL_GROUP, IDLE_RATIO_DESCRIPTION, threadLevelTags), (MeasurableStat)new Avg());
            this.allSensors.add(this.idleRatioSensor);
            this.activeRestoreRatioSensor = metrics.threadLevelSensor(threadId, "active-restore-ratio", Sensor.RecordingLevel.INFO, new Sensor[0]);
            this.activeRestoreRatioSensor.add(new MetricName("active-restore-ratio", STATE_LEVEL_GROUP, RESTORE_RATIO_DESCRIPTION, threadLevelTags), (MeasurableStat)new Avg());
            this.allSensors.add(this.activeRestoreRatioSensor);
            this.standbyRestoreRatioSensor = metrics.threadLevelSensor(threadId, "standby-update-ratio", Sensor.RecordingLevel.INFO, new Sensor[0]);
            this.standbyRestoreRatioSensor.add(new MetricName("standby-update-ratio", STATE_LEVEL_GROUP, UPDATE_RATIO_DESCRIPTION, threadLevelTags), (MeasurableStat)new Avg());
            this.allSensors.add(this.standbyRestoreRatioSensor);
            this.checkpointRatioSensor = metrics.threadLevelSensor(threadId, "checkpoint-ratio", Sensor.RecordingLevel.INFO, new Sensor[0]);
            this.checkpointRatioSensor.add(new MetricName("checkpoint-ratio", STATE_LEVEL_GROUP, CHECKPOINT_RATIO_DESCRIPTION, threadLevelTags), (MeasurableStat)new Avg());
            this.allSensors.add(this.checkpointRatioSensor);
            this.restoreSensor = metrics.threadLevelSensor(threadId, "restore-records", Sensor.RecordingLevel.INFO, new Sensor[0]);
            this.restoreSensor.add(new MetricName("restore-records-rate", STATE_LEVEL_GROUP, RESTORE_RECORDS_RATE_DESCRIPTION, threadLevelTags), (MeasurableStat)new Rate());
            this.restoreSensor.add(new MetricName("restore-call-rate", STATE_LEVEL_GROUP, RESTORE_RATE_DESCRIPTION, threadLevelTags), (MeasurableStat)new Rate((SampledStat)new WindowedCount()));
            this.allSensors.add(this.restoreSensor);
        }

        void clear() {
            while (!this.allSensors.isEmpty()) {
                DefaultStateUpdater.this.metrics.removeSensor(this.allSensors.pop());
            }
            while (!this.allMetricNames.isEmpty()) {
                DefaultStateUpdater.this.metrics.removeMetric(this.allMetricNames.pop());
            }
        }
    }
}

