/****************************************************************************
 * drivers/ioexpander/tca64xx.h
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.  The
 * ASF licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 ****************************************************************************/

/* References:
 *   "16-bit I2C-bus and SMBus I/O port with interrupt product datasheet",
 *   Rev. 08 - 22 October 2009, NXP
 */

#ifndef __DRIVERS_IOEXPANDER_TCA64XX_H
#define __DRIVERS_IOEXPANDER_TCA64XX_H

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>

#include <nuttx/wdog.h>
#include <nuttx/clock.h>
#include <nuttx/mutex.h>
#include <nuttx/wqueue.h>
#include <nuttx/ioexpander/ioexpander.h>
#include <nuttx/ioexpander/tca64xx.h>

#include <nuttx/i2c/i2c_master.h>
#include <nuttx/irq.h>

#if defined(CONFIG_IOEXPANDER) && defined(CONFIG_IOEXPANDER_TCA64XX)

/****************************************************************************
 * Pre-processor Definitions
 ****************************************************************************/

/* Configuration ************************************************************/

/* Prerequisites:
 *   CONFIG_I2C
 *     I2C support is required
 *   CONFIG_IOEXPANDER
 *     Enables I/O expander support
 *
 * CONFIG_IOEXPANDER_TCA64XX
 *   Enables support for the TCA64XX driver (Needs CONFIG_INPUT)
 * CONFIG_TCA64XX_MULTIPLE
 *   Can be defined to support multiple TCA64XX devices on board.
 * CONFIG_TCA64XX_INT_NCALLBACKS
 *   Maximum number of supported pin interrupt callbacks.
 * CONFIG_TCA64XX_INT_POLL
 *   Enables a poll for missed interrupts
 * CONFIG_TCA64XX_INT_POLLDELAY
 *   If CONFIG_TCA64XX_INT_POLL=y, then this is the delay in microseconds
 *   between polls for missed interrupts.
 */

#ifndef CONFIG_I2C
#  error "CONFIG_I2C is required by TCA64XX"
#endif

#ifdef CONFIG_IOEXPANDER_INT_ENABLE
#  ifndef CONFIG_TCA64XX_INT_NCALLBACKS
#    define CONFIG_TCA64XX_INT_NCALLBACKS 4
#  endif
#endif

#ifdef CONFIG_IOEXPANDER_INT_ENABLE
#  ifndef CONFIG_SCHED_WORKQUEUE
#    error Work queue support required.  CONFIG_SCHED_WORKQUEUE must be selected.
#  endif
#endif

#ifndef CONFIG_TCA64XX_INT_POLLDELAY
#  define CONFIG_TCA64XX_INT_POLLDELAY 500000
#endif

/* TCA64XX Definitions ******************************************************/

/* I2C frequency */

#define TCA64XX_I2C_MAXFREQUENCY        400000       /* 400KHz */

/* TCA64XX Parts ************************************************************/

#define TCA64XX_INVALID_REG             0xff

#define TCA6408_INPUT_REG               0x00
#define TCA6408_OUTPUT_REG              0x01
#define TCA6408_POLARITY_REG            0x02
#define TCA6408_CONFIG_REG              0x03

#define TCA6408_NR_GPIOS                8

#define TCA6416_INPUT0_REG              0x00
#define TCA6416_INPUT1_REG              0x01
#define TCA6416_OUTPUT0_REG             0x02
#define TCA6416_OUTPUT1_REG             0x03
#define TCA6416_POLARITY0_REG           0x04
#define TCA6416_POLARITY1_REG           0x05
#define TCA6416_CONFIG0_REG             0x06
#define TCA6416_CONFIG1_REG             0x07

#define TCA6416_NR_GPIOS                16

#define TCA6424_INPUT0_REG              0x00
#define TCA6424_INPUT1_REG              0x01
#define TCA6424_INPUT2_REG              0x02
#define TCA6424_OUTPUT0_REG             0x04
#define TCA6424_OUTPUT1_REG             0x05
#define TCA6424_OUTPUT2_REG             0x06
#define TCA6424_POLARITY0_REG           0x08
#define TCA6424_POLARITY1_REG           0x09
#define TCA6424_POLARITY2_REG           0x0A
#define TCA6424_CONFIG0_REG             0x0C
#define TCA6424_CONFIG1_REG             0x0D
#define TCA6424_CONFIG2_REG             0x0E

#define TCA6424_NR_GPIOS                24

#define TCA64XX_NR_GPIO_MAX             TCA6424_NR_GPIOS

#define PCAL6416A_INPUT0_REG            0x00
#define PCAL6416A_INPUT1_REG            0x01
#define PCAL6416A_OUTPUT0_REG           0x02
#define PCAL6416A_OUTPUT1_REG           0x03
#define PCAL6416A_POLARITY0_REG         0x04
#define PCAL6416A_POLARITY1_REG         0x05
#define PCAL6416A_CONFIG0_REG           0x06
#define PCAL6416A_CONFIG1_REG           0x07
#define PCAL6416A_PU_ENABLE0_REG        0x46
#define PCAL6416A_PU_ENABLE1_REG        0x47
#define PCAL6416A_PUPD_SELECT0_REG      0x48
#define PCAL6416A_PUPD_SELECT1_REG      0x49
#define PCAL6416A_NR_GPIOS              16

/* 1us (datasheet: reset pulse duration (Tw) is 4ns */

#define TCA64XX_TW                      1

/* 1us (datasheet: time to reset (Treset) is 600ns */

#define TCA64XX_TRESET                  1

#define TCA64XX_IRQ_TYPE_EDGE_BOTH      0x00000000
#define TCA64XX_IRQ_TYPE_EDGE_RISING    0x00000001
#define TCA64XX_IRQ_TYPE_EDGE_FALLING   0x00000002
#define TCA64XX_IRQ_TYPE_LEVEL_HIGH     0x00000001
#define TCA64XX_IRQ_TYPE_LEVEL_LOW      0x00000002

#define TCA64XX_IRQ_TYPE_EDGE           0x00000000
#define TCA64XX_IRQ_TYPE_LEVEL          0x00000001

#define TCA64XX_POLLDELAY       (CONFIG_TCA64XX_INT_POLLDELAY / USEC_PER_TICK)

#define TCA64_LEVEL_SENSITIVE(d,p) \
  (((d)->trigger  & ((ioe_pinset_t)1 << (p))) == 0)
#define TCA64_LEVEL_HIGH(d,p) \
  (((d)->level[0] & ((ioe_pinset_t)1 << (p))) != 0)
#define TCA64_LEVEL_LOW(d,p) \
  (((d)->level[1] & ((ioe_pinset_t)1 << (p))) != 0)

#define TCA64_EDGE_SENSITIVE(d,p) \
  (((d)->trigger  & ((ioe_pinset_t)1 << (p))) != 0)
#define TCA64_EDGE_RISING(d,p) \
  (((d)->level[0] & ((ioe_pinset_t)1 << (p))) != 0)
#define TCA64_EDGE_FALLING(d,p) \
  (((d)->level[1] & ((ioe_pinset_t)1 << (p))) != 0)
#define TCA64_EDGE_BOTH(d,p) \
  (TCA64_LEVEL_RISING(d,p) && TCA64_LEVEL_FALLING(d,p))

/****************************************************************************
 * Public Types
 ****************************************************************************/

/* This structure represents the configuration of one part */

struct tca64_part_s
{
  uint8_t tp_id;       /* Part ID (see enum tca64xx_part_e) */
  uint8_t tp_ngpios;   /* Number of supported GPIOs */
  uint8_t tp_input;    /* Address of first input register */
  uint8_t tp_output;   /* Address of first output register */
  uint8_t tp_polarity; /* Address of first polarity register */
  uint8_t tp_config;   /* Address of first configuration register */
  uint8_t tp_puenable;
  uint8_t tp_puselect;
};

#ifdef CONFIG_IOEXPANDER_INT_ENABLE
/* This type represents on registered pin interrupt callback */

struct tca64_callback_s
{
  ioe_pinset_t pinset;              /* Set of pin interrupts that will generate
                                     * the callback. */
  ioe_callback_t cbfunc;            /* The saved callback function pointer */
  FAR void *cbarg;                  /* Callback argument */
};
#endif

/* This structure represents the state of the TCA64XX driver */

struct tca64_dev_s
{
  struct ioexpander_dev_s dev;       /* Nested structure to allow casting as public gpio
                                      * expander. */
  FAR struct tca64_config_s *config; /* Board configuration data */
  FAR struct i2c_master_s *i2c;      /* Saved I2C driver instance */
  uint8_t part;                      /* TCA64xx part ID (see enum tca64xx_part_e) */
  mutex_t lock;                      /* Mutual exclusion */

#ifdef CONFIG_IOEXPANDER_INT_ENABLE
#ifdef CONFIG_TCA64XX_INT_POLL
  struct wdog_s wdog;                /* Timer used to poll for missed interrupts */
#endif

  ioe_pinset_t input;                /* Last input registers */
  ioe_pinset_t intstat;              /* Pending interrupts */
  ioe_pinset_t trigger;              /* Bit encoded: 0=level 1=edge */
  ioe_pinset_t level[2];             /* Bit encoded: 01=high/rising, 10 low/falling, 11 both */
  struct work_s work;                /* Supports the interrupt handling "bottom half" */

  /* Saved callback information for each I/O expander client */

  struct tca64_callback_s cb[CONFIG_TCA64XX_INT_NCALLBACKS];
#endif
};

#endif /* CONFIG_IOEXPANDER && CONFIG_IOEXPANDER_TCA64XX */
#endif /* __DRIVERS_IOEXPANDER_TCA64XX_H */
