/****************************************************************************
 * include/gcov.h
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.  The
 * ASF licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 ****************************************************************************/

#ifndef __INCLUDE_GCOV_H
#define __INCLUDE_GCOV_H

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <sys/types.h>

/****************************************************************************
 * Pre-processor Definitions
 ****************************************************************************/

/* The GCOV 12 gcno/gcda format has slight change,
 * Please refer to gcov-io.h in the GCC 12 for
 * more details.
 */

#if __GNUC__ >= 12
#  define GCOV_12_FORMAT
#endif

#if __GNUC__ >= 14
#  define GCOV_COUNTERS 9u
#elif __GNUC__ >= 10
#  define GCOV_COUNTERS 8u
#elif __GNUC__ >= 8
#  define GCOV_COUNTERS 9u
#else
#  define GCOV_COUNTERS 10u
#endif

/****************************************************************************
 * Public Types
 ****************************************************************************/

struct gcov_fn_info;
typedef uint64_t gcov_type;

/** Profiling data per object file
 *
 * This data is generated by gcc during compilation and doesn't change
 * at run-time with the exception of the next pointer.
 */

struct gcov_info
{
  unsigned int version;                /* Gcov version (same as GCC version) */
  FAR struct gcov_info *next;          /* List head for a singly-linked list */
  unsigned int stamp;                  /* Uniquifying time stamp */
#ifdef GCOV_12_FORMAT
  unsigned int checksum;               /* unique object checksum */
#endif
  FAR const char *filename;            /* Name of the associated gcda data file */
  void (*merge[GCOV_COUNTERS])(FAR gcov_type *, unsigned int);
  unsigned int n_functions;            /* number of instrumented functions */
  FAR struct gcov_fn_info **functions; /* function information */
};

/****************************************************************************
 * Public Data
 ****************************************************************************/

extern FAR struct gcov_info *__gcov_info_start;
extern FAR struct gcov_info *__gcov_info_end;

/****************************************************************************
 * Public Function Prototypes
 ****************************************************************************/

#ifdef __cplusplus
#define EXTERN extern "C"
extern "C"
{
#else
#define EXTERN extern
#endif

/****************************************************************************
 * Name: __gcov_reset
 *
 * Description:
 *   Set all counters to zero.
 *
 ****************************************************************************/

extern void __gcov_reset(void);

/****************************************************************************
 * Name: __gcov_dump
 *
 * Description:
 *   Write profile information to a file.
 *
 ****************************************************************************/

extern void __gcov_dump(void);

/****************************************************************************
 * Name: __gcov_info_to_gcda
 *
 * Description:
 *   Convert the gcov information referenced by INFO to a gcda data stream.
 *
 * Parameters:
 *   info - Pointer to the gcov information.
 *   filename - Callback function to get the filename.
 *   dump - Callback function to write the gcda data.
 *   allocate - Callback function to allocate memory.
 *   arg - User-provided argument.
 *
 ****************************************************************************/

extern void __gcov_info_to_gcda(FAR const struct gcov_info *info,
                                FAR void (*filename)(FAR const char *,
                                                     FAR void *),
                                FAR void (*dump)(FAR const void *,
                                                 size_t, FAR void *),
                                FAR void *(*allocate)(unsigned int,
                                                      FAR void *),
                                FAR void *arg);

/****************************************************************************
 * Name: __gcov_filename_to_gcfn
 *
 * Description:
 *   Convert the filename to a gcfn data stream.
 *
 * Parameters:
 *   filename - Pointer to the filename.
 *   dump - Callback function to write the gcfn data.
 *   arg - User-provided argument.
 *
 ****************************************************************************/

extern void __gcov_filename_to_gcfn(FAR const char *filename,
                                    FAR void (*dump)(FAR const void *,
                                                     unsigned int,
                                                     FAR void *),
                                     FAR void *arg);

/****************************************************************************
 * Name: __gcov_dump_to_memory
 *
 * Description:
 *   Dump gcov data directly to memory
 *
 * Parameters:
 *   ptr - Memory Address
 *   size - Memory block size
 *
 ****************************************************************************/

size_t __gcov_dump_to_memory(FAR void *ptr, size_t size);

#undef EXTERN
#ifdef __cplusplus
}
#endif

#endif /* __INCLUDE_GCOV_H */
