/* -*- Mode: C++; c-file-style: "stroustrup"; indent-tabs-mode: nil -*- */
/*
 * Db.h
 *   Pure virtual base class for database interaction classes.
 *
 * $Id: Db.h,v 1.12 2002/01/02 23:54:22 benoit Exp $
 *
 * Copyright (c) 2000-2001 Remi Lefebvre <remi@debian.org>
 * Copyright (c) 2000 Benoit Joly <benoit@dhis.net>
 * Copyright (c) 2000 Luca Filipozzi <lfilipoz@dhis.net>
 *
 * DDT comes with ABSOLUTELY NO WARRANTY and is licenced under the
 * GNU General Public License (version 2 or later). This license
 * can be retrieved from http://www.gnu.org/copyleft/gnu.html.
 *
 */

#ifndef DB_H
#define DB_H

#include <vector.h>
#include "global.h"
#include "Dns.h"
#include "Logger.h"
#include "UserAccount.h"
#include "DnsRecord.h"

/**
 * Pure virtual class for database interaction.
 */
class Db
{
  public:
    /// Constructor
    Db() { };

    /// destructor
    virtual ~Db() { };


    /*********** userAccounts ***********/

    /// Add a user account.
    virtual int addUserAccount(UserAccount *account) = 0;

    /// Remove a user account.
    virtual void delUserAccount(int id) = 0;

    /// check if a given account exist
    virtual bool accountIdExists(int id) = 0;

    /// Modify a field in a user accounts table.
    virtual void modUserAccount(int id, char *field, char *value) = 0;
    /// modify an int field
    virtual void modUserAccount(int id, char *field, int value) = 0;

    /// Fetch an account from the database by id number.
    virtual bool fetchAccountInfo(int id, UserAccount *account) = 0;

    /// Get the account id number from an fqdn.
    virtual int findUserAccountIdFromFqdn(const char *fqdn) = 0;

    /// get the IP address of an accountid
    virtual unsigned long fetchAcctAddr(int id) = 0;


    /// Dump all known informations for a given host identification.
//    virtual int dumpAccountInfo (int id) {};

    /// update the last access field
//    virtual void updateLastAccess(int id) = 0;

    virtual void getActiveAccounts() = 0;

    virtual int pruneActiveAccount(void (*func)(int, unsigned long, 
                                                time_t)) = 0;


    /***********  dnsRecords *********/

    /// Add a DNS record.
    virtual void addDnsRecord(int id, const char *dname, DnsRecordType type,
                              const char *data) = 0;

    /// Remove every DNS records
    virtual void delDnsRecords(int id) = 0;

    /// Remove a DNS record.    
    virtual void delDnsRecord(int id, const char *dname, DnsRecordType type,
                              const char *data) = 0;

    virtual void listDnsRecords(int id, vector<DnsRecord> &dnsRecordList) = 0;
    
  protected:
    /// Logger object
    Logger *log;
};

#endif // DB_H
