/*
 * Copyright (c) 2018-2020 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __ARM_FEATURE_SVE

#include <algorithm>

#include "arm_gemm.hpp"

#include "../../asmlib.hpp"
#include "../../utils.hpp"

namespace arm_gemm {

void sve_hybrid_fp32_mla_4VLx4(const float *A, int lda, const float *B, float *C, int ldc, int M, int N, int K, const float *bias, Activation act, bool accumulate) {
    const int K_stride = K;
    const long loops_count = ((K + 4) / 8) - 1;
    K -= loops_count * 8;
    const long regs_count = (K / 4) - 1;
    K -= (regs_count + 1) * 4;
    const long leftovers = K;
    float nullbias[256];
    if (!accumulate && !bias) {
        memset(nullbias, 0, (4 * get_vector_length<float>() * sizeof(float)));
    }
    float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
    float maxval =   static_cast<float>(std::numeric_limits<float>::infinity());
    const float * const minptr = &minval;
    const float * const maxptr = &maxval;

    switch(act.type)
    {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            minval = 0.0f;
            break;
    }

    int rows_to_compute;

    for (int y=0; y<M; y+=rows_to_compute) {
        const float * const a_ptr0_base = A + (y * lda);
        const unsigned long ldab = lda * sizeof(float);

        float *c_ptr0 = C + (y * ldc);

        rows_to_compute = M-y;
        if (rows_to_compute > 4) {
            if (rows_to_compute % 4) {
                rows_to_compute = 4 - 1;
            } else {
                rows_to_compute = 4;
            }
        }

        for (int x0=0; x0<N; x0+=(4 * get_vector_length<float>())) {
            const long width = std::min((unsigned long)N-x0, (4 * get_vector_length<float>()));
            long loops = loops_count;
            long regs = regs_count;
            long temp = 0;
            long blocks = leftovers;
            const float *a_ptr0 = a_ptr0_base;
            const float *b_ptr0 = B + (K_stride * x0);
            const unsigned long ldcb = ldc * sizeof(float);
            const float *biasptr = bias ? bias+x0 : nullbias;

            switch(rows_to_compute) {
                case 1:
                    __asm __volatile (
                        "whilelt p6.s, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.s\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p3.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "ld1w z16.s, p0/z, [%[biasptr]]\n"
                        "ld1w z17.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[biasptr], #3, MUL VL]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z16.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z16.s, z12.s, z0.s[3]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        "fmla z17.s, z13.s, z0.s[3]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[3]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[3]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[0]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.s, z9.s, z4.s[0]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.s, z10.s, z4.s[0]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.s, z11.s, z4.s[0]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[1]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0], #-0x10]\n"
                        "fmla z17.s, z13.s, z4.s[1]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z18.s, z14.s, z4.s[1]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z19.s, z15.s, z4.s[1]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[2]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z9.s, z4.s[2]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z10.s, z4.s[2]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z11.s, z4.s[2]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[3]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z17.s, z13.s, z4.s[3]\n"
                        "fmla z18.s, z14.s, z4.s[3]\n"
                        "fmla z19.s, z15.s, z4.s[3]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "fmla z16.s, z12.s, z0.s[3]\n"
                        "fmla z17.s, z13.s, z0.s[3]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[3]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[3]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[0]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.s, z9.s, z4.s[0]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.s, z10.s, z4.s[0]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.s, z11.s, z4.s[0]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[1]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.s, z13.s, z4.s[1]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.s, z14.s, z4.s[1]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z15.s, z4.s[1]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[2]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z9.s, z4.s[2]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z10.s, z4.s[2]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z11.s, z4.s[2]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[3]\n"
                        "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                        "fmla z17.s, z13.s, z4.s[3]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z18.s, z14.s, z4.s[3]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        "fmla z19.s, z15.s, z4.s[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "b 5f\n"
                        "4:\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "ld1rqw z4.s, p6/z, [%[a_ptr0]]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        "fmla z16.s, z12.s, z0.s[3]\n"
                        "fmla z17.s, z13.s, z0.s[3]\n"
                        "fmla z18.s, z14.s, z0.s[3]\n"
                        "fmla z19.s, z15.s, z0.s[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z17.s, z9.s, z4.s[0]\n"
                        "fmla z18.s, z10.s, z4.s[0]\n"
                        "fmla z19.s, z11.s, z4.s[0]\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z17.s, z13.s, z4.s[1]\n"
                        "fmla z18.s, z14.s, z4.s[1]\n"
                        "fmla z19.s, z15.s, z4.s[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[2]\n"
                        "fmla z17.s, z9.s, z4.s[2]\n"
                        "fmla z18.s, z10.s, z4.s[2]\n"
                        "fmla z19.s, z11.s, z4.s[2]\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "st1w z16.s, p0, [%[c_ptr0]]\n"
                        "st1w z17.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "st1w z18.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "st1w z19.s, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                    );
                    break;
                case 2:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "c_ptr1 .req X1\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "whilelt p6.s, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.s\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p3.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "ld1w z16.s, p0/z, [%[biasptr]]\n"
                        "ld1w z17.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[biasptr], #3, MUL VL]\n"
                        "mov z20.d, z16.d\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "mov z21.d, z17.d\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "mov z22.d, z18.d\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "mov z23.d, z19.d\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z16.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1w z20.s, p0/z, [c_ptr1]\n"
                        "ld1w z21.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1w z22.s, p2/z, [c_ptr1, #2, MUL VL]\n"
                        "ld1w z23.s, p3/z, [c_ptr1, #3, MUL VL]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.s, z8.s, z1.s[0]\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        "fmla z21.s, z9.s, z1.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z22.s, z10.s, z1.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "fmla z23.s, z11.s, z1.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        "fmla z20.s, z12.s, z1.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        "fmla z21.s, z13.s, z1.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "fmla z22.s, z14.s, z1.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "fmla z23.s, z15.s, z1.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.s, z8.s, z1.s[2]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "fmla z21.s, z9.s, z1.s[2]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z22.s, z10.s, z1.s[2]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "fmla z23.s, z11.s, z1.s[2]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[3]\n"
                        "fmla z20.s, z12.s, z1.s[3]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[3]\n"
                        "fmla z21.s, z13.s, z1.s[3]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[3]\n"
                        "fmla z22.s, z14.s, z1.s[3]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[3]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0], #-0x10]\n"
                        "fmla z23.s, z15.s, z1.s[3]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[0]\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1, #-0x10]\n"
                        "fmla z20.s, z8.s, z5.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.s, z9.s, z4.s[0]\n"
                        "fmla z21.s, z9.s, z5.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.s, z10.s, z4.s[0]\n"
                        "fmla z22.s, z10.s, z5.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z11.s, z4.s[0]\n"
                        "fmla z23.s, z11.s, z5.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[1]\n"
                        "fmla z20.s, z12.s, z5.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z13.s, z4.s[1]\n"
                        "fmla z21.s, z13.s, z5.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z14.s, z4.s[1]\n"
                        "fmla z22.s, z14.s, z5.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z15.s, z4.s[1]\n"
                        "fmla z23.s, z15.s, z5.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.s, z8.s, z5.s[2]\n"
                        "fmla z17.s, z9.s, z4.s[2]\n"
                        "fmla z21.s, z9.s, z5.s[2]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z18.s, z10.s, z4.s[2]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z22.s, z10.s, z5.s[2]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.s, z11.s, z4.s[2]\n"
                        "fmla z23.s, z11.s, z5.s[2]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[3]\n"
                        "fmla z20.s, z12.s, z5.s[3]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.s, z13.s, z4.s[3]\n"
                        "fmla z21.s, z13.s, z5.s[3]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.s, z14.s, z4.s[3]\n"
                        "fmla z22.s, z14.s, z5.s[3]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.s, z15.s, z4.s[3]\n"
                        "fmla z23.s, z15.s, z5.s[3]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.s, z8.s, z1.s[0]\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        "fmla z21.s, z9.s, z1.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z22.s, z10.s, z1.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "fmla z23.s, z11.s, z1.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "fmla z20.s, z12.s, z1.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "fmla z21.s, z13.s, z1.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "fmla z22.s, z14.s, z1.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "fmla z23.s, z15.s, z1.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.s, z8.s, z1.s[2]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "fmla z21.s, z9.s, z1.s[2]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z22.s, z10.s, z1.s[2]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "fmla z23.s, z11.s, z1.s[2]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[3]\n"
                        "fmla z20.s, z12.s, z1.s[3]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[3]\n"
                        "fmla z21.s, z13.s, z1.s[3]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[3]\n"
                        "fmla z22.s, z14.s, z1.s[3]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[3]\n"
                        "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                        "fmla z23.s, z15.s, z1.s[3]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[0]\n"
                        "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                        "fmla z20.s, z8.s, z5.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.s, z9.s, z4.s[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        "fmla z21.s, z9.s, z5.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.s, z10.s, z4.s[0]\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        "fmla z22.s, z10.s, z5.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z11.s, z4.s[0]\n"
                        "fmla z23.s, z11.s, z5.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[1]\n"
                        "fmla z20.s, z12.s, z5.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z13.s, z4.s[1]\n"
                        "fmla z21.s, z13.s, z5.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z14.s, z4.s[1]\n"
                        "fmla z22.s, z14.s, z5.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z15.s, z4.s[1]\n"
                        "fmla z23.s, z15.s, z5.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z20.s, z8.s, z5.s[2]\n"
                        "fmla z17.s, z9.s, z4.s[2]\n"
                        "fmla z21.s, z9.s, z5.s[2]\n"
                        "fmla z18.s, z10.s, z4.s[2]\n"
                        "fmla z22.s, z10.s, z5.s[2]\n"
                        "fmla z19.s, z11.s, z4.s[2]\n"
                        "fmla z23.s, z11.s, z5.s[2]\n"
                        "fmla z16.s, z12.s, z4.s[3]\n"
                        "fmla z20.s, z12.s, z5.s[3]\n"
                        "fmla z17.s, z13.s, z4.s[3]\n"
                        "fmla z21.s, z13.s, z5.s[3]\n"
                        "fmla z18.s, z14.s, z4.s[3]\n"
                        "fmla z22.s, z14.s, z5.s[3]\n"
                        "fmla z19.s, z15.s, z4.s[3]\n"
                        "fmla z23.s, z15.s, z5.s[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.s, z8.s, z1.s[0]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "fmla z21.s, z9.s, z1.s[0]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "fmla z22.s, z10.s, z1.s[0]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "fmla z23.s, z11.s, z1.s[0]\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.s, z12.s, z1.s[1]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "fmla z21.s, z13.s, z1.s[1]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "fmla z22.s, z14.s, z1.s[1]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "fmla z23.s, z15.s, z1.s[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "fmla z20.s, z8.s, z1.s[2]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "fmla z21.s, z9.s, z1.s[2]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "fmla z22.s, z10.s, z1.s[2]\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "fmla z23.s, z11.s, z1.s[2]\n"
                        "b 5f\n"
                        "4:\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.s, z8.s, z1.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "ld1rqw z4.s, p6/z, [%[a_ptr0]]\n"
                        "fmla z21.s, z9.s, z1.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "ld1rqw z5.s, p6/z, [a_ptr1]\n"
                        "fmla z22.s, z10.s, z1.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        "fmla z23.s, z11.s, z1.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        "fmla z20.s, z12.s, z1.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "fmla z21.s, z13.s, z1.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "fmla z22.s, z14.s, z1.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "fmla z23.s, z15.s, z1.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z20.s, z8.s, z1.s[2]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "fmla z21.s, z9.s, z1.s[2]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "fmla z22.s, z10.s, z1.s[2]\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "fmla z23.s, z11.s, z1.s[2]\n"
                        "fmla z16.s, z12.s, z0.s[3]\n"
                        "fmla z20.s, z12.s, z1.s[3]\n"
                        "fmla z17.s, z13.s, z0.s[3]\n"
                        "fmla z21.s, z13.s, z1.s[3]\n"
                        "fmla z18.s, z14.s, z0.s[3]\n"
                        "fmla z22.s, z14.s, z1.s[3]\n"
                        "fmla z19.s, z15.s, z0.s[3]\n"
                        "fmla z23.s, z15.s, z1.s[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.s, z8.s, z5.s[0]\n"
                        "fmla z17.s, z9.s, z4.s[0]\n"
                        "fmla z21.s, z9.s, z5.s[0]\n"
                        "fmla z18.s, z10.s, z4.s[0]\n"
                        "fmla z22.s, z10.s, z5.s[0]\n"
                        "fmla z19.s, z11.s, z4.s[0]\n"
                        "fmla z23.s, z11.s, z5.s[0]\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.s, z12.s, z5.s[1]\n"
                        "fmla z17.s, z13.s, z4.s[1]\n"
                        "fmla z21.s, z13.s, z5.s[1]\n"
                        "fmla z18.s, z14.s, z4.s[1]\n"
                        "fmla z22.s, z14.s, z5.s[1]\n"
                        "fmla z19.s, z15.s, z4.s[1]\n"
                        "fmla z23.s, z15.s, z5.s[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[2]\n"
                        "fmla z20.s, z8.s, z5.s[2]\n"
                        "fmla z17.s, z9.s, z4.s[2]\n"
                        "fmla z21.s, z9.s, z5.s[2]\n"
                        "fmla z18.s, z10.s, z4.s[2]\n"
                        "fmla z22.s, z10.s, z5.s[2]\n"
                        "fmla z19.s, z11.s, z4.s[2]\n"
                        "fmla z23.s, z11.s, z5.s[2]\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "st1w z16.s, p0, [%[c_ptr0]]\n"
                        "fmax z20.s, p7/m, z20.s, z14.s\n"
                        "fmax z21.s, p7/m, z21.s, z14.s\n"
                        "fmax z22.s, p7/m, z22.s, z14.s\n"
                        "st1w z17.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmax z23.s, p7/m, z23.s, z14.s\n"
                        "fmin z20.s, p7/m, z20.s, z15.s\n"
                        "fmin z21.s, p7/m, z21.s, z15.s\n"
                        "st1w z18.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "fmin z22.s, p7/m, z22.s, z15.s\n"
                        "fmin z23.s, p7/m, z23.s, z15.s\n"
                        "st1w z19.s, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        "st1w z20.s, p0, [c_ptr1]\n"
                        "st1w z21.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "st1w z22.s, p2, [c_ptr1, #2, MUL VL]\n"
                        "st1w z23.s, p3, [c_ptr1, #3, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq c_ptr1\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "cc", "memory"
                    );
                    break;
                case 3:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "c_ptr1 .req X2\n"
                        "c_ptr2 .req X3\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "whilelt p6.s, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.s\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p3.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "ld1w z16.s, p0/z, [%[biasptr]]\n"
                        "ld1w z17.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[biasptr], #3, MUL VL]\n"
                        "mov z20.d, z16.d\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "mov z21.d, z17.d\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "mov z22.d, z18.d\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "mov z23.d, z19.d\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "mov z24.d, z16.d\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "mov z25.d, z17.d\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "mov z26.d, z18.d\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z27.d, z19.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z16.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1w z20.s, p0/z, [c_ptr1]\n"
                        "ld1w z21.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1w z22.s, p2/z, [c_ptr1, #2, MUL VL]\n"
                        "ld1w z23.s, p3/z, [c_ptr1, #3, MUL VL]\n"
                        "ld1w z24.s, p0/z, [c_ptr2]\n"
                        "ld1w z25.s, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "ld1w z26.s, p2/z, [c_ptr2, #2, MUL VL]\n"
                        "ld1w z27.s, p3/z, [c_ptr2, #3, MUL VL]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.s, z8.s, z1.s[0]\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "fmla z24.s, z8.s, z2.s[0]\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        "fmla z21.s, z9.s, z1.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z25.s, z9.s, z2.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "fmla z22.s, z10.s, z1.s[0]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        "fmla z26.s, z10.s, z2.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        "fmla z23.s, z11.s, z1.s[0]\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        "fmla z27.s, z11.s, z2.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "fmla z20.s, z12.s, z1.s[1]\n"
                        "fmla z24.s, z12.s, z2.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "fmla z21.s, z13.s, z1.s[1]\n"
                        "fmla z25.s, z13.s, z2.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "fmla z22.s, z14.s, z1.s[1]\n"
                        "fmla z26.s, z14.s, z2.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "fmla z23.s, z15.s, z1.s[1]\n"
                        "fmla z27.s, z15.s, z2.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.s, z8.s, z1.s[2]\n"
                        "fmla z24.s, z8.s, z2.s[2]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z21.s, z9.s, z1.s[2]\n"
                        "fmla z25.s, z9.s, z2.s[2]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "fmla z22.s, z10.s, z1.s[2]\n"
                        "fmla z26.s, z10.s, z2.s[2]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "fmla z23.s, z11.s, z1.s[2]\n"
                        "fmla z27.s, z11.s, z2.s[2]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[3]\n"
                        "fmla z20.s, z12.s, z1.s[3]\n"
                        "fmla z24.s, z12.s, z2.s[3]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[3]\n"
                        "fmla z21.s, z13.s, z1.s[3]\n"
                        "fmla z25.s, z13.s, z2.s[3]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[3]\n"
                        "fmla z22.s, z14.s, z1.s[3]\n"
                        "fmla z26.s, z14.s, z2.s[3]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[3]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0], #-0x10]\n"
                        "fmla z23.s, z15.s, z1.s[3]\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1, #-0x10]\n"
                        "fmla z27.s, z15.s, z2.s[3]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[0]\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2, #-0x10]\n"
                        "fmla z20.s, z8.s, z5.s[0]\n"
                        "fmla z24.s, z8.s, z6.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.s, z9.s, z4.s[0]\n"
                        "fmla z21.s, z9.s, z5.s[0]\n"
                        "fmla z25.s, z9.s, z6.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.s, z10.s, z4.s[0]\n"
                        "fmla z22.s, z10.s, z5.s[0]\n"
                        "fmla z26.s, z10.s, z6.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z11.s, z4.s[0]\n"
                        "fmla z23.s, z11.s, z5.s[0]\n"
                        "fmla z27.s, z11.s, z6.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[1]\n"
                        "fmla z20.s, z12.s, z5.s[1]\n"
                        "fmla z24.s, z12.s, z6.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z13.s, z4.s[1]\n"
                        "fmla z21.s, z13.s, z5.s[1]\n"
                        "fmla z25.s, z13.s, z6.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z14.s, z4.s[1]\n"
                        "fmla z22.s, z14.s, z5.s[1]\n"
                        "fmla z26.s, z14.s, z6.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z15.s, z4.s[1]\n"
                        "fmla z23.s, z15.s, z5.s[1]\n"
                        "fmla z27.s, z15.s, z6.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.s, z8.s, z5.s[2]\n"
                        "fmla z24.s, z8.s, z6.s[2]\n"
                        "fmla z17.s, z9.s, z4.s[2]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z21.s, z9.s, z5.s[2]\n"
                        "fmla z25.s, z9.s, z6.s[2]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.s, z10.s, z4.s[2]\n"
                        "fmla z22.s, z10.s, z5.s[2]\n"
                        "fmla z26.s, z10.s, z6.s[2]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.s, z11.s, z4.s[2]\n"
                        "fmla z23.s, z11.s, z5.s[2]\n"
                        "fmla z27.s, z11.s, z6.s[2]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[3]\n"
                        "fmla z20.s, z12.s, z5.s[3]\n"
                        "fmla z24.s, z12.s, z6.s[3]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.s, z13.s, z4.s[3]\n"
                        "fmla z21.s, z13.s, z5.s[3]\n"
                        "fmla z25.s, z13.s, z6.s[3]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.s, z14.s, z4.s[3]\n"
                        "fmla z22.s, z14.s, z5.s[3]\n"
                        "fmla z26.s, z14.s, z6.s[3]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.s, z15.s, z4.s[3]\n"
                        "fmla z23.s, z15.s, z5.s[3]\n"
                        "fmla z27.s, z15.s, z6.s[3]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.s, z8.s, z1.s[0]\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "fmla z24.s, z8.s, z2.s[0]\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        "fmla z21.s, z9.s, z1.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z25.s, z9.s, z2.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "fmla z22.s, z10.s, z1.s[0]\n"
                        "fmla z26.s, z10.s, z2.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "fmla z23.s, z11.s, z1.s[0]\n"
                        "fmla z27.s, z11.s, z2.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "fmla z20.s, z12.s, z1.s[1]\n"
                        "fmla z24.s, z12.s, z2.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "fmla z21.s, z13.s, z1.s[1]\n"
                        "fmla z25.s, z13.s, z2.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "fmla z22.s, z14.s, z1.s[1]\n"
                        "fmla z26.s, z14.s, z2.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "fmla z23.s, z15.s, z1.s[1]\n"
                        "fmla z27.s, z15.s, z2.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.s, z8.s, z1.s[2]\n"
                        "fmla z24.s, z8.s, z2.s[2]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z21.s, z9.s, z1.s[2]\n"
                        "fmla z25.s, z9.s, z2.s[2]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "fmla z22.s, z10.s, z1.s[2]\n"
                        "fmla z26.s, z10.s, z2.s[2]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "fmla z23.s, z11.s, z1.s[2]\n"
                        "fmla z27.s, z11.s, z2.s[2]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[3]\n"
                        "fmla z20.s, z12.s, z1.s[3]\n"
                        "fmla z24.s, z12.s, z2.s[3]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[3]\n"
                        "fmla z21.s, z13.s, z1.s[3]\n"
                        "fmla z25.s, z13.s, z2.s[3]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[3]\n"
                        "fmla z22.s, z14.s, z1.s[3]\n"
                        "fmla z26.s, z14.s, z2.s[3]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[3]\n"
                        "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                        "fmla z23.s, z15.s, z1.s[3]\n"
                        "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                        "fmla z27.s, z15.s, z2.s[3]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[0]\n"
                        "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                        "fmla z20.s, z8.s, z5.s[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        "fmla z24.s, z8.s, z6.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.s, z9.s, z4.s[0]\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        "fmla z21.s, z9.s, z5.s[0]\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        "fmla z25.s, z9.s, z6.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.s, z10.s, z4.s[0]\n"
                        "fmla z22.s, z10.s, z5.s[0]\n"
                        "fmla z26.s, z10.s, z6.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z11.s, z4.s[0]\n"
                        "fmla z23.s, z11.s, z5.s[0]\n"
                        "fmla z27.s, z11.s, z6.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[1]\n"
                        "fmla z20.s, z12.s, z5.s[1]\n"
                        "fmla z24.s, z12.s, z6.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z13.s, z4.s[1]\n"
                        "fmla z21.s, z13.s, z5.s[1]\n"
                        "fmla z25.s, z13.s, z6.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z14.s, z4.s[1]\n"
                        "fmla z22.s, z14.s, z5.s[1]\n"
                        "fmla z26.s, z14.s, z6.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z15.s, z4.s[1]\n"
                        "fmla z23.s, z15.s, z5.s[1]\n"
                        "fmla z27.s, z15.s, z6.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z20.s, z8.s, z5.s[2]\n"
                        "fmla z24.s, z8.s, z6.s[2]\n"
                        "fmla z17.s, z9.s, z4.s[2]\n"
                        "fmla z21.s, z9.s, z5.s[2]\n"
                        "fmla z25.s, z9.s, z6.s[2]\n"
                        "fmla z18.s, z10.s, z4.s[2]\n"
                        "fmla z22.s, z10.s, z5.s[2]\n"
                        "fmla z26.s, z10.s, z6.s[2]\n"
                        "fmla z19.s, z11.s, z4.s[2]\n"
                        "fmla z23.s, z11.s, z5.s[2]\n"
                        "fmla z27.s, z11.s, z6.s[2]\n"
                        "fmla z16.s, z12.s, z4.s[3]\n"
                        "fmla z20.s, z12.s, z5.s[3]\n"
                        "fmla z24.s, z12.s, z6.s[3]\n"
                        "fmla z17.s, z13.s, z4.s[3]\n"
                        "fmla z21.s, z13.s, z5.s[3]\n"
                        "fmla z25.s, z13.s, z6.s[3]\n"
                        "fmla z18.s, z14.s, z4.s[3]\n"
                        "fmla z22.s, z14.s, z5.s[3]\n"
                        "fmla z26.s, z14.s, z6.s[3]\n"
                        "fmla z19.s, z15.s, z4.s[3]\n"
                        "fmla z23.s, z15.s, z5.s[3]\n"
                        "fmla z27.s, z15.s, z6.s[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.s, z8.s, z1.s[0]\n"
                        "fmla z24.s, z8.s, z2.s[0]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "fmla z21.s, z9.s, z1.s[0]\n"
                        "fmla z25.s, z9.s, z2.s[0]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "fmla z22.s, z10.s, z1.s[0]\n"
                        "fmla z26.s, z10.s, z2.s[0]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "fmla z23.s, z11.s, z1.s[0]\n"
                        "fmla z27.s, z11.s, z2.s[0]\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.s, z12.s, z1.s[1]\n"
                        "fmla z24.s, z12.s, z2.s[1]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "fmla z21.s, z13.s, z1.s[1]\n"
                        "fmla z25.s, z13.s, z2.s[1]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "fmla z22.s, z14.s, z1.s[1]\n"
                        "fmla z26.s, z14.s, z2.s[1]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "fmla z23.s, z15.s, z1.s[1]\n"
                        "fmla z27.s, z15.s, z2.s[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "fmla z20.s, z8.s, z1.s[2]\n"
                        "fmla z24.s, z8.s, z2.s[2]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "fmla z21.s, z9.s, z1.s[2]\n"
                        "fmla z25.s, z9.s, z2.s[2]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "fmla z22.s, z10.s, z1.s[2]\n"
                        "fmla z26.s, z10.s, z2.s[2]\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "fmla z23.s, z11.s, z1.s[2]\n"
                        "fmla z27.s, z11.s, z2.s[2]\n"
                        "b 5f\n"
                        "4:\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.s, z8.s, z1.s[0]\n"
                        "ld1rqw z4.s, p6/z, [%[a_ptr0]]\n"
                        "fmla z24.s, z8.s, z2.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "ld1rqw z5.s, p6/z, [a_ptr1]\n"
                        "fmla z21.s, z9.s, z1.s[0]\n"
                        "ld1rqw z6.s, p6/z, [a_ptr2]\n"
                        "fmla z25.s, z9.s, z2.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        "fmla z22.s, z10.s, z1.s[0]\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        "fmla z26.s, z10.s, z2.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        "fmla z23.s, z11.s, z1.s[0]\n"
                        "fmla z27.s, z11.s, z2.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "fmla z20.s, z12.s, z1.s[1]\n"
                        "fmla z24.s, z12.s, z2.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "fmla z21.s, z13.s, z1.s[1]\n"
                        "fmla z25.s, z13.s, z2.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "fmla z22.s, z14.s, z1.s[1]\n"
                        "fmla z26.s, z14.s, z2.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "fmla z23.s, z15.s, z1.s[1]\n"
                        "fmla z27.s, z15.s, z2.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z20.s, z8.s, z1.s[2]\n"
                        "fmla z24.s, z8.s, z2.s[2]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "fmla z21.s, z9.s, z1.s[2]\n"
                        "fmla z25.s, z9.s, z2.s[2]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "fmla z22.s, z10.s, z1.s[2]\n"
                        "fmla z26.s, z10.s, z2.s[2]\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "fmla z23.s, z11.s, z1.s[2]\n"
                        "fmla z27.s, z11.s, z2.s[2]\n"
                        "fmla z16.s, z12.s, z0.s[3]\n"
                        "fmla z20.s, z12.s, z1.s[3]\n"
                        "fmla z24.s, z12.s, z2.s[3]\n"
                        "fmla z17.s, z13.s, z0.s[3]\n"
                        "fmla z21.s, z13.s, z1.s[3]\n"
                        "fmla z25.s, z13.s, z2.s[3]\n"
                        "fmla z18.s, z14.s, z0.s[3]\n"
                        "fmla z22.s, z14.s, z1.s[3]\n"
                        "fmla z26.s, z14.s, z2.s[3]\n"
                        "fmla z19.s, z15.s, z0.s[3]\n"
                        "fmla z23.s, z15.s, z1.s[3]\n"
                        "fmla z27.s, z15.s, z2.s[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.s, z8.s, z5.s[0]\n"
                        "fmla z24.s, z8.s, z6.s[0]\n"
                        "fmla z17.s, z9.s, z4.s[0]\n"
                        "fmla z21.s, z9.s, z5.s[0]\n"
                        "fmla z25.s, z9.s, z6.s[0]\n"
                        "fmla z18.s, z10.s, z4.s[0]\n"
                        "fmla z22.s, z10.s, z5.s[0]\n"
                        "fmla z26.s, z10.s, z6.s[0]\n"
                        "fmla z19.s, z11.s, z4.s[0]\n"
                        "fmla z23.s, z11.s, z5.s[0]\n"
                        "fmla z27.s, z11.s, z6.s[0]\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.s, z12.s, z5.s[1]\n"
                        "fmla z24.s, z12.s, z6.s[1]\n"
                        "fmla z17.s, z13.s, z4.s[1]\n"
                        "fmla z21.s, z13.s, z5.s[1]\n"
                        "fmla z25.s, z13.s, z6.s[1]\n"
                        "fmla z18.s, z14.s, z4.s[1]\n"
                        "fmla z22.s, z14.s, z5.s[1]\n"
                        "fmla z26.s, z14.s, z6.s[1]\n"
                        "fmla z19.s, z15.s, z4.s[1]\n"
                        "fmla z23.s, z15.s, z5.s[1]\n"
                        "fmla z27.s, z15.s, z6.s[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[2]\n"
                        "fmla z20.s, z8.s, z5.s[2]\n"
                        "fmla z24.s, z8.s, z6.s[2]\n"
                        "fmla z17.s, z9.s, z4.s[2]\n"
                        "fmla z21.s, z9.s, z5.s[2]\n"
                        "fmla z25.s, z9.s, z6.s[2]\n"
                        "fmla z18.s, z10.s, z4.s[2]\n"
                        "fmla z22.s, z10.s, z5.s[2]\n"
                        "fmla z26.s, z10.s, z6.s[2]\n"
                        "fmla z19.s, z11.s, z4.s[2]\n"
                        "fmla z23.s, z11.s, z5.s[2]\n"
                        "fmla z27.s, z11.s, z6.s[2]\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "st1w z16.s, p0, [%[c_ptr0]]\n"
                        "fmax z20.s, p7/m, z20.s, z14.s\n"
                        "fmax z21.s, p7/m, z21.s, z14.s\n"
                        "fmax z22.s, p7/m, z22.s, z14.s\n"
                        "st1w z17.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmax z23.s, p7/m, z23.s, z14.s\n"
                        "fmin z20.s, p7/m, z20.s, z15.s\n"
                        "fmin z21.s, p7/m, z21.s, z15.s\n"
                        "st1w z18.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "fmin z22.s, p7/m, z22.s, z15.s\n"
                        "fmin z23.s, p7/m, z23.s, z15.s\n"
                        "fmax z24.s, p7/m, z24.s, z14.s\n"
                        "st1w z19.s, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "fmax z25.s, p7/m, z25.s, z14.s\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        "fmax z26.s, p7/m, z26.s, z14.s\n"
                        "st1w z20.s, p0, [c_ptr1]\n"
                        "fmin z24.s, p7/m, z24.s, z15.s\n"
                        "fmin z25.s, p7/m, z25.s, z15.s\n"
                        "fmax z27.s, p7/m, z27.s, z14.s\n"
                        "st1w z21.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "fmin z26.s, p7/m, z26.s, z15.s\n"
                        "fmin z27.s, p7/m, z27.s, z15.s\n"
                        "st1w z22.s, p2, [c_ptr1, #2, MUL VL]\n"
                        "st1w z23.s, p3, [c_ptr1, #3, MUL VL]\n"
                        "st1w z24.s, p0, [c_ptr2]\n"
                        "st1w z25.s, p1, [c_ptr2, #1, MUL VL]\n"
                        "st1w z26.s, p2, [c_ptr2, #2, MUL VL]\n"
                        "st1w z27.s, p3, [c_ptr2, #3, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "cc", "memory"
                    );
                    break;
                default:
                case 4:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "a_ptr3 .req X2\n"
                        "c_ptr1 .req X3\n"
                        "c_ptr2 .req X4\n"
                        "c_ptr3 .req X5\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "add a_ptr3, a_ptr2, %[lda]\n"
                        "add c_ptr3, c_ptr2, %[ldc]\n"
                        "whilelt p6.s, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.s\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p3.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "ld1w z16.s, p0/z, [%[biasptr]]\n"
                        "ld1w z17.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[biasptr], #3, MUL VL]\n"
                        "mov z20.d, z16.d\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "mov z21.d, z17.d\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "mov z22.d, z18.d\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "mov z23.d, z19.d\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                        "mov z24.d, z16.d\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "mov z25.d, z17.d\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "mov z26.d, z18.d\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "mov z27.d, z19.d\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z28.d, z16.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "mov z29.d, z17.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "mov z30.d, z18.d\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "mov z31.d, z19.d\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z16.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1w z20.s, p0/z, [c_ptr1]\n"
                        "ld1w z21.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1w z22.s, p2/z, [c_ptr1, #2, MUL VL]\n"
                        "ld1w z23.s, p3/z, [c_ptr1, #3, MUL VL]\n"
                        "ld1w z24.s, p0/z, [c_ptr2]\n"
                        "ld1w z25.s, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "ld1w z26.s, p2/z, [c_ptr2, #2, MUL VL]\n"
                        "ld1w z27.s, p3/z, [c_ptr2, #3, MUL VL]\n"
                        "ld1w z28.s, p0/z, [c_ptr3]\n"
                        "ld1w z29.s, p1/z, [c_ptr3, #1, MUL VL]\n"
                        "ld1w z30.s, p2/z, [c_ptr3, #2, MUL VL]\n"
                        "ld1w z31.s, p3/z, [c_ptr3, #3, MUL VL]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.s, z8.s, z1.s[0]\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "fmla z24.s, z8.s, z2.s[0]\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        "fmla z28.s, z8.s, z3.s[0]\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "ld1rqw z7.s, p7/z, [a_ptr3]\n"
                        "fmla z21.s, z9.s, z1.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z25.s, z9.s, z2.s[0]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "fmla z29.s, z9.s, z3.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        "fmla z22.s, z10.s, z1.s[0]\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        "fmla z26.s, z10.s, z2.s[0]\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        "fmla z30.s, z10.s, z3.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "add a_ptr3, a_ptr3, #0x20\n"
                        "fmla z23.s, z11.s, z1.s[0]\n"
                        "fmla z27.s, z11.s, z2.s[0]\n"
                        "fmla z31.s, z11.s, z3.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "fmla z20.s, z12.s, z1.s[1]\n"
                        "fmla z24.s, z12.s, z2.s[1]\n"
                        "fmla z28.s, z12.s, z3.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "fmla z21.s, z13.s, z1.s[1]\n"
                        "fmla z25.s, z13.s, z2.s[1]\n"
                        "fmla z29.s, z13.s, z3.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "fmla z22.s, z14.s, z1.s[1]\n"
                        "fmla z26.s, z14.s, z2.s[1]\n"
                        "fmla z30.s, z14.s, z3.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "fmla z23.s, z15.s, z1.s[1]\n"
                        "fmla z27.s, z15.s, z2.s[1]\n"
                        "fmla z31.s, z15.s, z3.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.s, z8.s, z1.s[2]\n"
                        "fmla z24.s, z8.s, z2.s[2]\n"
                        "fmla z28.s, z8.s, z3.s[2]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "fmla z21.s, z9.s, z1.s[2]\n"
                        "fmla z25.s, z9.s, z2.s[2]\n"
                        "fmla z29.s, z9.s, z3.s[2]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "fmla z22.s, z10.s, z1.s[2]\n"
                        "fmla z26.s, z10.s, z2.s[2]\n"
                        "fmla z30.s, z10.s, z3.s[2]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "fmla z23.s, z11.s, z1.s[2]\n"
                        "fmla z27.s, z11.s, z2.s[2]\n"
                        "fmla z31.s, z11.s, z3.s[2]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[3]\n"
                        "fmla z20.s, z12.s, z1.s[3]\n"
                        "fmla z24.s, z12.s, z2.s[3]\n"
                        "fmla z28.s, z12.s, z3.s[3]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[3]\n"
                        "fmla z21.s, z13.s, z1.s[3]\n"
                        "fmla z25.s, z13.s, z2.s[3]\n"
                        "fmla z29.s, z13.s, z3.s[3]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[3]\n"
                        "fmla z22.s, z14.s, z1.s[3]\n"
                        "fmla z26.s, z14.s, z2.s[3]\n"
                        "fmla z30.s, z14.s, z3.s[3]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[3]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0], #-0x10]\n"
                        "fmla z23.s, z15.s, z1.s[3]\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1, #-0x10]\n"
                        "fmla z27.s, z15.s, z2.s[3]\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2, #-0x10]\n"
                        "fmla z31.s, z15.s, z3.s[3]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[0]\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3, #-0x10]\n"
                        "fmla z20.s, z8.s, z5.s[0]\n"
                        "fmla z24.s, z8.s, z6.s[0]\n"
                        "fmla z28.s, z8.s, z7.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.s, z9.s, z4.s[0]\n"
                        "fmla z21.s, z9.s, z5.s[0]\n"
                        "fmla z25.s, z9.s, z6.s[0]\n"
                        "fmla z29.s, z9.s, z7.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.s, z10.s, z4.s[0]\n"
                        "fmla z22.s, z10.s, z5.s[0]\n"
                        "fmla z26.s, z10.s, z6.s[0]\n"
                        "fmla z30.s, z10.s, z7.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z11.s, z4.s[0]\n"
                        "fmla z23.s, z11.s, z5.s[0]\n"
                        "fmla z27.s, z11.s, z6.s[0]\n"
                        "fmla z31.s, z11.s, z7.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[1]\n"
                        "fmla z20.s, z12.s, z5.s[1]\n"
                        "fmla z24.s, z12.s, z6.s[1]\n"
                        "fmla z28.s, z12.s, z7.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z13.s, z4.s[1]\n"
                        "fmla z21.s, z13.s, z5.s[1]\n"
                        "fmla z25.s, z13.s, z6.s[1]\n"
                        "fmla z29.s, z13.s, z7.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z14.s, z4.s[1]\n"
                        "fmla z22.s, z14.s, z5.s[1]\n"
                        "fmla z26.s, z14.s, z6.s[1]\n"
                        "fmla z30.s, z14.s, z7.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z15.s, z4.s[1]\n"
                        "fmla z23.s, z15.s, z5.s[1]\n"
                        "fmla z27.s, z15.s, z6.s[1]\n"
                        "fmla z31.s, z15.s, z7.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.s, z8.s, z5.s[2]\n"
                        "fmla z24.s, z8.s, z6.s[2]\n"
                        "fmla z28.s, z8.s, z7.s[2]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z17.s, z9.s, z4.s[2]\n"
                        "fmla z21.s, z9.s, z5.s[2]\n"
                        "fmla z25.s, z9.s, z6.s[2]\n"
                        "fmla z29.s, z9.s, z7.s[2]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.s, z10.s, z4.s[2]\n"
                        "fmla z22.s, z10.s, z5.s[2]\n"
                        "fmla z26.s, z10.s, z6.s[2]\n"
                        "fmla z30.s, z10.s, z7.s[2]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.s, z11.s, z4.s[2]\n"
                        "fmla z23.s, z11.s, z5.s[2]\n"
                        "fmla z27.s, z11.s, z6.s[2]\n"
                        "fmla z31.s, z11.s, z7.s[2]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[3]\n"
                        "fmla z20.s, z12.s, z5.s[3]\n"
                        "fmla z24.s, z12.s, z6.s[3]\n"
                        "fmla z28.s, z12.s, z7.s[3]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.s, z13.s, z4.s[3]\n"
                        "fmla z21.s, z13.s, z5.s[3]\n"
                        "fmla z25.s, z13.s, z6.s[3]\n"
                        "fmla z29.s, z13.s, z7.s[3]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.s, z14.s, z4.s[3]\n"
                        "fmla z22.s, z14.s, z5.s[3]\n"
                        "fmla z26.s, z14.s, z6.s[3]\n"
                        "fmla z30.s, z14.s, z7.s[3]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.s, z15.s, z4.s[3]\n"
                        "fmla z23.s, z15.s, z5.s[3]\n"
                        "fmla z27.s, z15.s, z6.s[3]\n"
                        "fmla z31.s, z15.s, z7.s[3]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.s, z8.s, z1.s[0]\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "fmla z24.s, z8.s, z2.s[0]\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        "fmla z28.s, z8.s, z3.s[0]\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "ld1rqw z7.s, p7/z, [a_ptr3]\n"
                        "fmla z21.s, z9.s, z1.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z25.s, z9.s, z2.s[0]\n"
                        "fmla z29.s, z9.s, z3.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "fmla z22.s, z10.s, z1.s[0]\n"
                        "fmla z26.s, z10.s, z2.s[0]\n"
                        "fmla z30.s, z10.s, z3.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "fmla z23.s, z11.s, z1.s[0]\n"
                        "fmla z27.s, z11.s, z2.s[0]\n"
                        "fmla z31.s, z11.s, z3.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "fmla z20.s, z12.s, z1.s[1]\n"
                        "fmla z24.s, z12.s, z2.s[1]\n"
                        "fmla z28.s, z12.s, z3.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "fmla z21.s, z13.s, z1.s[1]\n"
                        "fmla z25.s, z13.s, z2.s[1]\n"
                        "fmla z29.s, z13.s, z3.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "fmla z22.s, z14.s, z1.s[1]\n"
                        "fmla z26.s, z14.s, z2.s[1]\n"
                        "fmla z30.s, z14.s, z3.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "fmla z23.s, z15.s, z1.s[1]\n"
                        "fmla z27.s, z15.s, z2.s[1]\n"
                        "fmla z31.s, z15.s, z3.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.s, z8.s, z1.s[2]\n"
                        "fmla z24.s, z8.s, z2.s[2]\n"
                        "fmla z28.s, z8.s, z3.s[2]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "fmla z21.s, z9.s, z1.s[2]\n"
                        "fmla z25.s, z9.s, z2.s[2]\n"
                        "fmla z29.s, z9.s, z3.s[2]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "fmla z22.s, z10.s, z1.s[2]\n"
                        "fmla z26.s, z10.s, z2.s[2]\n"
                        "fmla z30.s, z10.s, z3.s[2]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "fmla z23.s, z11.s, z1.s[2]\n"
                        "fmla z27.s, z11.s, z2.s[2]\n"
                        "fmla z31.s, z11.s, z3.s[2]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[3]\n"
                        "fmla z20.s, z12.s, z1.s[3]\n"
                        "fmla z24.s, z12.s, z2.s[3]\n"
                        "fmla z28.s, z12.s, z3.s[3]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[3]\n"
                        "fmla z21.s, z13.s, z1.s[3]\n"
                        "fmla z25.s, z13.s, z2.s[3]\n"
                        "fmla z29.s, z13.s, z3.s[3]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[3]\n"
                        "fmla z22.s, z14.s, z1.s[3]\n"
                        "fmla z26.s, z14.s, z2.s[3]\n"
                        "fmla z30.s, z14.s, z3.s[3]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[3]\n"
                        "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                        "fmla z23.s, z15.s, z1.s[3]\n"
                        "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                        "fmla z27.s, z15.s, z2.s[3]\n"
                        "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                        "fmla z31.s, z15.s, z3.s[3]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[0]\n"
                        "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                        "fmla z20.s, z8.s, z5.s[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        "fmla z24.s, z8.s, z6.s[0]\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        "fmla z28.s, z8.s, z7.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.s, z9.s, z4.s[0]\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        "fmla z21.s, z9.s, z5.s[0]\n"
                        "addvl a_ptr3, a_ptr3, #2\n"
                        "fmla z25.s, z9.s, z6.s[0]\n"
                        "fmla z29.s, z9.s, z7.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.s, z10.s, z4.s[0]\n"
                        "fmla z22.s, z10.s, z5.s[0]\n"
                        "fmla z26.s, z10.s, z6.s[0]\n"
                        "fmla z30.s, z10.s, z7.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z11.s, z4.s[0]\n"
                        "fmla z23.s, z11.s, z5.s[0]\n"
                        "fmla z27.s, z11.s, z6.s[0]\n"
                        "fmla z31.s, z11.s, z7.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[1]\n"
                        "fmla z20.s, z12.s, z5.s[1]\n"
                        "fmla z24.s, z12.s, z6.s[1]\n"
                        "fmla z28.s, z12.s, z7.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z13.s, z4.s[1]\n"
                        "fmla z21.s, z13.s, z5.s[1]\n"
                        "fmla z25.s, z13.s, z6.s[1]\n"
                        "fmla z29.s, z13.s, z7.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z14.s, z4.s[1]\n"
                        "fmla z22.s, z14.s, z5.s[1]\n"
                        "fmla z26.s, z14.s, z6.s[1]\n"
                        "fmla z30.s, z14.s, z7.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z15.s, z4.s[1]\n"
                        "fmla z23.s, z15.s, z5.s[1]\n"
                        "fmla z27.s, z15.s, z6.s[1]\n"
                        "fmla z31.s, z15.s, z7.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z20.s, z8.s, z5.s[2]\n"
                        "fmla z24.s, z8.s, z6.s[2]\n"
                        "fmla z28.s, z8.s, z7.s[2]\n"
                        "fmla z17.s, z9.s, z4.s[2]\n"
                        "fmla z21.s, z9.s, z5.s[2]\n"
                        "fmla z25.s, z9.s, z6.s[2]\n"
                        "fmla z29.s, z9.s, z7.s[2]\n"
                        "fmla z18.s, z10.s, z4.s[2]\n"
                        "fmla z22.s, z10.s, z5.s[2]\n"
                        "fmla z26.s, z10.s, z6.s[2]\n"
                        "fmla z30.s, z10.s, z7.s[2]\n"
                        "fmla z19.s, z11.s, z4.s[2]\n"
                        "fmla z23.s, z11.s, z5.s[2]\n"
                        "fmla z27.s, z11.s, z6.s[2]\n"
                        "fmla z31.s, z11.s, z7.s[2]\n"
                        "fmla z16.s, z12.s, z4.s[3]\n"
                        "fmla z20.s, z12.s, z5.s[3]\n"
                        "fmla z24.s, z12.s, z6.s[3]\n"
                        "fmla z28.s, z12.s, z7.s[3]\n"
                        "fmla z17.s, z13.s, z4.s[3]\n"
                        "fmla z21.s, z13.s, z5.s[3]\n"
                        "fmla z25.s, z13.s, z6.s[3]\n"
                        "fmla z29.s, z13.s, z7.s[3]\n"
                        "fmla z18.s, z14.s, z4.s[3]\n"
                        "fmla z22.s, z14.s, z5.s[3]\n"
                        "fmla z26.s, z14.s, z6.s[3]\n"
                        "fmla z30.s, z14.s, z7.s[3]\n"
                        "fmla z19.s, z15.s, z4.s[3]\n"
                        "fmla z23.s, z15.s, z5.s[3]\n"
                        "fmla z27.s, z15.s, z6.s[3]\n"
                        "fmla z31.s, z15.s, z7.s[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.s, z8.s, z1.s[0]\n"
                        "fmla z24.s, z8.s, z2.s[0]\n"
                        "fmla z28.s, z8.s, z3.s[0]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "fmla z21.s, z9.s, z1.s[0]\n"
                        "fmla z25.s, z9.s, z2.s[0]\n"
                        "fmla z29.s, z9.s, z3.s[0]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "fmla z22.s, z10.s, z1.s[0]\n"
                        "fmla z26.s, z10.s, z2.s[0]\n"
                        "fmla z30.s, z10.s, z3.s[0]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "fmla z23.s, z11.s, z1.s[0]\n"
                        "fmla z27.s, z11.s, z2.s[0]\n"
                        "fmla z31.s, z11.s, z3.s[0]\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.s, z12.s, z1.s[1]\n"
                        "fmla z24.s, z12.s, z2.s[1]\n"
                        "fmla z28.s, z12.s, z3.s[1]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "fmla z21.s, z13.s, z1.s[1]\n"
                        "fmla z25.s, z13.s, z2.s[1]\n"
                        "fmla z29.s, z13.s, z3.s[1]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "fmla z22.s, z14.s, z1.s[1]\n"
                        "fmla z26.s, z14.s, z2.s[1]\n"
                        "fmla z30.s, z14.s, z3.s[1]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "fmla z23.s, z15.s, z1.s[1]\n"
                        "fmla z27.s, z15.s, z2.s[1]\n"
                        "fmla z31.s, z15.s, z3.s[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "fmla z20.s, z8.s, z1.s[2]\n"
                        "fmla z24.s, z8.s, z2.s[2]\n"
                        "fmla z28.s, z8.s, z3.s[2]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "fmla z21.s, z9.s, z1.s[2]\n"
                        "fmla z25.s, z9.s, z2.s[2]\n"
                        "fmla z29.s, z9.s, z3.s[2]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "fmla z22.s, z10.s, z1.s[2]\n"
                        "fmla z26.s, z10.s, z2.s[2]\n"
                        "fmla z30.s, z10.s, z3.s[2]\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "fmla z23.s, z11.s, z1.s[2]\n"
                        "fmla z27.s, z11.s, z2.s[2]\n"
                        "fmla z31.s, z11.s, z3.s[2]\n"
                        "b 5f\n"
                        "4:\n"
                        "fmla z16.s, z8.s, z0.s[0]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.s, z8.s, z1.s[0]\n"
                        "ld1rqw z4.s, p6/z, [%[a_ptr0]]\n"
                        "fmla z24.s, z8.s, z2.s[0]\n"
                        "ld1rqw z5.s, p6/z, [a_ptr1]\n"
                        "fmla z28.s, z8.s, z3.s[0]\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.s, z9.s, z0.s[0]\n"
                        "ld1rqw z6.s, p6/z, [a_ptr2]\n"
                        "fmla z21.s, z9.s, z1.s[0]\n"
                        "ld1rqw z7.s, p6/z, [a_ptr3]\n"
                        "fmla z25.s, z9.s, z2.s[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        "fmla z29.s, z9.s, z3.s[0]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.s, z10.s, z0.s[0]\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        "fmla z22.s, z10.s, z1.s[0]\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        "fmla z26.s, z10.s, z2.s[0]\n"
                        "addvl a_ptr3, a_ptr3, #1\n"
                        "fmla z30.s, z10.s, z3.s[0]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.s, z11.s, z0.s[0]\n"
                        "fmla z23.s, z11.s, z1.s[0]\n"
                        "fmla z27.s, z11.s, z2.s[0]\n"
                        "fmla z31.s, z11.s, z3.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.s, z12.s, z0.s[1]\n"
                        "fmla z20.s, z12.s, z1.s[1]\n"
                        "fmla z24.s, z12.s, z2.s[1]\n"
                        "fmla z28.s, z12.s, z3.s[1]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.s, z13.s, z0.s[1]\n"
                        "fmla z21.s, z13.s, z1.s[1]\n"
                        "fmla z25.s, z13.s, z2.s[1]\n"
                        "fmla z29.s, z13.s, z3.s[1]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.s, z14.s, z0.s[1]\n"
                        "fmla z22.s, z14.s, z1.s[1]\n"
                        "fmla z26.s, z14.s, z2.s[1]\n"
                        "fmla z30.s, z14.s, z3.s[1]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.s, z15.s, z0.s[1]\n"
                        "fmla z23.s, z15.s, z1.s[1]\n"
                        "fmla z27.s, z15.s, z2.s[1]\n"
                        "fmla z31.s, z15.s, z3.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.s, z8.s, z0.s[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z20.s, z8.s, z1.s[2]\n"
                        "fmla z24.s, z8.s, z2.s[2]\n"
                        "fmla z28.s, z8.s, z3.s[2]\n"
                        "fmla z17.s, z9.s, z0.s[2]\n"
                        "fmla z21.s, z9.s, z1.s[2]\n"
                        "fmla z25.s, z9.s, z2.s[2]\n"
                        "fmla z29.s, z9.s, z3.s[2]\n"
                        "fmla z18.s, z10.s, z0.s[2]\n"
                        "fmla z22.s, z10.s, z1.s[2]\n"
                        "fmla z26.s, z10.s, z2.s[2]\n"
                        "fmla z30.s, z10.s, z3.s[2]\n"
                        "fmla z19.s, z11.s, z0.s[2]\n"
                        "fmla z23.s, z11.s, z1.s[2]\n"
                        "fmla z27.s, z11.s, z2.s[2]\n"
                        "fmla z31.s, z11.s, z3.s[2]\n"
                        "fmla z16.s, z12.s, z0.s[3]\n"
                        "fmla z20.s, z12.s, z1.s[3]\n"
                        "fmla z24.s, z12.s, z2.s[3]\n"
                        "fmla z28.s, z12.s, z3.s[3]\n"
                        "fmla z17.s, z13.s, z0.s[3]\n"
                        "fmla z21.s, z13.s, z1.s[3]\n"
                        "fmla z25.s, z13.s, z2.s[3]\n"
                        "fmla z29.s, z13.s, z3.s[3]\n"
                        "fmla z18.s, z14.s, z0.s[3]\n"
                        "fmla z22.s, z14.s, z1.s[3]\n"
                        "fmla z26.s, z14.s, z2.s[3]\n"
                        "fmla z30.s, z14.s, z3.s[3]\n"
                        "fmla z19.s, z15.s, z0.s[3]\n"
                        "fmla z23.s, z15.s, z1.s[3]\n"
                        "fmla z27.s, z15.s, z2.s[3]\n"
                        "fmla z31.s, z15.s, z3.s[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[0]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.s, z8.s, z5.s[0]\n"
                        "fmla z24.s, z8.s, z6.s[0]\n"
                        "fmla z28.s, z8.s, z7.s[0]\n"
                        "fmla z17.s, z9.s, z4.s[0]\n"
                        "fmla z21.s, z9.s, z5.s[0]\n"
                        "fmla z25.s, z9.s, z6.s[0]\n"
                        "fmla z29.s, z9.s, z7.s[0]\n"
                        "fmla z18.s, z10.s, z4.s[0]\n"
                        "fmla z22.s, z10.s, z5.s[0]\n"
                        "fmla z26.s, z10.s, z6.s[0]\n"
                        "fmla z30.s, z10.s, z7.s[0]\n"
                        "fmla z19.s, z11.s, z4.s[0]\n"
                        "fmla z23.s, z11.s, z5.s[0]\n"
                        "fmla z27.s, z11.s, z6.s[0]\n"
                        "fmla z31.s, z11.s, z7.s[0]\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.s, z12.s, z4.s[1]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.s, z12.s, z5.s[1]\n"
                        "fmla z24.s, z12.s, z6.s[1]\n"
                        "fmla z28.s, z12.s, z7.s[1]\n"
                        "fmla z17.s, z13.s, z4.s[1]\n"
                        "fmla z21.s, z13.s, z5.s[1]\n"
                        "fmla z25.s, z13.s, z6.s[1]\n"
                        "fmla z29.s, z13.s, z7.s[1]\n"
                        "fmla z18.s, z14.s, z4.s[1]\n"
                        "fmla z22.s, z14.s, z5.s[1]\n"
                        "fmla z26.s, z14.s, z6.s[1]\n"
                        "fmla z30.s, z14.s, z7.s[1]\n"
                        "fmla z19.s, z15.s, z4.s[1]\n"
                        "fmla z23.s, z15.s, z5.s[1]\n"
                        "fmla z27.s, z15.s, z6.s[1]\n"
                        "fmla z31.s, z15.s, z7.s[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1w z8.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1w z9.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1w z10.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1w z11.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.s, z8.s, z4.s[2]\n"
                        "fmla z20.s, z8.s, z5.s[2]\n"
                        "fmla z24.s, z8.s, z6.s[2]\n"
                        "fmla z28.s, z8.s, z7.s[2]\n"
                        "fmla z17.s, z9.s, z4.s[2]\n"
                        "fmla z21.s, z9.s, z5.s[2]\n"
                        "fmla z25.s, z9.s, z6.s[2]\n"
                        "fmla z29.s, z9.s, z7.s[2]\n"
                        "fmla z18.s, z10.s, z4.s[2]\n"
                        "fmla z22.s, z10.s, z5.s[2]\n"
                        "fmla z26.s, z10.s, z6.s[2]\n"
                        "fmla z30.s, z10.s, z7.s[2]\n"
                        "fmla z19.s, z11.s, z4.s[2]\n"
                        "fmla z23.s, z11.s, z5.s[2]\n"
                        "fmla z27.s, z11.s, z6.s[2]\n"
                        "fmla z31.s, z11.s, z7.s[2]\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "st1w z16.s, p0, [%[c_ptr0]]\n"
                        "fmax z20.s, p7/m, z20.s, z14.s\n"
                        "fmax z21.s, p7/m, z21.s, z14.s\n"
                        "fmax z22.s, p7/m, z22.s, z14.s\n"
                        "st1w z17.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmax z23.s, p7/m, z23.s, z14.s\n"
                        "fmin z20.s, p7/m, z20.s, z15.s\n"
                        "fmin z21.s, p7/m, z21.s, z15.s\n"
                        "st1w z18.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "fmin z22.s, p7/m, z22.s, z15.s\n"
                        "fmin z23.s, p7/m, z23.s, z15.s\n"
                        "fmax z24.s, p7/m, z24.s, z14.s\n"
                        "st1w z19.s, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "fmax z25.s, p7/m, z25.s, z14.s\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        "fmax z26.s, p7/m, z26.s, z14.s\n"
                        "st1w z20.s, p0, [c_ptr1]\n"
                        "fmin z24.s, p7/m, z24.s, z15.s\n"
                        "fmin z25.s, p7/m, z25.s, z15.s\n"
                        "fmax z27.s, p7/m, z27.s, z14.s\n"
                        "st1w z21.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "fmin z26.s, p7/m, z26.s, z15.s\n"
                        "fmax z28.s, p7/m, z28.s, z14.s\n"
                        "fmax z29.s, p7/m, z29.s, z14.s\n"
                        "st1w z22.s, p2, [c_ptr1, #2, MUL VL]\n"
                        "fmin z27.s, p7/m, z27.s, z15.s\n"
                        "fmax z30.s, p7/m, z30.s, z14.s\n"
                        "fmin z28.s, p7/m, z28.s, z15.s\n"
                        "st1w z23.s, p3, [c_ptr1, #3, MUL VL]\n"
                        "fmin z29.s, p7/m, z29.s, z15.s\n"
                        "fmax z31.s, p7/m, z31.s, z14.s\n"
                        "fmin z30.s, p7/m, z30.s, z15.s\n"
                        "st1w z24.s, p0, [c_ptr2]\n"
                        "fmin z31.s, p7/m, z31.s, z15.s\n"
                        "st1w z25.s, p1, [c_ptr2, #1, MUL VL]\n"
                        "st1w z26.s, p2, [c_ptr2, #2, MUL VL]\n"
                        "st1w z27.s, p3, [c_ptr2, #3, MUL VL]\n"
                        "st1w z28.s, p0, [c_ptr3]\n"
                        "st1w z29.s, p1, [c_ptr3, #1, MUL VL]\n"
                        "st1w z30.s, p2, [c_ptr3, #2, MUL VL]\n"
                        "st1w z31.s, p3, [c_ptr3, #3, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq a_ptr3\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        ".unreq c_ptr3\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "x4", "x5", "cc", "memory"
                    );
                    break;
            }

        }
    }
}

} // namespace arm_gemm

#endif // __ARM_FEATURE_SVE
