# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Unit tests for the server-side no-operation task."""

from django.core.exceptions import ValidationError

from debusine.artifacts.models import WorkRequestResults
from debusine.test.django import TestCase


class ServerNoopTests(TestCase):
    """Test the ServerNoop task."""

    def test_execute_result_true(self) -> None:
        """If result=SUCCESS, the task returns SUCCESS."""
        noop = self.playground.create_server_task(
            task_name="servernoop",
            task_data={"result": WorkRequestResults.SUCCESS},
        ).get_task()
        self.assertEqual(noop._execute(), WorkRequestResults.SUCCESS)

    def test_execute_result_false(self) -> None:
        """If result=FAILURE, the task returns FAILURE."""
        noop = self.playground.create_server_task(
            task_name="servernoop",
            task_data={"result": WorkRequestResults.FAILURE},
        ).get_task()
        self.assertEqual(noop._execute(), WorkRequestResults.FAILURE)

    def test_execute_exception(self) -> None:
        """If exception=True, the task raises an exception."""
        noop = self.playground.create_server_task(
            task_name="servernoop", task_data={"exception": True}
        ).get_task()
        self.assertRaisesRegex(
            RuntimeError, "Client requested an exception", noop._execute
        )

    def test_no_additional_properties(self) -> None:
        """Assert no additional properties."""
        error_msg = "Extra inputs are not permitted"
        with self.assertRaisesRegex(ValidationError, error_msg):
            self.playground.create_server_task(
                task_name="servernoop", task_data={"input": {}}
            )

    def test_label(self) -> None:
        """Test get_label."""
        task = self.playground.create_server_task(
            task_name="servernoop",
            task_data={"result": WorkRequestResults.SUCCESS},
        ).get_task()
        self.assertEqual(task.get_label(), "noop")
