// SPDX-License-Identifier: GPL-2.0
/*
 * Sophgo SG2042 SoC pinctrl driver.
 *
 * Copyright (C) 2024 Inochi Amaoto <inochiama@outlook.com>
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/of.h>

#include <linux/pinctrl/pinctrl.h>
#include <linux/pinctrl/pinmux.h>

#include <dt-bindings/pinctrl/pinctrl-sg2042.h>

#include "pinctrl-sg2042.h"

static int sg2042_get_pull_up(const struct sophgo_pin *sp, const u32 *psmap)
{
	return 35000;
}

static int sg2042_get_pull_down(const struct sophgo_pin *sp, const u32 *psmap)
{
	return 28000;
}

static const u32 sg2042_oc_map[] = {
	5400,  8100,  10700, 13400,
	16100, 18800, 21400, 24100,
	26800, 29400, 32100, 34800,
	37400, 40100, 42800, 45400
};

static int sg2042_get_oc_map(const struct sophgo_pin *sp, const u32 *psmap,
			     const u32 **map)
{
	*map = sg2042_oc_map;
	return ARRAY_SIZE(sg2042_oc_map);
}

static const struct sophgo_vddio_cfg_ops sg2042_vddio_cfg_ops = {
	.get_pull_up		= sg2042_get_pull_up,
	.get_pull_down		= sg2042_get_pull_down,
	.get_oc_map		= sg2042_get_oc_map,
};

static const struct pinctrl_pin_desc sg2042_pins[] = {
	PINCTRL_PIN(PIN_LPC_LCLK,		"lpc_lclk"),
	PINCTRL_PIN(PIN_LPC_LFRAME,		"lpc_lframe"),
	PINCTRL_PIN(PIN_LPC_LAD0,		"lpc_lad0"),
	PINCTRL_PIN(PIN_LPC_LAD1,		"lpc_lad1"),
	PINCTRL_PIN(PIN_LPC_LAD2,		"lpc_lad2"),
	PINCTRL_PIN(PIN_LPC_LAD3,		"lpc_lad3"),
	PINCTRL_PIN(PIN_LPC_LDRQ0,		"lpc_ldrq0"),
	PINCTRL_PIN(PIN_LPC_LDRQ1,		"lpc_ldrq1"),
	PINCTRL_PIN(PIN_LPC_SERIRQ,		"lpc_serirq"),
	PINCTRL_PIN(PIN_LPC_CLKRUN,		"lpc_clkrun"),
	PINCTRL_PIN(PIN_LPC_LPME,		"lpc_lpme"),
	PINCTRL_PIN(PIN_LPC_LPCPD,		"lpc_lpcpd"),
	PINCTRL_PIN(PIN_LPC_LSMI,		"lpc_lsmi"),
	PINCTRL_PIN(PIN_PCIE0_L0_RESET,		"pcie0_l0_reset"),
	PINCTRL_PIN(PIN_PCIE0_L1_RESET,		"pcie0_l1_reset"),
	PINCTRL_PIN(PIN_PCIE0_L0_WAKEUP,	"pcie0_l0_wakeup"),
	PINCTRL_PIN(PIN_PCIE0_L1_WAKEUP,	"pcie0_l1_wakeup"),
	PINCTRL_PIN(PIN_PCIE0_L0_CLKREQ_IN,	"pcie0_l0_clkreq_in"),
	PINCTRL_PIN(PIN_PCIE0_L1_CLKREQ_IN,	"pcie0_l1_clkreq_in"),
	PINCTRL_PIN(PIN_PCIE1_L0_RESET,		"pcie1_l0_reset"),
	PINCTRL_PIN(PIN_PCIE1_L1_RESET,		"pcie1_l1_reset"),
	PINCTRL_PIN(PIN_PCIE1_L0_WAKEUP,	"pcie1_l0_wakeup"),
	PINCTRL_PIN(PIN_PCIE1_L1_WAKEUP,	"pcie1_l1_wakeup"),
	PINCTRL_PIN(PIN_PCIE1_L0_CLKREQ_IN,	"pcie1_l0_clkreq_in"),
	PINCTRL_PIN(PIN_PCIE1_L1_CLKREQ_IN,	"pcie1_l1_clkreq_in"),
	PINCTRL_PIN(PIN_SPIF0_CLK_SEL1,		"spif0_clk_sel1"),
	PINCTRL_PIN(PIN_SPIF0_CLK_SEL0,		"spif0_clk_sel0"),
	PINCTRL_PIN(PIN_SPIF0_WP,		"spif0_wp"),
	PINCTRL_PIN(PIN_SPIF0_HOLD,		"spif0_hold"),
	PINCTRL_PIN(PIN_SPIF0_SDI,		"spif0_sdi"),
	PINCTRL_PIN(PIN_SPIF0_CS,		"spif0_cs"),
	PINCTRL_PIN(PIN_SPIF0_SCK,		"spif0_sck"),
	PINCTRL_PIN(PIN_SPIF0_SDO,		"spif0_sdo"),
	PINCTRL_PIN(PIN_SPIF1_CLK_SEL1,		"spif1_clk_sel1"),
	PINCTRL_PIN(PIN_SPIF1_CLK_SEL0,		"spif1_clk_sel0"),
	PINCTRL_PIN(PIN_SPIF1_WP,		"spif1_wp"),
	PINCTRL_PIN(PIN_SPIF1_HOLD,		"spif1_hold"),
	PINCTRL_PIN(PIN_SPIF1_SDI,		"spif1_sdi"),
	PINCTRL_PIN(PIN_SPIF1_CS,		"spif1_cs"),
	PINCTRL_PIN(PIN_SPIF1_SCK,		"spif1_sck"),
	PINCTRL_PIN(PIN_SPIF1_SDO,		"spif1_sdo"),
	PINCTRL_PIN(PIN_EMMC_WP,		"emmc_wp"),
	PINCTRL_PIN(PIN_EMMC_CD,		"emmc_cd"),
	PINCTRL_PIN(PIN_EMMC_RST,		"emmc_rst"),
	PINCTRL_PIN(PIN_EMMC_PWR_EN,		"emmc_pwr_en"),
	PINCTRL_PIN(PIN_SDIO_CD,		"sdio_cd"),
	PINCTRL_PIN(PIN_SDIO_WP,		"sdio_wp"),
	PINCTRL_PIN(PIN_SDIO_RST,		"sdio_rst"),
	PINCTRL_PIN(PIN_SDIO_PWR_EN,		"sdio_pwr_en"),
	PINCTRL_PIN(PIN_RGMII0_TXD0,		"rgmii0_txd0"),
	PINCTRL_PIN(PIN_RGMII0_TXD1,		"rgmii0_txd1"),
	PINCTRL_PIN(PIN_RGMII0_TXD2,		"rgmii0_txd2"),
	PINCTRL_PIN(PIN_RGMII0_TXD3,		"rgmii0_txd3"),
	PINCTRL_PIN(PIN_RGMII0_TXCTRL,		"rgmii0_txctrl"),
	PINCTRL_PIN(PIN_RGMII0_RXD0,		"rgmii0_rxd0"),
	PINCTRL_PIN(PIN_RGMII0_RXD1,		"rgmii0_rxd1"),
	PINCTRL_PIN(PIN_RGMII0_RXD2,		"rgmii0_rxd2"),
	PINCTRL_PIN(PIN_RGMII0_RXD3,		"rgmii0_rxd3"),
	PINCTRL_PIN(PIN_RGMII0_RXCTRL,		"rgmii0_rxctrl"),
	PINCTRL_PIN(PIN_RGMII0_TXC,		"rgmii0_txc"),
	PINCTRL_PIN(PIN_RGMII0_RXC,		"rgmii0_rxc"),
	PINCTRL_PIN(PIN_RGMII0_REFCLKO,		"rgmii0_refclko"),
	PINCTRL_PIN(PIN_RGMII0_IRQ,		"rgmii0_irq"),
	PINCTRL_PIN(PIN_RGMII0_MDC,		"rgmii0_mdc"),
	PINCTRL_PIN(PIN_RGMII0_MDIO,		"rgmii0_mdio"),
	PINCTRL_PIN(PIN_PWM0,			"pwm0"),
	PINCTRL_PIN(PIN_PWM1,			"pwm1"),
	PINCTRL_PIN(PIN_PWM2,			"pwm2"),
	PINCTRL_PIN(PIN_PWM3,			"pwm3"),
	PINCTRL_PIN(PIN_FAN0,			"fan0"),
	PINCTRL_PIN(PIN_FAN1,			"fan1"),
	PINCTRL_PIN(PIN_FAN2,			"fan2"),
	PINCTRL_PIN(PIN_FAN3,			"fan3"),
	PINCTRL_PIN(PIN_IIC0_SDA,		"iic0_sda"),
	PINCTRL_PIN(PIN_IIC0_SCL,		"iic0_scl"),
	PINCTRL_PIN(PIN_IIC1_SDA,		"iic1_sda"),
	PINCTRL_PIN(PIN_IIC1_SCL,		"iic1_scl"),
	PINCTRL_PIN(PIN_IIC2_SDA,		"iic2_sda"),
	PINCTRL_PIN(PIN_IIC2_SCL,		"iic2_scl"),
	PINCTRL_PIN(PIN_IIC3_SDA,		"iic3_sda"),
	PINCTRL_PIN(PIN_IIC3_SCL,		"iic3_scl"),
	PINCTRL_PIN(PIN_UART0_TX,		"uart0_tx"),
	PINCTRL_PIN(PIN_UART0_RX,		"uart0_rx"),
	PINCTRL_PIN(PIN_UART0_RTS,		"uart0_rts"),
	PINCTRL_PIN(PIN_UART0_CTS,		"uart0_cts"),
	PINCTRL_PIN(PIN_UART1_TX,		"uart1_tx"),
	PINCTRL_PIN(PIN_UART1_RX,		"uart1_rx"),
	PINCTRL_PIN(PIN_UART1_RTS,		"uart1_rts"),
	PINCTRL_PIN(PIN_UART1_CTS,		"uart1_cts"),
	PINCTRL_PIN(PIN_UART2_TX,		"uart2_tx"),
	PINCTRL_PIN(PIN_UART2_RX,		"uart2_rx"),
	PINCTRL_PIN(PIN_UART2_RTS,		"uart2_rts"),
	PINCTRL_PIN(PIN_UART2_CTS,		"uart2_cts"),
	PINCTRL_PIN(PIN_UART3_TX,		"uart3_tx"),
	PINCTRL_PIN(PIN_UART3_RX,		"uart3_rx"),
	PINCTRL_PIN(PIN_UART3_RTS,		"uart3_rts"),
	PINCTRL_PIN(PIN_UART3_CTS,		"uart3_cts"),
	PINCTRL_PIN(PIN_SPI0_CS0,		"spi0_cs0"),
	PINCTRL_PIN(PIN_SPI0_CS1,		"spi0_cs1"),
	PINCTRL_PIN(PIN_SPI0_SDI,		"spi0_sdi"),
	PINCTRL_PIN(PIN_SPI0_SDO,		"spi0_sdo"),
	PINCTRL_PIN(PIN_SPI0_SCK,		"spi0_sck"),
	PINCTRL_PIN(PIN_SPI1_CS0,		"spi1_cs0"),
	PINCTRL_PIN(PIN_SPI1_CS1,		"spi1_cs1"),
	PINCTRL_PIN(PIN_SPI1_SDI,		"spi1_sdi"),
	PINCTRL_PIN(PIN_SPI1_SDO,		"spi1_sdo"),
	PINCTRL_PIN(PIN_SPI1_SCK,		"spi1_sck"),
	PINCTRL_PIN(PIN_JTAG0_TDO,		"jtag0_tdo"),
	PINCTRL_PIN(PIN_JTAG0_TCK,		"jtag0_tck"),
	PINCTRL_PIN(PIN_JTAG0_TDI,		"jtag0_tdi"),
	PINCTRL_PIN(PIN_JTAG0_TMS,		"jtag0_tms"),
	PINCTRL_PIN(PIN_JTAG0_TRST,		"jtag0_trst"),
	PINCTRL_PIN(PIN_JTAG0_SRST,		"jtag0_srst"),
	PINCTRL_PIN(PIN_JTAG1_TDO,		"jtag1_tdo"),
	PINCTRL_PIN(PIN_JTAG1_TCK,		"jtag1_tck"),
	PINCTRL_PIN(PIN_JTAG1_TDI,		"jtag1_tdi"),
	PINCTRL_PIN(PIN_JTAG1_TMS,		"jtag1_tms"),
	PINCTRL_PIN(PIN_JTAG1_TRST,		"jtag1_trst"),
	PINCTRL_PIN(PIN_JTAG1_SRST,		"jtag1_srst"),
	PINCTRL_PIN(PIN_JTAG2_TDO,		"jtag2_tdo"),
	PINCTRL_PIN(PIN_JTAG2_TCK,		"jtag2_tck"),
	PINCTRL_PIN(PIN_JTAG2_TDI,		"jtag2_tdi"),
	PINCTRL_PIN(PIN_JTAG2_TMS,		"jtag2_tms"),
	PINCTRL_PIN(PIN_JTAG2_TRST,		"jtag2_trst"),
	PINCTRL_PIN(PIN_JTAG2_SRST,		"jtag2_srst"),
	PINCTRL_PIN(PIN_GPIO0,			"gpio0"),
	PINCTRL_PIN(PIN_GPIO1,			"gpio1"),
	PINCTRL_PIN(PIN_GPIO2,			"gpio2"),
	PINCTRL_PIN(PIN_GPIO3,			"gpio3"),
	PINCTRL_PIN(PIN_GPIO4,			"gpio4"),
	PINCTRL_PIN(PIN_GPIO5,			"gpio5"),
	PINCTRL_PIN(PIN_GPIO6,			"gpio6"),
	PINCTRL_PIN(PIN_GPIO7,			"gpio7"),
	PINCTRL_PIN(PIN_GPIO8,			"gpio8"),
	PINCTRL_PIN(PIN_GPIO9,			"gpio9"),
	PINCTRL_PIN(PIN_GPIO10,			"gpio10"),
	PINCTRL_PIN(PIN_GPIO11,			"gpio11"),
	PINCTRL_PIN(PIN_GPIO12,			"gpio12"),
	PINCTRL_PIN(PIN_GPIO13,			"gpio13"),
	PINCTRL_PIN(PIN_GPIO14,			"gpio14"),
	PINCTRL_PIN(PIN_GPIO15,			"gpio15"),
	PINCTRL_PIN(PIN_GPIO16,			"gpio16"),
	PINCTRL_PIN(PIN_GPIO17,			"gpio17"),
	PINCTRL_PIN(PIN_GPIO18,			"gpio18"),
	PINCTRL_PIN(PIN_GPIO19,			"gpio19"),
	PINCTRL_PIN(PIN_GPIO20,			"gpio20"),
	PINCTRL_PIN(PIN_GPIO21,			"gpio21"),
	PINCTRL_PIN(PIN_GPIO22,			"gpio22"),
	PINCTRL_PIN(PIN_GPIO23,			"gpio23"),
	PINCTRL_PIN(PIN_GPIO24,			"gpio24"),
	PINCTRL_PIN(PIN_GPIO25,			"gpio25"),
	PINCTRL_PIN(PIN_GPIO26,			"gpio26"),
	PINCTRL_PIN(PIN_GPIO27,			"gpio27"),
	PINCTRL_PIN(PIN_GPIO28,			"gpio28"),
	PINCTRL_PIN(PIN_GPIO29,			"gpio29"),
	PINCTRL_PIN(PIN_GPIO30,			"gpio30"),
	PINCTRL_PIN(PIN_GPIO31,			"gpio31"),
	PINCTRL_PIN(PIN_MODE_SEL0,		"mode_sel0"),
	PINCTRL_PIN(PIN_MODE_SEL1,		"mode_sel1"),
	PINCTRL_PIN(PIN_MODE_SEL2,		"mode_sel2"),
	PINCTRL_PIN(PIN_BOOT_SEL0,		"boot_sel0"),
	PINCTRL_PIN(PIN_BOOT_SEL1,		"boot_sel1"),
	PINCTRL_PIN(PIN_BOOT_SEL2,		"boot_sel2"),
	PINCTRL_PIN(PIN_BOOT_SEL3,		"boot_sel3"),
	PINCTRL_PIN(PIN_BOOT_SEL4,		"boot_sel4"),
	PINCTRL_PIN(PIN_BOOT_SEL5,		"boot_sel5"),
	PINCTRL_PIN(PIN_BOOT_SEL6,		"boot_sel6"),
	PINCTRL_PIN(PIN_BOOT_SEL7,		"boot_sel7"),
	PINCTRL_PIN(PIN_MULTI_SCKT,		"multi_sckt"),
	PINCTRL_PIN(PIN_SCKT_ID0,		"sckt_id0"),
	PINCTRL_PIN(PIN_SCKT_ID1,		"sckt_id1"),
	PINCTRL_PIN(PIN_PLL_CLK_IN_MAIN,	"pll_clk_in_main"),
	PINCTRL_PIN(PIN_PLL_CLK_IN_DDR_L,	"pll_clk_in_ddr_l"),
	PINCTRL_PIN(PIN_PLL_CLK_IN_DDR_R,	"pll_clk_in_ddr_r"),
	PINCTRL_PIN(PIN_XTAL_32K,		"xtal_32k"),
	PINCTRL_PIN(PIN_SYS_RST,		"sys_rst"),
	PINCTRL_PIN(PIN_PWR_BUTTON,		"pwr_button"),
	PINCTRL_PIN(PIN_TEST_EN,		"test_en"),
	PINCTRL_PIN(PIN_TEST_MODE_MBIST,	"test_mode_mbist"),
	PINCTRL_PIN(PIN_TEST_MODE_SCAN,		"test_mode_scan"),
	PINCTRL_PIN(PIN_TEST_MODE_BSD,		"test_mode_bsd"),
	PINCTRL_PIN(PIN_BISR_BYP,		"bisr_byp"),
};

static const struct sg2042_pin sg2042_pin_data[ARRAY_SIZE(sg2042_pins)] = {
	SG2042_GENERAL_PIN(PIN_LPC_LCLK, 0x000,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_LPC_LFRAME, 0x000,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_LPC_LAD0, 0x004,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_LPC_LAD1, 0x004,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_LPC_LAD2, 0x008,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_LPC_LAD3, 0x008,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_LPC_LDRQ0, 0x00c,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_LPC_LDRQ1, 0x00c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_LPC_SERIRQ, 0x010,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_LPC_CLKRUN, 0x010,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_LPC_LPME, 0x014,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_LPC_LPCPD, 0x014,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_LPC_LSMI, 0x018,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_PCIE0_L0_RESET, 0x018,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_PCIE0_L1_RESET, 0x01c,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_PCIE0_L0_WAKEUP, 0x01c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_PCIE0_L1_WAKEUP, 0x020,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_PCIE0_L0_CLKREQ_IN, 0x020,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_PCIE0_L1_CLKREQ_IN, 0x024,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_PCIE1_L0_RESET, 0x024,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_PCIE1_L1_RESET, 0x028,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_PCIE1_L0_WAKEUP, 0x028,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_PCIE1_L1_WAKEUP, 0x02c,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_PCIE1_L0_CLKREQ_IN, 0x02c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_PCIE1_L1_CLKREQ_IN, 0x030,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF0_CLK_SEL1, 0x030,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF0_CLK_SEL0, 0x034,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF0_WP, 0x034,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF0_HOLD, 0x038,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF0_SDI, 0x038,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF0_CS, 0x03c,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF0_SCK, 0x03c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF0_SDO, 0x040,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF1_CLK_SEL1, 0x040,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF1_CLK_SEL0, 0x044,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF1_WP, 0x044,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF1_HOLD, 0x048,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF1_SDI, 0x048,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF1_CS, 0x04c,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF1_SCK, 0x04c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPIF1_SDO, 0x050,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_EMMC_WP, 0x050,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_EMMC_CD, 0x054,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_EMMC_RST, 0x054,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_EMMC_PWR_EN, 0x058,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SDIO_CD, 0x058,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SDIO_WP, 0x05c,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SDIO_RST, 0x05c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SDIO_PWR_EN, 0x060,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_RGMII0_TXD0, 0x060,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_TXD1, 0x064,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_RGMII0_TXD2, 0x064,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_TXD3, 0x068,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_RGMII0_TXCTRL, 0x068,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_RXD0, 0x06c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_RGMII0_RXD1, 0x06c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_RXD2, 0x070,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_RGMII0_RXD3, 0x070,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_RXCTRL, 0x074,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_RGMII0_TXC, 0x074,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_RXC, 0x078,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_RGMII0_REFCLKO, 0x078,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_IRQ, 0x07c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_RGMII0_MDC, 0x07c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_MDIO, 0x080,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PWM0, 0x080,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_PWM1, 0x084,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_PWM2, 0x084,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_PWM3, 0x088,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_FAN0, 0x088,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_FAN1, 0x08c,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_FAN2, 0x08c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_FAN3, 0x090,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_IIC0_SDA, 0x090,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_IIC0_SCL, 0x094,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_IIC1_SDA, 0x094,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_IIC1_SCL, 0x098,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_IIC2_SDA, 0x098,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_IIC2_SCL, 0x09c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_IIC3_SDA, 0x09c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_IIC3_SCL, 0x0a0,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_UART0_TX, 0x0a0,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_UART0_RX, 0x0a4,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_UART0_RTS, 0x0a4,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_UART0_CTS, 0x0a8,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_UART1_TX, 0x0a8,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_UART1_RX, 0x0ac,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_UART1_RTS, 0x0ac,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_UART1_CTS, 0x0b0,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_UART2_TX, 0x0b0,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_UART2_RX, 0x0b4,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_UART2_RTS, 0x0b4,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_UART2_CTS, 0x0b8,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_UART3_TX, 0x0b8,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_UART3_RX, 0x0bc,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_UART3_RTS, 0x0bc,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_UART3_CTS, 0x0c0,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPI0_CS0, 0x0c0,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPI0_CS1, 0x0c4,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPI0_SDI, 0x0c4,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPI0_SDO, 0x0c8,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPI0_SCK, 0x0c8,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPI1_CS0, 0x0cc,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPI1_CS1, 0x0cc,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPI1_SDI, 0x0d0,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPI1_SDO, 0x0d0,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_SPI1_SCK, 0x0d4,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG0_TDO, 0x0d4,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG0_TCK, 0x0d8,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG0_TDI, 0x0d8,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG0_TMS, 0x0dc,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG0_TRST, 0x0dc,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG0_SRST, 0x0e0,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG1_TDO, 0x0e0,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG1_TCK, 0x0e4,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG1_TDI, 0x0e4,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG1_TMS, 0x0e8,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG1_TRST, 0x0e8,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG1_SRST, 0x0ec,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG2_TDO, 0x0ec,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG2_TCK, 0x0f0,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG2_TDI, 0x0f0,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG2_TMS, 0x0f4,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG2_TRST, 0x0f4,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_JTAG2_SRST, 0x0f8,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO0, 0x0f8,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO1, 0x0fc,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO2, 0x0fc,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO3, 0x100,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO4, 0x100,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO5, 0x104,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO6, 0x104,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO7, 0x108,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO8, 0x108,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO9, 0x10c,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO10, 0x10c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO11, 0x110,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO12, 0x110,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO13, 0x114,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO14, 0x114,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO15, 0x118,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO16, 0x118,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO17, 0x11c,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO18, 0x11c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO19, 0x120,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO20, 0x120,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO21, 0x124,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO22, 0x124,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO23, 0x128,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO24, 0x128,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO25, 0x12c,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO26, 0x12c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO27, 0x130,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO28, 0x130,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO29, 0x134,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO30, 0x134,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_GPIO31, 0x138,
			   PIN_FLAG_ONLY_ONE_PULL),
	SG2042_GENERAL_PIN(PIN_MODE_SEL0, 0x138,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL |
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_MODE_SEL1, 0x13c,
			   PIN_FLAG_ONLY_ONE_PULL | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_MODE_SEL2, 0x13c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL |
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL0, 0x140,
			   PIN_FLAG_ONLY_ONE_PULL | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL1, 0x140,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL |
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL2, 0x144,
			   PIN_FLAG_ONLY_ONE_PULL | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL3, 0x144,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL |
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL4, 0x148,
			   PIN_FLAG_ONLY_ONE_PULL | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL5, 0x148,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL |
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL6, 0x14c,
			   PIN_FLAG_ONLY_ONE_PULL | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL7, 0x14c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL |
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_MULTI_SCKT, 0x150,
			   PIN_FLAG_ONLY_ONE_PULL | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_SCKT_ID0, 0x150,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL |
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_SCKT_ID1, 0x154,
			   PIN_FLAG_ONLY_ONE_PULL | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_PLL_CLK_IN_MAIN, 0x154,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL |
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_PLL_CLK_IN_DDR_L, 0x158,
			   PIN_FLAG_ONLY_ONE_PULL | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_PLL_CLK_IN_DDR_R, 0x158,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL |
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_XTAL_32K, 0x15c,
			   PIN_FLAG_ONLY_ONE_PULL | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_SYS_RST, 0x15c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL |
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_PWR_BUTTON, 0x160,
			   PIN_FLAG_ONLY_ONE_PULL | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_TEST_EN, 0x160,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL |
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_TEST_MODE_MBIST, 0x164,
			   PIN_FLAG_ONLY_ONE_PULL | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_TEST_MODE_SCAN, 0x164,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL |
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_TEST_MODE_BSD, 0x168,
			   PIN_FLAG_ONLY_ONE_PULL | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BISR_BYP, 0x168,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_ONLY_ONE_PULL |
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
};

static const struct sophgo_pinctrl_data sg2042_pindata = {
	.pins		= sg2042_pins,
	.pindata	= sg2042_pin_data,
	.vddio_ops	= &sg2042_vddio_cfg_ops,
	.cfg_ops	= &sg2042_cfg_ops,
	.pctl_ops	= &sg2042_pctrl_ops,
	.pmx_ops	= &sg2042_pmx_ops,
	.pconf_ops	= &sg2042_pconf_ops,
	.npins		= ARRAY_SIZE(sg2042_pins),
	.pinsize	= sizeof(struct sg2042_pin),
};

static const struct of_device_id sg2042_pinctrl_ids[] = {
	{ .compatible = "sophgo,sg2042-pinctrl", .data = &sg2042_pindata },
	{ }
};
MODULE_DEVICE_TABLE(of, sg2042_pinctrl_ids);

static struct platform_driver sg2042_pinctrl_driver = {
	.probe	= sophgo_pinctrl_probe,
	.driver	= {
		.name			= "sg2042-pinctrl",
		.suppress_bind_attrs	= true,
		.of_match_table		= sg2042_pinctrl_ids,
	},
};
module_platform_driver(sg2042_pinctrl_driver);

MODULE_DESCRIPTION("Pinctrl driver for the SG2002 series SoC");
MODULE_LICENSE("GPL");
