/*
  This file is part of TALER
  Copyright (C) 2023-2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file lib/exchange_api_reveal_withdraw.c
 * @brief Implementation of /reveal-withdraw requests
 * @author Özgür Kesim
 */

#include "taler/platform.h"
#include <gnunet/gnunet_common.h>
#include <jansson.h>
#include <microhttpd.h> /* just for HTTP status codes */
#include <gnunet/gnunet_util_lib.h>
#include <gnunet/gnunet_json_lib.h>
#include <gnunet/gnunet_curl_lib.h>
#include "taler/taler_curl_lib.h"
#include "taler/taler_json_lib.h"
#include "taler/taler_exchange_service.h"
#include "exchange_api_common.h"
#include "exchange_api_handle.h"
#include "taler/taler_signatures.h"
#include "exchange_api_curl_defaults.h"

/**
 * Handler for a running reveal-withdraw  request
 */
struct TALER_EXCHANGE_RevealWithdrawHandle
{
  /**
   * The commitment from the previous call withdraw
   */
  const struct TALER_HashBlindedPlanchetsP *planchets_h;

  /**
   * Number of coins for which to reveal tuples of seeds
   */
  size_t num_coins;

  /**
   * The TALER_CNC_KAPPA-1 tuple of seeds to reveal
   */
  struct TALER_RevealWithdrawMasterSeedsP seeds;

  /**
   * The url for the reveal request
   */
  char *request_url;

  /**
   * CURL handle for the request job.
   */
  struct GNUNET_CURL_Job *job;

  /**
   * Post Context
   */
  struct TALER_CURL_PostContext post_ctx;

  /**
   * Callback
   */
  TALER_EXCHANGE_RevealWithdrawCallback callback;

  /**
   * Reveal
   */
  void *callback_cls;
};


/**
 * We got a 200 OK response for the /reveal-withdraw operation.
 * Extract the signed blindedcoins and return it to the caller.
 *
 * @param wrh operation handle
 * @param j_response reply from the exchange
 * @return #GNUNET_OK on success, #GNUNET_SYSERR on errors
 */
static enum GNUNET_GenericReturnValue
reveal_withdraw_ok (
  struct TALER_EXCHANGE_RevealWithdrawHandle *wrh,
  const json_t *j_response)
{
  struct TALER_EXCHANGE_RevealWithdrawResponse response = {
    .hr.reply = j_response,
    .hr.http_status = MHD_HTTP_OK,
  };
  const json_t *j_sigs;
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_array_const ("ev_sigs",
                                  &j_sigs),
    GNUNET_JSON_spec_end ()
  };

  if (GNUNET_OK !=
      GNUNET_JSON_parse (j_response,
                         spec,
                         NULL, NULL))
  {
    GNUNET_break_op (0);
    return GNUNET_SYSERR;
  }

  if (wrh->num_coins != json_array_size (j_sigs))
  {
    /* Number of coins generated does not match our expectation */
    GNUNET_break_op (0);
    return GNUNET_SYSERR;
  }

  {
    struct TALER_BlindedDenominationSignature denom_sigs[wrh->num_coins];
    json_t *j_sig;
    size_t n;

    /* Reconstruct the coins and unblind the signatures */
    json_array_foreach (j_sigs, n, j_sig)
    {
      struct GNUNET_JSON_Specification ispec[] = {
        TALER_JSON_spec_blinded_denom_sig (NULL,
                                           &denom_sigs[n]),
        GNUNET_JSON_spec_end ()
      };

      if (GNUNET_OK !=
          GNUNET_JSON_parse (j_sig,
                             ispec,
                             NULL, NULL))
      {
        GNUNET_break_op (0);
        return GNUNET_SYSERR;
      }
    }

    response.details.ok.num_sigs = wrh->num_coins;
    response.details.ok.blinded_denom_sigs = denom_sigs;
    wrh->callback (wrh->callback_cls,
                   &response);
    /* Make sure the callback isn't called again */
    wrh->callback = NULL;
    /* Free resources */
    for (size_t i = 0; i < wrh->num_coins; i++)
      TALER_blinded_denom_sig_free (&denom_sigs[i]);
  }

  return GNUNET_OK;
}


/**
 * Function called when we're done processing the
 * HTTP /reveal-withdraw request.
 *
 * @param cls the `struct TALER_EXCHANGE_RevealWithdrawHandle`
 * @param response_code The HTTP response code
 * @param response response data
 */
static void
handle_reveal_withdraw_finished (
  void *cls,
  long response_code,
  const void *response)
{
  struct TALER_EXCHANGE_RevealWithdrawHandle *wrh = cls;
  const json_t *j_response = response;
  struct TALER_EXCHANGE_RevealWithdrawResponse awr = {
    .hr.reply = j_response,
    .hr.http_status = (unsigned int) response_code
  };

  wrh->job = NULL;
  switch (response_code)
  {
  case 0:
    awr.hr.ec = TALER_EC_GENERIC_INVALID_RESPONSE;
    break;
  case MHD_HTTP_OK:
    {
      enum GNUNET_GenericReturnValue ret;

      ret = reveal_withdraw_ok (wrh,
                                j_response);
      if (GNUNET_OK != ret)
      {
        GNUNET_break_op (0);
        awr.hr.http_status = 0;
        awr.hr.ec = TALER_EC_GENERIC_REPLY_MALFORMED;
        break;
      }
      GNUNET_assert (NULL == wrh->callback);
      TALER_EXCHANGE_reveal_withdraw_cancel (wrh);
      return;
    }
  case MHD_HTTP_BAD_REQUEST:
    /* This should never happen, either us or the exchange is buggy
       (or API version conflict); just pass JSON reply to the application */
    awr.hr.ec = TALER_JSON_get_error_code (j_response);
    awr.hr.hint = TALER_JSON_get_error_hint (j_response);
    break;
  case MHD_HTTP_NOT_FOUND:
    /* Nothing really to verify, the exchange basically just says
       that it doesn't know this age-withdraw commitment. */
    awr.hr.ec = TALER_JSON_get_error_code (j_response);
    awr.hr.hint = TALER_JSON_get_error_hint (j_response);
    break;
  case MHD_HTTP_CONFLICT:
    /* An age commitment for one of the coins did not fulfill
     * the required maximum age requirement of the corresponding
     * reserve.
     * Error code: TALER_EC_EXCHANGE_GENERIC_COIN_AGE_REQUIREMENT_FAILURE
     * or TALER_EC_EXCHANGE_AGE_WITHDRAW_REVEAL_INVALID_HASH.
     */
    awr.hr.ec = TALER_JSON_get_error_code (j_response);
    awr.hr.hint = TALER_JSON_get_error_hint (j_response);
    break;
  case MHD_HTTP_INTERNAL_SERVER_ERROR:
    /* Server had an internal issue; we should retry, but this API
       leaves this to the application */
    awr.hr.ec = TALER_JSON_get_error_code (j_response);
    awr.hr.hint = TALER_JSON_get_error_hint (j_response);
    break;
  default:
    /* unexpected response code */
    GNUNET_break_op (0);
    awr.hr.ec = TALER_JSON_get_error_code (j_response);
    awr.hr.hint = TALER_JSON_get_error_hint (j_response);
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u/%d for exchange age-withdraw\n",
                (unsigned int) response_code,
                (int) awr.hr.ec);
    break;
  }
  wrh->callback (wrh->callback_cls,
                 &awr);
  TALER_EXCHANGE_reveal_withdraw_cancel (wrh);
}


/**
 * Call /reveal-withdraw
 *
 * @param curl_ctx The context for CURL
 * @param wrh The handler
 */
static void
perform_protocol (
  struct GNUNET_CURL_Context *curl_ctx,
  struct TALER_EXCHANGE_RevealWithdrawHandle *wrh)
{
  CURL *curlh;
  json_t *j_array_of_secrets;

  j_array_of_secrets = json_array ();
  GNUNET_assert (NULL != j_array_of_secrets);

  for (uint8_t k = 0; k < TALER_CNC_KAPPA - 1; k++)
  {
    json_t *j_sec = GNUNET_JSON_from_data_auto (&wrh->seeds.tuple[k]);
    GNUNET_assert (NULL != j_sec);
    GNUNET_assert (0 == json_array_append_new (j_array_of_secrets,
                                               j_sec));
  }
  {
    json_t *j_request_body;

    j_request_body = GNUNET_JSON_PACK (
      GNUNET_JSON_pack_data_auto ("planchets_h",
                                  wrh->planchets_h),
      GNUNET_JSON_pack_array_steal ("disclosed_batch_seeds",
                                    j_array_of_secrets));
    GNUNET_assert (NULL != j_request_body);

    curlh = TALER_EXCHANGE_curl_easy_get_ (wrh->request_url);
    GNUNET_assert (NULL != curlh);
    GNUNET_assert (GNUNET_OK ==
                   TALER_curl_easy_post (&wrh->post_ctx,
                                         curlh,
                                         j_request_body));

    json_decref (j_request_body);
  }

  wrh->job = GNUNET_CURL_job_add2 (
    curl_ctx,
    curlh,
    wrh->post_ctx.headers,
    &handle_reveal_withdraw_finished,
    wrh);
  if (NULL == wrh->job)
  {
    GNUNET_break (0);
    if (NULL != curlh)
      curl_easy_cleanup (curlh);
    TALER_EXCHANGE_reveal_withdraw_cancel (wrh);
  }

  return;
}


struct TALER_EXCHANGE_RevealWithdrawHandle *
TALER_EXCHANGE_reveal_withdraw (
  struct GNUNET_CURL_Context *curl_ctx,
  const char *exchange_url,
  size_t num_coins,
  const struct TALER_HashBlindedPlanchetsP *planchets_h,
  const struct TALER_RevealWithdrawMasterSeedsP *seeds,
  TALER_EXCHANGE_RevealWithdrawCallback reveal_cb,
  void *reveal_cb_cls)
{
  struct TALER_EXCHANGE_RevealWithdrawHandle *wrh =
    GNUNET_new (struct TALER_EXCHANGE_RevealWithdrawHandle);
  wrh->planchets_h = planchets_h;
  wrh->num_coins = num_coins;
  wrh->seeds = *seeds;
  wrh->callback = reveal_cb;
  wrh->callback_cls = reveal_cb_cls;
  wrh->request_url = TALER_url_join (exchange_url,
                                     "reveal-withdraw",
                                     NULL);
  if (NULL == wrh->request_url)
  {
    GNUNET_break (0);
    GNUNET_free (wrh);
    return NULL;
  }

  perform_protocol (curl_ctx, wrh);

  return wrh;
}


void
TALER_EXCHANGE_reveal_withdraw_cancel (
  struct TALER_EXCHANGE_RevealWithdrawHandle *wrh)
{
  if (NULL != wrh->job)
  {
    GNUNET_CURL_job_cancel (wrh->job);
    wrh->job = NULL;
  }
  TALER_curl_easy_post_finished (&wrh->post_ctx);

  if (NULL != wrh->request_url)
    GNUNET_free (wrh->request_url);

  GNUNET_free (wrh);
}


/* exchange_api_reveal_withdraw.c */
