/*
  This file is part of TALER
  (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
  details.

  You should have received a copy of the GNU Affero General Public License
  along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file merchant/backend/taler-merchant-httpd_private-get-report-ID.c
 * @brief implementation of GET /private/reports/$REPORT_ID
 * @author Christian Grothoff
 */
#include "platform.h"
#include "taler-merchant-httpd_private-get-report-ID.h"
#include <taler/taler_json_lib.h>


MHD_RESULT
TMH_private_get_report (const struct TMH_RequestHandler *rh,
                        struct MHD_Connection *connection,
                        struct TMH_HandlerContext *hc)
{
  const char *report_id_str = hc->infix;
  unsigned long long report_id;
  char *report_program_section;
  char *report_description;
  char *mime_type;
  char *data_source;
  char *target_address;
  struct GNUNET_TIME_Relative frequency;
  struct GNUNET_TIME_Relative frequency_shift;
  struct GNUNET_TIME_Absolute next_transmission;
  enum TALER_ErrorCode last_error_code;
  char *last_error_detail;
  enum GNUNET_DB_QueryStatus qs;

  (void) rh;

  {
    char dummy;

    if (1 != sscanf (report_id_str,
                     "%llu%c",
                     &report_id,
                     &dummy))
    {
      GNUNET_break_op (0);
      return TALER_MHD_reply_with_error (connection,
                                         MHD_HTTP_BAD_REQUEST,
                                         TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                         "report_id");
    }
  }

  qs = TMH_db->select_report (TMH_db->cls,
                              hc->instance->settings.id,
                              (uint64_t) report_id,
                              &report_program_section,
                              &report_description,
                              &mime_type,
                              &data_source,
                              &target_address,
                              &frequency,
                              &frequency_shift,
                              &next_transmission,
                              &last_error_code,
                              &last_error_detail);

  if (qs < 0)
  {
    GNUNET_break (0);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_INTERNAL_SERVER_ERROR,
                                       TALER_EC_GENERIC_DB_FETCH_FAILED,
                                       "select_report");
  }
  if (GNUNET_DB_STATUS_SUCCESS_NO_RESULTS == qs)
  {
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_NOT_FOUND,
                                       TALER_EC_MERCHANT_GENERIC_REPORT_UNKNOWN,
                                       report_id_str);
  }

  {
    json_t *response;

    response = GNUNET_JSON_PACK (
      GNUNET_JSON_pack_uint64 ("report_serial",
                               report_id),
      GNUNET_JSON_pack_string ("description",
                               report_description),
      GNUNET_JSON_pack_string ("program_section",
                               report_program_section),
      GNUNET_JSON_pack_string ("mime_type",
                               mime_type),
      GNUNET_JSON_pack_string ("data_source",
                               data_source),
      GNUNET_JSON_pack_string ("target_address",
                               target_address),
      GNUNET_JSON_pack_allow_null (
        GNUNET_JSON_pack_string ("last_error_detail",
                                 last_error_detail)),
      GNUNET_JSON_pack_time_rel ("report_frequency",
                                 frequency),
      GNUNET_JSON_pack_time_rel ("report_frequency_shift",
                                 frequency_shift));
    GNUNET_free (report_program_section);
    GNUNET_free (report_description);
    GNUNET_free (mime_type);
    GNUNET_free (data_source);
    GNUNET_free (target_address);
    GNUNET_free (last_error_detail);
    if (TALER_EC_NONE != last_error_code)
    {
      GNUNET_assert (0 ==
                     json_object_set_new (response,
                                          "last_error_code",
                                          json_integer (last_error_code)));
    }
    return TALER_MHD_reply_json_steal (connection,
                                       response,
                                       MHD_HTTP_OK);
  }
}
