/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import * as tests from "@gnu-taler/web-util/testing";
import { expect } from "chai";
import {
  useInstanceProducts,
  useProductDetails,
} from "./product.js";
import { ApiMockEnvironment } from "./testing.js";
import {
  API_CREATE_PRODUCT,
  API_DELETE_PRODUCT,
  API_GET_PRODUCT_BY_ID,
  API_LIST_PRODUCTS,
  API_UPDATE_PRODUCT_BY_ID,
} from "./urls.js";
import { TalerMerchantApi } from "@gnu-taler/taler-util";
import { useMerchantApiContext } from "@gnu-taler/web-util/browser";

describe("product api interaction with listing", () => {
  it("should evict cache when creating a product", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_LIST_PRODUCTS, {
      response: {
        products: [{ product_id: "1234", product_serial: 1 }],
      },
    });
    env.addRequestExpectation(API_GET_PRODUCT_BY_ID("1234"), {
      response: { price: "ARS:12" } as TalerMerchantApi.ProductDetail,
    });

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const query = useInstanceProducts();
        const { lib: api } = useMerchantApiContext();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          // expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          // expect(query.loading).true;
        },
        ({ query, api }) => {
          // expect(query.loading).undefined;
          // expect(query.ok).true;
          // if (!query.ok) return;
          // expect(query.data).deep.equals([{ id: "1234", price: "ARS:12" }]);

          env.addRequestExpectation(API_CREATE_PRODUCT, {
            request: {
              price: "ARS:23",
            } as TalerMerchantApi.ProductAddDetail,
          });

          env.addRequestExpectation(API_LIST_PRODUCTS, {
            response: {
              products: [{ product_id: "1234", product_serial: 1 }, { product_id: "2345", product_serial: 2 }],
            },
          });
          env.addRequestExpectation(API_GET_PRODUCT_BY_ID("1234"), {
            response: {
              price: "ARS:12",
            } as TalerMerchantApi.ProductDetail,
          });
          env.addRequestExpectation(API_GET_PRODUCT_BY_ID("1234"), {
            response: {
              price: "ARS:12",
            } as TalerMerchantApi.ProductDetail,
          });
          env.addRequestExpectation(API_GET_PRODUCT_BY_ID("2345"), {
            response: {
              price: "ARS:23",
            } as TalerMerchantApi.ProductDetail,
          });

          api.instance.addProduct(undefined, {
            price: "ARS:23",
          } as any);
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          // expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          // expect(query.loading).undefined;
          // expect(query.ok).true;
          // if (!query.ok) return;
          // expect(query.data).deep.equals([
          //   {
          //     id: "1234",
          //     price: "ARS:12",
          //   },
          //   {
          //     id: "2345",
          //     price: "ARS:23",
          //   },
          // ]);
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });

  it("should evict cache when updating a product", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_LIST_PRODUCTS, {
      response: {
        products: [{ product_id: "1234", product_serial: 1 }],
      },
    });
    env.addRequestExpectation(API_GET_PRODUCT_BY_ID("1234"), {
      response: { price: "ARS:12" } as TalerMerchantApi.ProductDetail,
    });

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const query = useInstanceProducts();
        const { lib: api } = useMerchantApiContext();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          // expect(query.loading).true;
        },
        ({ query, api }) => {
          // expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          // expect(query.loading).undefined;
          // expect(query.ok).true;
          // if (!query.ok) return;
          // expect(query.data).deep.equals([{ id: "1234", price: "ARS:12" }]);

          env.addRequestExpectation(API_UPDATE_PRODUCT_BY_ID("1234"), {
            request: {
              price: "ARS:13",
            } as TalerMerchantApi.ProductPatchDetail,
          });

          env.addRequestExpectation(API_LIST_PRODUCTS, {
            response: {
              products: [{ product_id: "1234", product_serial: 1 }],
            },
          });
          env.addRequestExpectation(API_GET_PRODUCT_BY_ID("1234"), {
            response: {
              price: "ARS:13",
            } as TalerMerchantApi.ProductDetail,
          });

          api.instance.updateProduct(undefined, "1234", {
            price: "ARS:13",
          } as any);
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          // expect(query.loading).undefined;
          // expect(query.ok).true;
          // if (!query.ok) return;
          // expect(query.data).deep.equals([
          //   {
          //     id: "1234",
          //     price: "ARS:13",
          //   },
          // ]);
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });

  it("should evict cache when deleting a product", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_LIST_PRODUCTS, {
      response: {
        products: [{ product_id: "1234", product_serial: 1 }, { product_id: "2345", product_serial: 2 }],
      },
    });
    env.addRequestExpectation(API_GET_PRODUCT_BY_ID("1234"), {
      response: { price: "ARS:12" } as TalerMerchantApi.ProductDetail,
    });
    env.addRequestExpectation(API_GET_PRODUCT_BY_ID("2345"), {
      response: { price: "ARS:23" } as TalerMerchantApi.ProductDetail,
    });

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const query = useInstanceProducts();
        const { lib: api } = useMerchantApiContext();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          // expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          // expect(query.loading).true;
        },
        ({ query, api }) => {
          // expect(query.loading).undefined;
          // expect(query.ok).true;
          // if (!query.ok) return;
          // expect(query.data).deep.equals([
          //   { id: "1234", price: "ARS:12" },
          //   { id: "2345", price: "ARS:23" },
          // ]);

          env.addRequestExpectation(API_DELETE_PRODUCT("2345"), {});

          env.addRequestExpectation(API_LIST_PRODUCTS, {
            response: {
              products: [{ product_id: "1234", product_serial: 1 }],
            },
          });

          env.addRequestExpectation(API_GET_PRODUCT_BY_ID("1234"), {
            response: {
              price: "ARS:12",
            } as TalerMerchantApi.ProductDetail,
          });
          api.instance.deleteProduct(undefined, "2345");
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          // expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          // expect(query.loading).undefined;
          // expect(query.ok).true;
          // if (!query.ok) return;
          // expect(query.data).deep.equals([{ id: "1234", price: "ARS:12" }]);
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });
});

describe("product api interaction with details", () => {
  it("should evict cache when updating a product", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_GET_PRODUCT_BY_ID("12"), {
      response: {
        description: "this is a description",
      } as TalerMerchantApi.ProductDetail,
    });

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const query = useProductDetails("12");
        const { lib: api } = useMerchantApiContext();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          // expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          // expect(query.loading).false;
          // expect(query.ok).true;
          // if (!query.ok) return;
          // expect(query.data).deep.equals({
          //   description: "this is a description",
          // });

          env.addRequestExpectation(API_UPDATE_PRODUCT_BY_ID("12"), {
            request: {
              description: "other description",
            } as TalerMerchantApi.ProductPatchDetail,
          });

          env.addRequestExpectation(API_GET_PRODUCT_BY_ID("12"), {
            response: {
              description: "other description",
            } as TalerMerchantApi.ProductDetail,
          });

          api.instance.updateProduct(undefined, "12", {
            description: "other description",
          } as any);
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          // expect(query.loading).false;
          // expect(query.ok).true;
          // if (!query.ok) return;
          // expect(query.data).deep.equals({
          //   description: "other description",
          // });
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });
});
