// Qt objects
#include <qwidget.h>
#include <qobject.h>
#include <qobjectlist.h>
#include <qstring.h>
#include <qpoint.h>
#include <qfile.h>

// KDE includes
#include <klistview.h>
#include <klocale.h>
#include <keditcl.h>
#include <kstatusbar.h>
#include <kmessagebox.h>

// KMtraceViewer includes
#include "KMtraceModel.hh"
#include "KMtraceLeaksView.hh"
#include "KMtraceLeaksView.moc"
#include "KMtraceLeak.hh"
#include "KMtraceLeaksViewItem.hh"
#include "KMtraceViewerMainWindow.hh"
#include "KMtraceSuppressDialog.hh"

KMtraceLeaksView::KMtraceLeaksView( QWidget *parent, KMtraceModel *model )
  : KListView( parent )
{   
   setAllColumnsShowFocus( true );
   addColumn( i18n( "call stack" ) );
   addColumn( i18n( "bytes" ) );
   addColumn( i18n( "blocks" ) );
   addColumn( i18n( "1st. block bytes" ) );
   addColumn( i18n( "address" ) );

   // short '+' for leaks, sort by "bytes" descending per default
   // and short that and what we are sorting
   setRootIsDecorated( true );
   setSorting( 1, true );
   setShowSortIndicator( true );

   // fill the thing with life   
   setModel( model );

   // build the popup menu for leaks   
   m_menu = new KPopupMenu( this );
   m_menuItemIdSuppress = m_menu->insertItem( i18n( "&Suppress" ) );

   // the context menu should allow suppressions of the whole leak (call stack)
   connect( this, SIGNAL( contextMenuRequested( QListViewItem *, const QPoint& , int ) ),
            this, SLOT( slotContextLeak( QListViewItem *, const QPoint &, int ) ) );

   // changing the selected item causes the edit window with the
   // source code to change (if the current item is a call with
   // module only!)   
   connect( this, SIGNAL( selectionChanged( QListViewItem * ) ),
            this, SLOT( slotSelectionChanged( QListViewItem * ) ) );

   // double clicking on a item should expand or collapse depending on its
   // current state
   connect( this, SIGNAL( executed( QListViewItem * ) ),
            this, SLOT( slotExecuted( QListViewItem * ) ) );

   // reset values for the find dialog
   m_findDialog = 0;
   m_findIterator = 0;
   m_findItem = 0;
   m_posY = 0;

   // reset values for the suppress dialog
   m_suppressDialog = 0;

   // do not show suppressed leaks by default
   m_showSuppressed = false;
}

void KMtraceLeaksView::setModel( KMtraceModel *model )
{
   m_model = model;

   // remove all elements in the list
   clear( );

   // do nothing in the view if the model is empty
   if( !m_model ) return;

   QObjectList *leaksList = m_model->getLeaksList( );
   QObjectListIt leakIterator( *leaksList );
   QObject *obj;
   
   while( ( obj = leakIterator.current( ) ) != 0 ) {
      KMtraceLeak *leak = (KMtraceLeak *)obj;
      QString s1 = QString( "%1" ).arg( leak->getBytes( ) );
      QString s2 = QString( "%1" ).arg( leak->getBlocks( ) );
      QString s3 = QString( "%1" ).arg( leak->getFirstBytes( ) );
      QString result;
      if( leak->isSuppressed( ) ) {
         result.sprintf( i18n( "Suppressed Leak of %d bytes in %d blocks" ),
                         leak->getBytes( ), leak->getBlocks( ) );
      } else {
         result.sprintf( i18n( "Leak of %d bytes in %d blocks" ),
                         leak->getBytes( ), leak->getBlocks( ) );
      }
      KMtraceLeaksViewItem *item = new KMtraceLeaksViewItem( this, result, s1, s2, s3, leak->getFirstAddress( ) );
      item->setModel( leak );
      leak->setView( item );
      connect( leak, SIGNAL( suppressionChanged( KMtraceLeak * ) ),
               this, SLOT( slotSuppressionChanged( KMtraceLeak * ) ) );
      connect( leak, SIGNAL( suppressionChanged( KMtraceLeak * ) ),
               m_model, SLOT( slotSuppressionChanged( KMtraceLeak * ) ) );

      QObjectList *callStack = leak->getCallStack( );
      QObjectListIt callIterator( *callStack );
      QObject *obj2;
      int stackCounter=0;
      
      while( ( obj2 = callIterator.current( ) ) != 0 ) {
         KMtraceCall *call = (KMtraceCall *)obj2;
         QString result;
         result.sprintf( "#%02d %s %s%s%s(%s%s%s)", stackCounter, call->getAddress( ).latin1( ),
	                 call->getModule( ).latin1( ),
			 ( call->getLine( ) != 0 ) ? ":" : "",
			 ( call->getLine( ) != 0 ) ? QString( "%1" ).arg( call->getLine( ) ).latin1( ) : "",
			 call->getFunction( ).latin1( ),
			 ( call->getOffset( ).length( ) > 0 ) ? "+" : "",
			 ( call->getOffset( ).length( ) ) ? call->getOffset( ).latin1( ) : ""
		       );
         KMtraceLeaksViewItem *child = new KMtraceLeaksViewItem( item, result );
         child->setModel( call );
         ++callIterator;
         ++stackCounter;
      }
      ++leakIterator;
   }
}

void KMtraceLeaksView::slotContextLeak( QListViewItem *item, const QPoint &point, int column )
{
   int index = -1;

   if( item ) {
      KMtraceLeaksViewItem *kitem = (KMtraceLeaksViewItem *)item;
      QObject *obj = kitem->getModel( );
   
      if( obj->isA( "KMtraceLeak" ) ) {
         index = m_menu->exec( point );
         if( index == m_menuItemIdSuppress ) {
            KMtraceLeak *leak = (KMtraceLeak *)obj;
            if( m_suppressDialog == 0 ) {
               m_suppressDialog = new KMtraceSuppressDialog( leak, this );
               connect( m_suppressDialog, SIGNAL( suppress( KMtraceLeak *, QString ) ),
                        this, SLOT( slotSuppress( KMtraceLeak *, QString ) ) );
            } else {
               m_suppressDialog->setLeak( leak );
            }
            m_suppressDialog->exec( );
         }
      }
   }
}

void KMtraceLeaksView::doSelectItem( QListViewItem *item )
{
   KMtraceLeaksViewItem *kitem = (KMtraceLeaksViewItem *)item;
   QObject *obj = kitem->getModel( );
   
   if( obj->isA( "KMtraceLeak" ) ) {
      emit moduleUnselect( );
   } else if( obj->isA( "KMtraceCall" ) ) {
      // check if we have a module here, if yes inform everybody who is
		// interested about the current module to be shown
      KMtraceCall *call = (KMtraceCall *)obj;
      QString module = call->getModule( );
      int line = call->getLine( );
      
      if( line > 0 ) {
         emit moduleChanged( module, line );
      } else {
         emit moduleUnselect( );
      }
   }
}

void KMtraceLeaksView::slotSelectionChanged( QListViewItem *item )
{
   doSelectItem( item );
}

void KMtraceLeaksView::slotExecuted( QListViewItem *item )
{
   item->setOpen( !item->isOpen( ) );
}

void KMtraceLeaksView::slotCollapseAll( )
{
   doCollapseAll( );
}

void KMtraceLeaksView::doCollapseAll( )
{
   QListViewItemIterator iterator( this );
   QListViewItem *item;
   
   while( ( item = iterator.current( ) ) != 0 ) {
      item->setOpen( FALSE );
      ++iterator;
   }

   // we deselect the source code view (because we lost
   // the focus on a module for sure)
   emit moduleUnselect( );
}

void KMtraceLeaksView::slotExpandAll( )
{
   doExpandAll( );
}

void KMtraceLeaksView::doExpandAll( )
{
   QListViewItemIterator iterator( this );
   QListViewItem *item;
   
   while( ( item = iterator.current( ) ) != 0 ) {
      item->setOpen( TRUE );
      ++iterator;
   }

   // let's see if we still have a selection if so, emit the right
   // signals
   item = selectedItem( );
   if( item != 0 ) {
      doSelectItem( item );
   }
}

void KMtraceLeaksView::slotFind( )
{
   if( m_findDialog == 0 ) {
      m_findDialog = new KMtraceLeaksFindDialog( this, "leaksearchdialog", false );
      connect( m_findDialog, SIGNAL( search( ) ), this, SLOT( slotSearch( ) ) );
      connect( m_findDialog, SIGNAL( done( ) ), this, SLOT( slotSearchDone( ) ) );
   }

   // make sure we show the current text if already in search mode
   QString string;
   string = m_findDialog->getText( );
   if( !string.isEmpty( ) ) {
      m_findDialog->setText( string );
   }
   
   m_findDialog->show( );
   m_findDialog->result( );
}

void KMtraceLeaksView::slotSearch( )
{
   if( m_findDialog == 0 ) {
      return;
   }
   
   int whereToFind = m_findDialog->getType( );
   QString stringToFind = m_findDialog->getText( );
   
   doSearch( whereToFind, stringToFind, m_findDialog->case_sensitive( ) );
}

void KMtraceLeaksView::slotSearchDone( )
{
   if( m_findDialog == 0 ) {
      return;
   }
   
   m_findDialog->hide( );
   setFocus( );
   
   delete m_findIterator;
   m_findIterator = 0;
   m_findItem = 0;
   m_posY = 0;
}

void KMtraceLeaksView::doSearch( int where, QString what,
                                 bool caseSensitive )
{
again:
   if( m_findIterator == 0 ) {
      m_findIterator = new QListViewItemIterator( this );
   }
   
   while( ( m_findItem = m_findIterator->current( ) ) != 0 ) {
      KMtraceLeaksViewItem *kitem = (KMtraceLeaksViewItem *)m_findItem;
      QObject *obj = kitem->getModel( );
      
      m_posY += m_findItem->height( );
      
      if( obj->isA( "KMtraceCall" ) ) {
         // only search in the right place
	 KMtraceCall *call = (KMtraceCall *)obj;
	 QString moduleOrLibrary = call->getModule( );
	 int line = call->getLine( );
	 bool isLibrary = ( line == 0 );
	 QString function = call->getFunction( );
	 bool found = false;
	 
	 // respect case sensitivness
	 if( !caseSensitive ) {
	    function = function.lower( );
	    moduleOrLibrary = moduleOrLibrary.lower( );
	    what = what.lower( );
	 }
	 
	 switch( where ) {
	    case 0: // library name must start with pattern (no .so.6 for instance!)
	       if( isLibrary ) {
	          found = ( moduleOrLibrary.find( what, 0 ) == 0 );
	       }
	       break;
	    case 1: // module must match from the beginning
	       if( !isLibrary ) {
	          found = moduleOrLibrary == what;
	       }
	       break;
	    case 2: // function
	          found = function == what;
	       break;
	 }
	 
	 if( found ) {
	    QListViewItem *parent = m_findItem->parent( );
	    parent->setOpen( TRUE );
	    m_posY += parent->height( );
	    setSelected( m_findItem, TRUE );
	    setCurrentItem( m_findItem );
	    ensureVisible( 0, m_posY );
	    ++(*m_findIterator);
	    return; // quit until we search again
	 }
      }
      ++(*m_findIterator);
   }
   
   // if we reach there are no more hits, we inform the user and he can
   // choose to start from the beginning again...
   int res = KMessageBox::questionYesNo( m_findDialog,
                i18n( "End of Leaks reached.\n"
		      "Continue from the beginning?" ),
		i18n( "Find Leak" ) );
   if( res == KMessageBox::Yes ) {
     delete m_findIterator;
     m_findIterator = 0;
     m_findItem = 0;
     m_posY = 0;
     goto again;
   }
}

void KMtraceLeaksView::slotSuppress( KMtraceLeak *leak, QString string )
{
   if( m_suppressDialog == 0 ) {
      return;
   }

   // close the dialog
   m_suppressDialog->close( );

   // add the new suppression to the model, the model itself will
   // determine how many leaks match this new suppression and will
   // issue messages accordingly...
   KMtraceSuppression *suppression = new KMtraceSuppression( string, MatchByString );
   if( m_model ) {
      m_model->addSuppression( suppression );
   }
}

void KMtraceLeaksView::slotSuppressionChanged( KMtraceLeak *leak )
{
   KMtraceLeaksViewItem *kitem = leak->getView( );
   QString result;
   if( leak->isSuppressed( ) ) {
      result.sprintf( i18n( "Suppressed Leak of %d bytes in %d blocks" ),
                      leak->getBytes( ), leak->getBlocks( ) );
   } else {
      result.sprintf( i18n( "Leak of %d bytes in %d blocks" ),
                      leak->getBytes( ), leak->getBlocks( ) );
   }
   kitem->setText( 0, result );
   if( !m_showSuppressed ) {
      kitem->setVisible( !leak->isSuppressed( ) );
   }
}

void KMtraceLeaksView::setShowSuppressed( bool show )
{
   QListViewItemIterator iterator( this );
   QListViewItem *item;

   m_showSuppressed = show;
   
   while( ( item = iterator.current( ) ) != 0 ) {
      KMtraceLeaksViewItem *kitem = (KMtraceLeaksViewItem *)item;
      if( show ) {
         kitem->setVisible( true );
      } else {
         QObject *obj = kitem->getModel( );
         if( obj->isA( "KMtraceLeak" ) ) {
            KMtraceLeak *leak = (KMtraceLeak *)obj;
            QString result;
            if( leak->isSuppressed( ) ) {
               result.sprintf( i18n( "Suppressed Leak of %d bytes in %d blocks" ),
                               leak->getBytes( ), leak->getBlocks( ) );
            } else {
               result.sprintf( i18n( "Leak of %d bytes in %d blocks" ),
                               leak->getBytes( ), leak->getBlocks( ) );
            }
            kitem->setText( 0, result );
            if( !m_showSuppressed ) {
               kitem->setVisible( !leak->isSuppressed( ) );
            }
         }
      }
      ++iterator;
   }
}
