"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerADRoutes = registerADRoutes;

var _lodash = require("lodash");

var _constants = require("../utils/constants");

var _helpers = require("../utils/helpers");

var _adHelpers = require("./utils/adHelpers");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   * The OpenSearch Contributors require contributions made to
                                                                                                                                                                                                                   * this file be licensed under the Apache-2.0 license or a
                                                                                                                                                                                                                   * compatible open source license.
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   * Modifications Copyright OpenSearch Contributors. See
                                                                                                                                                                                                                   * GitHub history for details.
                                                                                                                                                                                                                   */

function registerADRoutes(apiRouter, adService) {
  apiRouter.post('/detectors', adService.putDetector);
  apiRouter.put('/detectors/{detectorId}', adService.putDetector);
  apiRouter.post('/detectors/_search', adService.searchDetector);
  apiRouter.post('/detectors/results/_search/', adService.searchResults);
  apiRouter.post('/detectors/results/_search', adService.searchResults);
  apiRouter.post('/detectors/results/_search/{resultIndex}/{onlyQueryCustomResultIndex}', adService.searchResults);
  apiRouter.get('/detectors/{detectorId}', adService.getDetector);
  apiRouter.get('/detectors', adService.getDetectors);
  apiRouter.post('/detectors/preview', adService.previewDetector);
  apiRouter.get('/detectors/{id}/results/{isHistorical}/{resultIndex}/{onlyQueryCustomResultIndex}', adService.getAnomalyResults);
  apiRouter.get('/detectors/{id}/results/{isHistorical}', adService.getAnomalyResults);
  apiRouter.delete('/detectors/{detectorId}', adService.deleteDetector);
  apiRouter.post('/detectors/{detectorId}/start', adService.startDetector);
  apiRouter.post('/detectors/{detectorId}/stop/{isHistorical}', adService.stopDetector);
  apiRouter.get('/detectors/{detectorId}/_profile', adService.getDetectorProfile);
  apiRouter.get('/detectors/{detectorName}/_match', adService.matchDetector);
  apiRouter.get('/detectors/_count', adService.getDetectorCount);
  apiRouter.post('/detectors/{detectorId}/_topAnomalies/{isHistorical}', adService.getTopAnomalyResults);
  apiRouter.post('/detectors/_validate/{validationType}', adService.validateDetector);
}

class AdService {
  constructor(client) {
    _defineProperty(this, "client", void 0);

    _defineProperty(this, "deleteDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.deleteDetector', {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - deleteDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "previewDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const requestBody = JSON.stringify((0, _adHelpers.convertPreviewInputKeysToSnakeCase)(request.body));
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.previewDetector', {
          body: requestBody
        });
        const transformedKeys = (0, _helpers.mapKeysDeep)(response, _helpers.toCamel);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            //@ts-ignore
            response: (0, _adHelpers.anomalyResultMapper)(transformedKeys.anomalyResult)
          }
        });
      } catch (err) {
        console.log('Anomaly detector - previewDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "putDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params; //@ts-ignore

        const ifSeqNo = request.body.seqNo; //@ts-ignore

        const ifPrimaryTerm = request.body.primaryTerm;
        const requestBody = JSON.stringify((0, _adHelpers.convertDetectorKeysToSnakeCase)(request.body));
        let params = {
          detectorId: detectorId,
          ifSeqNo: ifSeqNo,
          ifPrimaryTerm: ifPrimaryTerm,
          body: requestBody
        };
        let response;

        if ((0, _lodash.isNumber)(ifSeqNo) && (0, _lodash.isNumber)(ifPrimaryTerm)) {
          response = await this.client.asScoped(request).callAsCurrentUser('ad.updateDetector', params);
        } else {
          response = await this.client.asScoped(request).callAsCurrentUser('ad.createDetector', {
            body: params.body
          });
        }

        const resp = { ...response.anomaly_detector,
          id: response._id,
          primaryTerm: response._primary_term,
          seqNo: response._seq_no
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: (0, _adHelpers.convertDetectorKeysToCamelCase)(resp)
          }
        });
      } catch (err) {
        console.log('Anomaly detector - PutDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "validateDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          validationType
        } = request.params;
        const requestBody = JSON.stringify((0, _adHelpers.convertPreviewInputKeysToSnakeCase)(request.body));
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.validateDetector', {
          body: requestBody,
          validationType: validationType
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - validateDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const detectorResponse = await this.client.asScoped(request).callAsCurrentUser('ad.getDetector', {
          detectorId
        }); // Populating static detector fields

        const staticFields = {
          id: detectorResponse._id,
          primaryTerm: detectorResponse._primary_term,
          seqNo: detectorResponse._seq_no,
          ...(0, _adHelpers.convertStaticFieldsToCamelCase)(detectorResponse.anomaly_detector)
        }; // Get real-time and historical task info to populate the
        // task and job-related fields
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no detector jobs have been ran on a new cluster.

        let realtimeTasksResponse = {};
        let historicalTasksResponse = {};

        try {
          realtimeTasksResponse = await this.client.asScoped(request).callAsCurrentUser('ad.searchTasks', {
            body: (0, _adHelpers.getLatestTaskForDetectorQuery)(detectorId, true)
          });
          historicalTasksResponse = await this.client.asScoped(request).callAsCurrentUser('ad.searchTasks', {
            body: (0, _adHelpers.getLatestTaskForDetectorQuery)(detectorId, false)
          });
        } catch (err) {
          if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }

        const realtimeTask = (0, _lodash.get)((0, _lodash.get)(realtimeTasksResponse, 'hits.hits', []).map(taskResponse => {
          return {
            id: (0, _lodash.get)(taskResponse, '_id'),
            ...(0, _lodash.get)(taskResponse, '_source')
          };
        }), 0);
        const historicalTask = (0, _lodash.get)((0, _lodash.get)(historicalTasksResponse, 'hits.hits', []).map(taskResponse => {
          return {
            id: (0, _lodash.get)(taskResponse, '_id'),
            ...(0, _lodash.get)(taskResponse, '_source')
          };
        }), 0);
        const taskAndJobFields = (0, _adHelpers.convertTaskAndJobFieldsToCamelCase)(realtimeTask, historicalTask, detectorResponse.anomaly_detector_job); // Combine the static and task-and-job-related fields into
        // a final response

        const finalResponse = { ...staticFields,
          ...taskAndJobFields
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: finalResponse
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get detector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "startDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        var _request$body, _request$body2;

        const {
          detectorId
        } = request.params; //@ts-ignore

        const startTime = (_request$body = request.body) === null || _request$body === void 0 ? void 0 : _request$body.startTime; //@ts-ignore

        const endTime = (_request$body2 = request.body) === null || _request$body2 === void 0 ? void 0 : _request$body2.endTime;
        let requestParams = {
          detectorId: detectorId
        };
        let requestPath = 'ad.startDetector'; // If a start and end time are passed: we want to start a historical analysis

        if ((0, _lodash.isNumber)(startTime) && (0, _lodash.isNumber)(endTime)) {
          requestParams = { ...requestParams,
            body: {
              start_time: startTime,
              end_time: endTime
            }
          };
          requestPath = 'ad.startHistoricalDetector';
        }

        const response = await this.client.asScoped(request).callAsCurrentUser(requestPath, requestParams);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - startDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "stopDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          detectorId,
          isHistorical
        } = request.params;
        isHistorical = JSON.parse(isHistorical);
        const requestPath = isHistorical ? 'ad.stopHistoricalDetector' : 'ad.stopDetector';
        const response = await this.client.asScoped(request).callAsCurrentUser(requestPath, {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - stopDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getDetectorProfile", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.detectorProfile', {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - detectorProfile', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "searchDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const requestBody = JSON.stringify(request.body);
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.searchDetector', {
          body: requestBody
        });
        const totalDetectors = (0, _lodash.get)(response, 'hits.total.value', 0);
        const detectors = (0, _lodash.get)(response, 'hits.hits', []).map(detector => ({ ...(0, _adHelpers.convertDetectorKeysToCamelCase)(detector._source),
          id: detector._id,
          seqNo: detector._seq_no,
          primaryTerm: detector._primary_term
        }));
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalDetectors,
              detectors
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search detectors', err);

        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectors: []
              }
            }
          });
        }

        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "searchResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        var {
          resultIndex,
          onlyQueryCustomResultIndex
        } = request.params;

        if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_AD_RESULT_INDEX_PREFIX)) {
          // Set resultIndex as '' means no custom result index specified, will only search anomaly result from default index.
          resultIndex = '';
        }

        let requestParams = {
          resultIndex: resultIndex,
          onlyQueryCustomResultIndex: onlyQueryCustomResultIndex
        };
        const requestBody = JSON.stringify(request.body);
        const response = !resultIndex ? await this.client.asScoped(request).callAsCurrentUser('ad.searchResults', {
          body: requestBody
        }) : await this.client.asScoped(request).callAsCurrentUser('ad.searchResultsFromCustomResultIndex', { ...requestParams,
          body: requestBody
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search anomaly result', err);

        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectors: []
              }
            }
          });
        }

        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getDetectors", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          from = 0,
          size = 20,
          search = '',
          indices = '',
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = 'name'
        } = request.query;
        const mustQueries = [];

        if (search.trim()) {
          mustQueries.push({
            query_string: {
              fields: ['name', 'description'],
              default_operator: 'AND',
              query: `*${search.trim().split('-').join('* *')}*`
            }
          });
        }

        if (indices.trim()) {
          mustQueries.push({
            query_string: {
              fields: ['indices'],
              default_operator: 'AND',
              query: `*${indices.trim().split('-').join('* *')}*`
            }
          });
        } //Allowed sorting columns


        const sortQueryMap = {
          name: {
            'name.keyword': sortDirection
          },
          indices: {
            'indices.keyword': sortDirection
          },
          lastUpdateTime: {
            last_update_time: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];

        if (sortQuery) {
          sort = sortQuery;
        } //Preparing search request


        const requestBody = {
          sort,
          size,
          from,
          query: {
            bool: {
              must: mustQueries
            }
          }
        };
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.searchDetector', {
          body: requestBody
        });
        const totalDetectors = (0, _lodash.get)(response, 'hits.total.value', 0); //Get all detectors from search detector API

        const allDetectors = (0, _lodash.get)(response, 'hits.hits', []).reduce((acc, detectorResponse) => ({ ...acc,
          [detectorResponse._id]: {
            id: detectorResponse._id,
            primaryTerm: detectorResponse._primary_term,
            seqNo: detectorResponse._seq_no,
            ...(0, _adHelpers.convertStaticFieldsToCamelCase)(detectorResponse._source)
          }
        }), {});
        const allDetectorsMap = Object.values(allDetectors).reduce((acc, detector) => ({ ...acc,
          [detector.id]: detector
        }), {}); //Given each detector from previous result, get aggregation to power list

        const allDetectorIds = Object.keys(allDetectorsMap);
        let requestParams = {
          // If specifying result index, will query anomaly result from both default and custom result indices.
          // If no valid result index specified, just query anomaly result from default result index.
          // Here we specify custom AD result index prefix pattern to query all custom result indices.
          resultIndex: _constants.CUSTOM_AD_RESULT_INDEX_PREFIX + '*',
          onlyQueryCustomResultIndex: 'false'
        };
        const aggregationResult = await this.client.asScoped(request).callAsCurrentUser('ad.searchResultsFromCustomResultIndex', { ...requestParams,
          body: (0, _adHelpers.getResultAggregationQuery)(allDetectorIds, {
            from,
            size,
            sortField,
            sortDirection,
            search,
            indices
          })
        });
        const aggsDetectors = (0, _lodash.get)(aggregationResult, 'aggregations.unique_detectors.buckets', []).reduce((acc, agg) => {
          return { ...acc,
            [agg.key]: { ...allDetectorsMap[agg.key],
              totalAnomalies: agg.total_anomalies_in_24hr.doc_count,
              lastActiveAnomaly: agg.latest_anomaly_time.value
            }
          };
        }, {}); // Aggregation will not return values where anomalies for detectors are not generated, loop through it and fill values with 0

        const unUsedDetectors = (0, _lodash.pullAll)(allDetectorIds, Object.keys(aggsDetectors)).reduce((acc, unusedDetector) => {
          return { ...acc,
            [unusedDetector]: { ...allDetectorsMap[unusedDetector],
              totalAnomalies: 0,
              lastActiveAnomaly: 0
            }
          };
        }, {}); // If sorting criteria is from the aggregation manage pagination in memory.

        let finalDetectors = (0, _lodash.orderBy)({ ...aggsDetectors,
          ...unUsedDetectors
        }, [sortField], [sortDirection]);

        if (!sortQueryMap[sortField]) {
          finalDetectors = Object.values(finalDetectors).slice(from, from + size).reduce((acc, detector) => ({ ...acc,
            [detector.id]: detector
          }), {});
        } // Fetch the latest realtime and historical tasks for all detectors
        // using terms aggregations
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no detector jobs have been ran on a new cluster.


        let realtimeTasksResponse = {};
        let historicalTasksResponse = {};

        try {
          realtimeTasksResponse = await this.client.asScoped(request).callAsCurrentUser('ad.searchTasks', {
            body: (0, _adHelpers.getLatestDetectorTasksQuery)(true)
          });
          historicalTasksResponse = await this.client.asScoped(request).callAsCurrentUser('ad.searchTasks', {
            body: (0, _adHelpers.getLatestDetectorTasksQuery)(false)
          });
        } catch (err) {
          if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }

        const realtimeTasks = (0, _lodash.get)(realtimeTasksResponse, 'aggregations.detectors.buckets', []).reduce((acc, bucket) => {
          return { ...acc,
            [bucket.key]: {
              realtimeTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});
        const historicalTasks = (0, _lodash.get)(historicalTasksResponse, 'aggregations.detectors.buckets', []).reduce((acc, bucket) => {
          return { ...acc,
            [bucket.key]: {
              historicalTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {}); // Get real-time and historical task info by looping through each detector & retrieving
        //    - curState by getting real-time task state
        //    - enabledTime by getting real-time task's execution_start time
        //    - taskId by getting historical task's _id

        finalDetectors.forEach(detector => {
          const realtimeTask = (0, _lodash.get)(realtimeTasks[detector.id], 'realtimeTask._source');
          detector.curState = (0, _adHelpers.getTaskState)(realtimeTask);
          detector.enabledTime = (0, _lodash.get)(realtimeTask, 'execution_start_time');
          detector.taskId = (0, _lodash.get)(historicalTasks[detector.id], 'historicalTask._id');
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalDetectors,
              detectorList: Object.values(finalDetectors)
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search detectors', err);

        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectorList: []
              }
            }
          });
        }

        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getAnomalyResults", async (context, request, opensearchDashboardsResponse) => {
      let {
        id,
        isHistorical,
        resultIndex,
        onlyQueryCustomResultIndex
      } = request.params;

      if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_AD_RESULT_INDEX_PREFIX)) {
        // Set resultIndex as '' means no custom result index specified, will only search anomaly result from default index.
        resultIndex = '';
      }

      isHistorical = JSON.parse(isHistorical); // Search by task id if historical, or by detector id if realtime

      const searchTerm = isHistorical ? {
        task_id: id
      } : {
        detector_id: id
      };

      try {
        const {
          from = 0,
          size = 20,
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = _constants.AD_DOC_FIELDS.DATA_START_TIME,
          startTime = 0,
          endTime = 0,
          fieldName = '',
          anomalyThreshold = -1,
          entityList = ''
        } = request.query;
        const entityListAsObj = entityList.length === 0 ? {} : JSON.parse(entityList);
        const entityFilters = (0, _lodash.isEmpty)(entityListAsObj) ? [] : (0, _adHelpers.getFiltersFromEntityList)(entityListAsObj); //Allowed sorting columns

        const sortQueryMap = {
          anomalyGrade: {
            anomaly_grade: sortDirection
          },
          confidence: {
            confidence: sortDirection
          },
          [_constants.AD_DOC_FIELDS.DATA_START_TIME]: {
            [_constants.AD_DOC_FIELDS.DATA_START_TIME]: sortDirection
          },
          [_constants.AD_DOC_FIELDS.DATA_END_TIME]: {
            [_constants.AD_DOC_FIELDS.DATA_END_TIME]: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];

        if (sortQuery) {
          sort = sortQuery;
        } //Preparing search request


        const requestBody = {
          sort,
          size,
          from,
          query: {
            bool: {
              filter: [{
                term: searchTerm
              }, {
                range: {
                  anomaly_grade: {
                    gt: anomalyThreshold
                  }
                }
              }, ...entityFilters]
            }
          }
        }; // If querying RT results: remove any results that include a task_id, as this indicates
        // a historical result from a historical task.

        if (!isHistorical) {
          requestBody.query.bool = { ...requestBody.query.bool,
            ...{
              must_not: {
                exists: {
                  field: 'task_id'
                }
              }
            }
          };
        }

        try {
          const filterSize = requestBody.query.bool.filter.length;

          if (fieldName) {
            (startTime || endTime) && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.format`, 'epoch_millis');
            startTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.gte`, startTime);
            endTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.lte`, endTime);
          }
        } catch (error) {
          console.log('wrong date range filter', error);
        }

        let requestParams = {
          resultIndex: resultIndex,
          onlyQueryCustomResultIndex: onlyQueryCustomResultIndex
        };
        const response = !resultIndex ? await this.client.asScoped(request).callAsCurrentUser('ad.searchResults', {
          body: requestBody
        }) : await this.client.asScoped(request).callAsCurrentUser('ad.searchResultsFromCustomResultIndex', { ...requestParams,
          body: requestBody
        });
        const totalResults = (0, _lodash.get)(response, 'hits.total.value', 0);
        const detectorResult = [];
        const featureResult = {};
        (0, _lodash.get)(response, 'hits.hits', []).forEach(result => {
          detectorResult.push({
            startTime: result._source.data_start_time,
            endTime: result._source.data_end_time,
            plotTime: result._source.data_end_time,
            contributions: result._source.anomaly_grade > 0 ? result._source.relevant_attribution : {},
            confidence: result._source.confidence != null && result._source.confidence !== 'NaN' && result._source.confidence > 0 ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(result._source.confidence)) : 0,
            anomalyGrade: result._source.anomaly_grade != null && result._source.anomaly_grade !== 'NaN' && result._source.anomaly_grade > 0 ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(result._source.anomaly_grade)) : 0,
            ...(result._source.entity != null ? {
              entity: result._source.entity
            } : {}),
            // TODO: we should refactor other places to read feature data from
            // AnomalyResult, instead of having separate FeatureData which is hard
            // to know feature data belongs to which anomaly result
            features: this.getFeatureData(result)
          });

          result._source.feature_data.forEach(featureData => {
            if (!featureResult[featureData.feature_id]) {
              featureResult[featureData.feature_id] = [];
            }

            featureResult[featureData.feature_id].push({
              startTime: result._source.data_start_time,
              endTime: result._source.data_end_time,
              plotTime: result._source.data_end_time,
              data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0,
              name: featureData.feature_name,
              expectedValue: this.getExpectedValue(result, featureData)
            });
          });
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalAnomalies: totalResults,
              results: detectorResult,
              featureResults: featureResult
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get results', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getTopAnomalyResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          detectorId,
          isHistorical
        } = request.params;
        isHistorical = JSON.parse(isHistorical);
        const requestPath = isHistorical ? 'ad.topHistoricalAnomalyResults' : 'ad.topAnomalyResults';
        const response = await this.client.asScoped(request).callAsCurrentUser(requestPath, {
          detectorId: detectorId,
          body: request.body
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getTopAnomalyResults', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "matchDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorName
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.matchDetector', {
          detectorName
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - matchDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getDetectorCount", async (context, request, opensearchDashboardsResponse) => {
      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.detectorCount');
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getDetectorCount', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getFeatureData", rawResult => {
      const featureResult = {};

      rawResult._source.feature_data.forEach(featureData => {
        featureResult[featureData.feature_id] = {
          startTime: rawResult._source.data_start_time,
          endTime: rawResult._source.data_end_time,
          plotTime: rawResult._source.data_end_time,
          data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0,
          name: featureData.feature_name,
          expectedValue: this.getExpectedValue(rawResult, featureData)
        };
      });

      return featureResult;
    });

    _defineProperty(this, "getExpectedValue", (rawResult, featureData) => {
      let expectedValue = featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0;

      if (rawResult._source.anomaly_grade > 0) {
        const expectedValueList = rawResult._source.expected_values;

        if ((expectedValueList === null || expectedValueList === void 0 ? void 0 : expectedValueList.length) > 0) {
          expectedValueList[0].value_list.forEach(expect => {
            if (expect.feature_id === featureData.feature_id) {
              expectedValue = expect.data;
            }
          });
        }
      }

      return expectedValue;
    });

    this.client = client;
  }

}

exports.default = AdService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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