/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.activemq;

import org.apache.activemq.command.ActiveMQMessage;
import org.apache.activemq.command.ActiveMQQueue;
import org.apache.activemq.command.ActiveMQTopic;
import org.junit.Test;

import javax.jms.Destination;
import javax.jms.MessageProducer;
import javax.jms.Session;

import java.util.ArrayList;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

public class CompositeQPolicyEntryTest extends RuntimeConfigTestSupport {

    String configurationSeed = "CompositeQPolicyEntryTest";

    @Test
    public void testMod() throws Exception {
        final String brokerConfig = configurationSeed + "-policy-ml-broker";
        applyNewConfig(brokerConfig, configurationSeed + "-policy-fc");
        startBroker(brokerConfig);
        assertTrue("broker alive", brokerService.isStarted());

        exerciseCompositeQueue();
        verifyTopicPFC("my.topic", true);

        applyNewConfig(brokerConfig, configurationSeed + "-policy-fc-add", SLEEP);
        verifyTopicPFC("my.topic", false);


    }



    private void exerciseCompositeQueue() throws Exception {
        ActiveMQConnection connection = new ActiveMQConnectionFactory("vm://localhost").createActiveMQConnection("user","user");
        try {
            connection.start();
            Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);

            ArrayList<Destination> dests = new ArrayList<>();
            dests.add(new ActiveMQQueue("my.filtered.destination"));
            dests.add(new ActiveMQQueue("my.composite.queue"));
            dests.add(new ActiveMQTopic("my.topic"));
            dests.add(new ActiveMQTopic("our.topic"));


            for (Destination destination : dests) {
                MessageProducer messageProducer = session.createProducer(destination);
                ActiveMQMessage activeMQMessage = new ActiveMQMessage();
                messageProducer.send(activeMQMessage);
                activeMQMessage = new ActiveMQMessage();
                activeMQMessage.setProperty("header", "one");
                messageProducer.send(activeMQMessage);
                activeMQMessage = new ActiveMQMessage();
                activeMQMessage.setProperty("header", "two");
                messageProducer.send(activeMQMessage);
            }

        } finally {
            connection.close();
        }
    }

    private void verifyTopicPFC(String dest, boolean pfc) throws Exception {
        ActiveMQConnection connection = new ActiveMQConnectionFactory("vm://localhost").createActiveMQConnection("user","user");
        try {
            connection.start();
            Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
            session.createConsumer(session.createTopic(dest));

            assertEquals(pfc, brokerService.getRegionBroker().getDestinationMap().get(new ActiveMQTopic(dest)).isProducerFlowControl());
        } finally {
            connection.close();
        }
    }
}
