package com.redhat.installer.layering.validator.container;

import org.osgi.framework.Version;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

public class BasicContainerInformationLoader implements ContainerInformationLoader {
    private final ContainerType containerType;
    private final VersionLoader versionLoader;
    private final Path installRoot;
    private final List<Path> requiredFiles;
    private final List<Path> productFiles;

    public BasicContainerInformationLoader(ContainerType containerType, VersionLoader versionLoader,
                                           Path installRoot, List<Path> requiredFiles, List<Path> productFiles) {
        this.containerType = containerType;
        this.versionLoader = versionLoader;
        this.installRoot = installRoot;
        this.requiredFiles = requiredFiles;
        this.productFiles = productFiles;
    }

    @Override
    public ContainerInformation load() {
        if (missesAnyPath(installRoot, requiredFiles)) {
            return new ContainerInformation(containerType, Version.emptyVersion, InstallState.NotInstalled);
        }

        if (containsAnyPath(installRoot, productFiles)) {
            return new ContainerInformation(containerType, Version.emptyVersion, InstallState.ProductAlreadyInstalled);
        }

        Version version = versionLoader.load();

        if (Version.emptyVersion.equals(version)) {
            return new ContainerInformation(containerType, version, InstallState.MissingVersion);
        }

        return new ContainerInformation(containerType, version, InstallState.Installed);
    }

    /**
     * Returns true if targetPath doesn't contain all paths in pathList, false if any are missing.
     * @param targetPath root to test for path existence
     * @param pathList list of paths to check for
     * @return true if all are present, false otherwise
     */
    private boolean missesAnyPath(Path targetPath, List<Path> pathList) {
        List<Path> missingFiles = findFiles(targetPath, pathList);
        return !missingFiles.isEmpty();
    }

    /**
     * Returns true if targetPath contains any path in pathList, false if none are present.
     * @param targetPath root to test for path existence
     * @param pathList list of paths to check for
     * @return true if any are present, false otherwise.
     */
    private boolean containsAnyPath(Path targetPath, List<Path> pathList){
        List<Path> missingFiles = findFiles(targetPath, pathList);
        return !(missingFiles.size() == pathList.size());
    }

    private List<Path> findFiles(Path installPath, List<Path> files) {
        List<Path> missingFiles = new ArrayList<>();
        for (Path file : files) {
            Path filePath = installPath.resolve(file);
            if (!Files.exists(filePath)) {
                missingFiles.add(filePath);
            }
        }
        return missingFiles;
    }
}
