package org.jboss.brmsbpmsuite.patching.client;

import org.jboss.brmsbpmsuite.patching.systemproperty.SystemPropertyPatcher;
import org.jboss.brmsbpmsuite.patching.xml.XMLDocumentDao;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import java.io.*;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class ConfigurationPatcher implements Patcher {
    private static final Logger LOGGER = LoggerFactory.getLogger(ConfigurationPatcher.class);

    private static final FilenameFilter FILENAME_FILTER = (dir, name) -> name.matches("standalone.*\\.xml");

    private final File patchesDir;
    private final File configDir;
    private final XMLDocumentDao documentDao;

    public ConfigurationPatcher(File patchesDir, File configDir) throws ParserConfigurationException, TransformerConfigurationException {
        this.patchesDir = patchesDir;
        this.configDir = configDir;
        this.documentDao = new XMLDocumentDao();
    }

    @Override
    public void apply() throws IOException {
        LOGGER.info("Applying updates to configuration root {}", configDir);

        for (File file : intersectDirs()) {
            try {
                File patchFile = new File(patchesDir, file.getName());
                File configFile = new File(configDir, file.getName());
                applyPatch(patchFile, configFile);
            } catch (SAXException | ParserConfigurationException | TransformerException e) {
                LOGGER.error("Could not apply the patch in file: " + file.getAbsolutePath(), e);
            }
        }
    }

    private List<File> intersectDirs() {
        File[] dir1Files = patchesDir.listFiles(ConfigurationPatcher.FILENAME_FILTER);
        if (dir1Files == null) {
            return new ArrayList<>();
        }

        return Stream.of(dir1Files)
                .map(dir1File -> new File(configDir, dir1File.getName()))
                .filter(File::exists)
                .collect(Collectors.toList());
    }

    private void applyPatch(File patchFile, File configFile) throws IOException, SAXException, ParserConfigurationException, TransformerException {
        LOGGER.trace("Applying patch {} to {}", patchFile, configFile);

        InputStream patchInputStream = new FileInputStream(patchFile);
        InputStream configIputStream = new FileInputStream(configFile);

        Document patchDocument = documentDao.load(patchInputStream);
        Document configDocument = documentDao.load(configIputStream);

        SystemPropertyPatcher propertyPatcher = new SystemPropertyPatcher(configDocument);
        Document updatedDocument = propertyPatcher.apply(patchDocument);

        OutputStream patchedOutputStream = new FileOutputStream(configFile);
        documentDao.save(patchedOutputStream, updatedDocument);
    }

    @Override
    public void backup(File backupDir) throws IOException {
        LOGGER.info("Backing-up (copying) configuration root {} to {}",  configDir, backupDir);

        backupDir.mkdirs();
        for (File file : intersectDirs()) {
            File configFile = new File(configDir, file.getName());
            File backupFile = new File(backupDir, file.getName());

            LOGGER.trace("Copying {} to {}", configFile, backupFile);

            Files.copy(configFile.toPath(), backupFile.toPath());
        }
    }

}
