package com.redhat.installer.layering.validator;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.installer.DataValidator;
import com.redhat.installer.layering.validator.container.*;
import com.redhat.installer.layering.validator.container.eap.EapInformationLoader;
import com.redhat.installer.layering.validator.container.eap.EapValidator;
import com.redhat.installer.layering.validator.container.tomcat.TomcatInformationLoader;
import com.redhat.installer.layering.validator.container.tomcat.TomcatValidator;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;

import static com.redhat.installer.layering.validator.container.InstallState.MissingVersion;
import static com.redhat.installer.layering.validator.container.InstallState.ProductAlreadyInstalled;
import static com.redhat.installer.layering.validator.container.VersionState.HighVersion;
import static com.redhat.installer.layering.validator.container.VersionState.LowVersion;

/**
 * Class that evaluates the directory the user provides is a valid platform.
 * Moderate rewrite and update of EapExistsValidator
 * <p>
 * Current valid platforms:
 * EAP 7.x.y
 * JWS 5.0 / Tomcat 9
 */
public class IsSupportedContainerValidator implements DataValidator {
    private AutomatedInstallData installData;
    private String message;

    @Override
    public Status validateData(AutomatedInstallData adata) {
        this.message = null;
        this.installData = adata;

        ContainerType discovered = getContainerInInstallationPath();

        installData.setVariable("eap.install", ContainerType.EAP == discovered ? "true" : "false");
        installData.setVariable("jws.install", ContainerType.Tomcat == discovered ? "true" : "false");

        return discovered == ContainerType.Unsupported ? Status.ERROR : Status.OK;
    }

    /**
     * Returns the container type discovered in the user's provided installation path.
     *
     * @return value from ContainerType enum that indicates what the target container type is.
     */
    private ContainerType getContainerInInstallationPath() {
        Path installPath = Paths.get(installData.getInstallPath());
        String productName = installData.getVariable("product.name");

        ContainerInformationLoader eapLoader = new EapInformationLoader(installPath, productName);
        ContainerInformationLoader tomcatLoader = new TomcatInformationLoader(installPath);

        int notInstalledCount = 0;
        List<ContainerInformationLoader> containerInformationLoaders = Arrays.asList(eapLoader, tomcatLoader);
        for (ContainerInformationLoader informationLoader : containerInformationLoaders) {
            ContainerInformation containerInformation = informationLoader.load();
            switch(containerInformation.state) {
                case Installed:
                    VersionValidator containerValidator = versionValidatorFor(containerInformation.type, installData);
                    VersionState versionState = containerValidator.validate(containerInformation.version);
                    if (versionState == VersionState.OK) {
                        return containerInformation.type;
                    }
                    addMessageForUnsupportedVersion(versionState, containerInformation, containerValidator);
                    break;
                case NotInstalled:
                    notInstalledCount++;
                    break;
                default:
                    addMessageForUnsupportedEnvironment(containerInformation.state, containerInformation);
            }
        }

        if (notInstalledCount == containerInformationLoaders.size()) {
            setMessage("container.no.supported");
        }

        return ContainerType.Unsupported;
    }

    private void addMessageForUnsupportedVersion(VersionState versionState, ContainerInformation containerInformation,
                                                 VersionValidator containerValidator) {
        String typeString = containerInformation.type.toString();
        String versionString = containerInformation.version.toString();
        if (versionState == LowVersion) {
            setMessage("container.low.version", typeString, containerValidator.minSupportedVersion(), versionString);
        } else if (versionState == HighVersion) {
            setMessage("container.high.version", typeString, containerValidator.maxSupportedVersion(), versionString);
        }
    }

    private void addMessageForUnsupportedEnvironment(InstallState installState, ContainerInformation containerInformation) {
        if (installState == MissingVersion) {
            String typeString = containerInformation.type.toString();
            setMessage("container.missing.version", typeString);
        } else if (installState == ProductAlreadyInstalled) {
            setMessage("container.product.already.installed");
        }
    }

    private VersionValidator versionValidatorFor(ContainerType type, AutomatedInstallData installData) {
        if (type == ContainerType.EAP) {
            return new EapValidator(installData);
        }
        return new TomcatValidator(installData);
    }

    @Override
    public String getErrorMessageId() {
        return "";
    }

    @Override
    public String getWarningMessageId() {
        return "";
    }

    public void setMessage(String messageId, String... vars) {
        String messageText = installData.langpack.getString(messageId);
        this.message = String.format(messageText, (Object[]) vars);
    }

    @Override
    public boolean getDefaultAnswer() {
        return true;
    }

    @Override
    public String getFormattedMessage() {
        return message;
    }

}
