package com.redhat.installer.layering.validator.container.tomcat;

import com.izforge.izpack.util.Debug;
import com.redhat.installer.layering.validator.container.VersionLoader;
import org.apache.commons.lang.SystemUtils;
import org.osgi.framework.Version;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.stream.Stream;

public class TomcatVersionLoader implements VersionLoader {
    private Path installRoot;

    public TomcatVersionLoader(Path installRoot) {
        this.installRoot = installRoot;
    }

    @Override
    public Version load() {
        Path versionExecutablePath = findVersionExecutable(installRoot);
        if (!Files.exists(versionExecutablePath)) {
            return Version.emptyVersion;
        }
        return extractVersion(getExecutableOutputAsLines(versionExecutablePath));
    }

    private Path findVersionExecutable(Path tomcatHome) {
        String filename;
        if (SystemUtils.IS_OS_WINDOWS) {
            filename = "bin/version.bat";
        } else {
            filename = "bin/version.sh";
        }
        return tomcatHome.resolve(filename);
    }

    private Version extractVersion(Stream<String> executableOutput) {
        return executableOutput.filter(line -> line.startsWith("Server version"))
                .map(line -> line.split("/"))
                .map(line -> line[1].trim())
                .findFirst()
                .map(Version::parseVersion)
                .orElse(Version.emptyVersion);
    }

    private Stream<String> getExecutableOutputAsLines(Path executable) {
        // read output of any arbitrary program into a string, then return it.
        ProcessBuilder pb = new ProcessBuilder(executable.toString());
        // run while setting CATALINA_HOME to the known correct value.
        if (SystemUtils.IS_OS_WINDOWS){
            pb.environment().put("CATALINA_HOME", installRoot.toString());
        }
        try {
            if (Files.exists(executable)) {
                Process exec = pb.start();
                exec.waitFor();
                BufferedReader br = new BufferedReader(new InputStreamReader(exec.getInputStream()));
                return br.lines();
            } else {
                Debug.log(String.format("%s executable does not exist.", executable.toString()));
                return Stream.empty();
            }
        } catch (InterruptedException e) {
            Debug.log(String.format("InterruptedException encountered when attempting to execute %s.", executable.toString()));
            return Stream.empty();
        } catch (IOException e) {
            Debug.log(String.format("IOException encountered when attempting to execute %s.", executable.toString()));
            return Stream.empty();
        }
    }
}
