/*
 * Copyright 2019 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.optaplanner.core.impl.score.stream.drools.uni;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;

import org.drools.model.Global;
import org.drools.model.PatternDSL;
import org.drools.model.Rule;
import org.drools.model.RuleItemBuilder;
import org.optaplanner.core.api.score.holder.AbstractScoreHolder;
import org.optaplanner.core.api.score.stream.uni.UniConstraintCollector;
import org.optaplanner.core.impl.score.stream.drools.DroolsConstraint;
import org.optaplanner.core.impl.score.stream.drools.DroolsConstraintFactory;
import org.optaplanner.core.impl.score.stream.drools.common.DroolsAbstractConstraintStream;
import org.optaplanner.core.impl.score.stream.drools.common.DroolsLogicalTuple;

import static org.drools.model.PatternDSL.declarationOf;
import static org.drools.model.PatternDSL.pattern;

public final class DroolsGroupingUniConstraintStream<Solution_, A, NewA>
        extends DroolsAbstractUniConstraintStream<Solution_, NewA> {

    private final DroolsAbstractUniConstraintStream<Solution_, A> parent;
    private final Function<A, NewA> groupKeyMapping;
    private final UniConstraintCollector<A, ?, NewA> collector;
    private int ruleId = -1;

    public DroolsGroupingUniConstraintStream(DroolsConstraintFactory<Solution_> constraintFactory,
            DroolsAbstractUniConstraintStream<Solution_, A> parent, Function<A, NewA> groupKeyMapping) {
        super(constraintFactory);
        this.parent = parent;
        this.groupKeyMapping = groupKeyMapping;
        this.collector = null;
    }

    public <ResultContainer_> DroolsGroupingUniConstraintStream(
            DroolsConstraintFactory<Solution_> constraintFactory,
            DroolsAbstractUniConstraintStream<Solution_, A> parent,
            UniConstraintCollector<A, ResultContainer_, NewA> collector) {
        super(constraintFactory);
        this.parent = parent;
        this.groupKeyMapping = null;
        this.collector = collector;
    }

    @Override
    public List<DroolsFromUniConstraintStream<Solution_, Object>> getFromStreamList() {
        return parent.getFromStreamList();
    }

    // ************************************************************************
    // Pattern creation
    // ************************************************************************

    @Override
    public Optional<Rule> buildRule(DroolsConstraint<Solution_> constraint,
            Global<? extends AbstractScoreHolder<?>> scoreHolderGlobal) {
        Object createdRuleId = createRuleIdIfAbsent(constraint.getConstraintFactory());
        Rule rule;
        if (groupKeyMapping != null) {
            String ruleName = "Helper rule #" + createdRuleId + " (GroupBy collecting)";
            rule = PatternDSL.rule(constraint.getConstraintPackage(), ruleName)
                    .build(parent.createCondition().completeWithLogicalInsert(createdRuleId, groupKeyMapping)
                            .toArray(new RuleItemBuilder<?>[0]));
        } else if (collector != null) {
            String ruleName = "Helper rule #" + createdRuleId + " (GroupBy remapping)";
            rule = PatternDSL.rule(constraint.getConstraintPackage(), ruleName)
                    .build(parent.createCondition().completeWithLogicalInsert(createdRuleId, collector)
                            .toArray(new RuleItemBuilder<?>[0]));
        } else {
            throw new IllegalStateException();
        }
        return Optional.of(rule);
    }

    @Override
    public DroolsUniCondition<NewA> createCondition() {
        return new DroolsUniCondition<>(declarationOf(DroolsLogicalTuple.class), var -> {
            Object createdRuleId = createRuleIdIfAbsent(getConstraintFactory());
            return pattern(var).expr(logicalTuple ->
                    Objects.equals(logicalTuple.getRuleId(), createdRuleId));
        });
    }

    /**
     * Guarantees that rule IDs are generated in an unbroken sequence.
     * Needed in order to be able to bind the condition in the new rule to the {@link DroolsLogicalTuple}s generated by
     * the old rule.
     * The idea here is that rule creating ({@link DroolsAbstractConstraintStream#buildRule(DroolsConstraint, Global)})
     * would call this first, establishing the rule ID with which to insert all the {@link DroolsLogicalTuple}s.
     * The later creation of the condition during {@link DroolsAbstractUniConstraintStream#createCondition()} would
     * just retrieve the value that already exists and query for all the {@link DroolsLogicalTuple}s having this value.
     * Even if the order of operations were switched (condition created before rule built), the ID would still be
     * correctly shared though.
     * @param constraintFactory never null, used to generate rule IDs
     * @return unique id for the rule
     */
    private int createRuleIdIfAbsent(DroolsConstraintFactory<Solution_> constraintFactory) {
        if (ruleId < 0) {
            ruleId = constraintFactory.getNextRuleId();
        }
        return ruleId;
    }

    @Override
    public String toString() {
        return "GroupBy()";
    }
}
