package com.redhat.installer.installation.maven.validator;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.installer.DataValidator;
import com.izforge.izpack.util.IoHelper;
import com.izforge.izpack.util.VariableSubstitutor;
import com.redhat.installer.installation.validator.CharacterPathValidator;

import java.io.File;

/**
 * Created by thauser on 5/1/15.
 */
public class MavenSettingsValidator implements DataValidator {
    AutomatedInstallData idata;
    String errorMessageId;
    String warningMessageId;
    String formattedMessage;

    @Override
    public Status validateData(AutomatedInstallData adata) {
        idata = adata;
        return setMavenSettingsPath();
    }

    private Status setMavenSettingsPath() {
        VariableSubstitutor vs = new VariableSubstitutor(idata.getVariables());
        boolean useDefault = Boolean.parseBoolean(idata.getVariable("useDefaultMavenSettingsLocation"));
        if (useDefault){
            idata.setVariable("MAVEN_SETTINGS_FULLPATH", vs.substitute(idata.getVariable("MAVEN_SETTINGS_FULLPATH.default")));
            return Status.OK;
        } else {
            String settingsPath = idata.getVariable("MAVEN_SETTINGS_FULLPATH");
            if (settingsPath.isEmpty()){
                setErrorAndMessage("MavenRepoCheckPanel.settings.empty");
                return Status.ERROR;
            }
            File settingsFile = new File(settingsPath);

            if (settingsFile.getPath().toLowerCase().startsWith("http")){
                setErrorAndMessage("MavenRepoCheckPanel.settings.invalid");
                return Status.ERROR;
            }

            String quickstartPath = idata.getVariable("install.quickstarts.path");
            if(settingsPath.equals(quickstartPath)) {
                settingsFile = new File(quickstartPath, "settings.xml");

            }

            if (settingsFile.isDirectory()) {
                settingsFile = new File(settingsPath, "settings.xml");
                idata.setVariable("MAVEN_SETTINGS_FULLPATH", settingsFile.getAbsolutePath());
            }

            if (!settingsFile.exists()){
                return createExampleSettings(settingsFile);
            }

            if (settingsFile.length() == 0) {
                setWarningAndMessage("MavenRepoCheckPanel.settings.empty.overwrite", settingsFile.getAbsolutePath());
                return Status.WARNING;
            }

            setWarningAndMessage("MavenRepoCheckPanel.settings.update.warning", settingsFile.getAbsolutePath());
            return Status.WARNING;
        }
    }

    private Status createExampleSettings(File settingsFile) {
        File existingParent = IoHelper.existingParent(settingsFile);

        if (existingParent == null){
            setErrorAndMessage("MavenRepoCheckPanel.drive.error",settingsFile.getAbsolutePath().substring(0,2));
            return Status.ERROR;
        } else if (!(existingParent.canWrite() && existingParent.canExecute())){
            setErrorAndMessage("MavenRepoCheckPanel.dir.error", existingParent.getAbsolutePath());
            return Status.ERROR;
        } else if (!CharacterPathValidator.validate(new File(existingParent.getPath()))){
            setErrorAndMessage("CharacterPathValidator.invalid", CharacterPathValidator.getInvalidCharacters());
            return Status.ERROR;
        }
        setWarningAndMessage("MavenRepoCheckPanel.settings.error",settingsFile.getAbsolutePath());
        idata.setVariable("MAVEN_SETTINGS_FULLPATH", settingsFile.getAbsolutePath());
        return Status.WARNING;
    }

    private void setWarningAndMessage(String id, String ... format){
        setWarningMessageId(id);
        setFormattedMessage(String.format(idata.langpack.getString(getWarningMessageId()),format));
    }

    private void setErrorAndMessage(String id, String ... format){
        setErrorMessageId(id);
        setFormattedMessage(String.format(idata.langpack.getString(getErrorMessageId()),format));
    }

    private void setErrorMessageId(String e){
        errorMessageId = e;
    }

    @Override
    public String getErrorMessageId() {
        return errorMessageId;
    }

    private void setWarningMessageId(String w){
        warningMessageId = w;
    }

    @Override
    public String getWarningMessageId() {
        return warningMessageId;
    }

    @Override
    public boolean getDefaultAnswer() {
        return true;
    }

    private void setFormattedMessage(String m){
        formattedMessage = m;
    }

    @Override
    public String getFormattedMessage() {
        return formattedMessage;
    }
}
