/*
 *      File/path utility implemented with Win32 API.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: filepath.h 328 2007-02-10 17:50:11Z nyaochi $ */

#ifndef	__FILEPATHUTIL_H__
#define	__FILEPATHUTIL_H__

#include <time.h>
#include <pmplib/ucs2char.h>

#ifdef	FILEPATH_EXPORTS
#define	FILEPATHAPI	__declspec(dllexport)
#else
#define	FILEPATHAPI
#endif


#ifdef	__cplusplus
extern "C" {
#endif/*__cplusplus*/

/** 
 * \addtogroup filepath File/Path Manupulation API
 * @{
 *
 *	The PMPlib File/Path Manupulation API is a utility for operations on file
 *	systems and string manupulation for file/path names. This API encapsulates
 *	the difference between Win32 and POSIX APIs, UCS-2 string and multi-byte
 *	characters, and path characters ('\\' or '/').
 */

/** 
 * \addtogroup filepath_string String manupulation for file and path names
 * @{
 *
 *	String manupulation routines manage strings so that they are grammartical
 *	for path and file names.
 */

/**
 * Add a (back)slash character to the end of a string.
 *
 *	This function adds a (back)slash character to the end of the string if
 *	the resultant string is grammertical. If the string already has a trailing
 *	(back)slash, this function will not change it.
 *
 *	@param	path		The pointer to the string.
 *	@retval	ucs2char_t*	Identical value to \a path argument.
 */
FILEPATHAPI ucs2char_t* filepath_addslash(ucs2char_t* path);

/**
 * Remove a (back)slash character from the end of a string.
 *
 *	This function removes a (back)slash character from the end of the string if
 *	the resultant string is grammertical. If the string already has no trailing
 *	(back)slash, this function will not change it.
 *
 *	@param	path		The pointer to the string.
 *	@retval	ucs2char_t*	The identical value to \a path argument.
 */
FILEPATHAPI ucs2char_t* filepath_removeslash(ucs2char_t* path);

/**
 * Concatenate two strings into one well-formed path.
 *
 *	@param	dst			The pointer to a buffer to receive the resultant path.
 *	@param	size		The size, in number of UCS-2 characters, of \a dst.
 *	@param	path		The pointer to a path name.
 *	@param	file		The pointer to a file name.
 *	@retval	const ucs2char_t*	The identical value to \a dst argument.
 */
FILEPATHAPI const ucs2char_t* filepath_combinepath(ucs2char_t* dst, size_t size, const ucs2char_t* path, const ucs2char_t* file);

/**
 * Remove the path portion of a filepath name.
 *
 *	@param	path		The pointer to the filepath name from which this
 *						function removes the path portion. The resultant string
 *						will contain only the file portion.
 */
FILEPATHAPI void filepath_strippath(ucs2char_t* path);

/**
 * Skip the path portion in a filepath name.
 *
 *	@param	path		The pointer to the filepath name from which this
 *						function skips the path portion. The skipped string
 *						will contain only the file portion.
 *	@retval	const ucs2char_t*	The pointer at which the file portion begins.
 */
FILEPATHAPI const ucs2char_t* filepath_skippath(const ucs2char_t* path);

/**
 * Remove the file portion from a string.
 *
 *	@param	path		The pointer to the string from which the file portion
 *						is removed.
 */
FILEPATHAPI void filepath_remove_filespec(ucs2char_t* path);

/**
 * Skip a portion in a filepath name.
 *
 *	@param	path		The pointer to the filepath name from which this
 *						function skips the portion.
 *	@param	root		The pointer to the string representing the portion to
 *						be skipped.
 *	@retval	const ucs2char_t*	The pointer at the next character after the
 *						portion \a root ends in \a path.
 */
FILEPATHAPI const ucs2char_t* filepath_skiproot(const ucs2char_t* path, const ucs2char_t* root);

/**
 * Skip a portion in filepath name, assuming it as the root directory.
 *
 *	The resultant string will begin with a path character.
 *
 *	@param	path		The pointer to the filepath name from which this
 *						function skips the portion.
 *	@param	root		The pointer to the string representing the portion to
 *						be skipped.
 *	@retval	const ucs2char_t*	The pointer at the next character after the
 *						portion \a root ends in \a path.
 */
FILEPATHAPI const ucs2char_t* filepath_changeroot(const ucs2char_t* path, const ucs2char_t* root);

/**
 * Skip a directory name in filepath name.
 *
 *	@param	path		The pointer to the filepath name from which this
 *						function skips a directory name.
 *	@retval	const ucs2char_t*	The pointer at the next character after the
 *						directory name ends in \a path.
 */
FILEPATHAPI const ucs2char_t* filepath_skipadirectory(const ucs2char_t* path);

/**
 * Add a file extension to the end of a string.
 *
 *	@param	path		The pointer to the string.
 *	@param	ext			The pointer to the file extension.
 */
FILEPATHAPI void filepath_add_extension(ucs2char_t* path, const ucs2char_t* ext);

/**
 * Remove a file extension from the end of a string.
 *
 *	@param	path		The pointer to the string from which the extension
 *						is removed.
 */
FILEPATHAPI void filepath_remove_extension(ucs2char_t* path);

/**
 * Convert a relative path to its corresponding absolute path.
 *
 *	@param	absolute	The pointer to a buffer to receive the resultant path.
 *	@param	base		The name of the base directory where the relative path
 *						begins.
 *	@param	relative	The relative path to be converted.
 *	@retval	int			Reserved (0).
 */
FILEPATHAPI int filepath_relative_to_absolute(ucs2char_t* absolute, const ucs2char_t* base, const ucs2char_t* relative);

/**
 * Canonicalize a path.
 *
 *	@param	dst			The pointer to a buffer to receive the resultant path.
 *	@param	src			The path to be canonicalized.
 *	@retval				Non-zero value if successful, zero otherwise.
 */
FILEPATHAPI int filepath_path_canonicalize(ucs2char_t* dst, const ucs2char_t* src);

/** 
 * Replaces all path separators with a backslash ('\') separator.
 * 
 *	This function replaces all path separators in the given path with
 *	Windows-style path separators (back-slashes). On POSIX systems, this means
 *	converting forward slash ('/') path separators into backslashes. On
 *	Windows, this is a no-op. The \p path is modified in place.
 *
 *	@param	path		The pointer to the path string to be converted.
 *	@return	int			Reserved.
 */
FILEPATHAPI int filepath_encode(ucs2char_t* path);

/** 
 * Replaces '\'-slashes in a path name with the native path separator.
 * 
 *	This function replaces all Windows-style path separators
 *	(back-slashes) in the given path with the native path separator. On
 *	POSIX systems, the replacement character is a forward slash ('/'). On
 *	Windows, this is a no-op. The \p path is modified in place.
 *
 *	@param	path		The pointer to the path string to be converted.
 *	@return	int			Reserved.
 */
FILEPATHAPI int filepath_decode(ucs2char_t* path);

/** @} */

/** 
 * \addtogroup filepath_filedir File and directory manupulation
 * @{
 *
 *	File and directory manupulation routines provide a unified API to
 *	manupulate a file system, encapsulating the difference between Win32 and
 *	POSIX APIs.
 */

/**
 * Prototype for the callback function for receiving found files.
 *
 *	@param	instance	The instance value.
 *	@param	found_path	The path-name portion of the found file.
 *	@param	found_file	The file-name portion of the found file.
 *	@retval	int			Reserved.
 */
typedef int (*filepath_findfile_callback)(void *instance, const ucs2char_t* found_path, const ucs2char_t* found_file);

/**
 * Find files.
 *
 *	@param	path		The directory in which this function starts the search.
 *	@param	recursive	Non-zero value to search files recursively under the
 *						directory \p path.
 *	@param	callback	The pointer to the callback function to receive the
 *						names of found files.
 *	@param	instance	A user-defined instance value that will be sent to the
 *						callback function.
 *	@retval	int			Reserved.
 */
FILEPATHAPI int find_file(const ucs2char_t* path, int recursive, filepath_findfile_callback callback, void *instance);

/**
 * Test the existence of a file.
 *
 *	@param	path		The filepath name to be tested.
 *	@retval	int			True or false.
 */
FILEPATHAPI int filepath_file_exists(const ucs2char_t* path);

/**
 * Test a file name has an extension.
 *
 *	@param	filename	The file name to be tested.
 *	@param	ext			The extension.
 *	@retval	int			True or false.
 */
FILEPATHAPI int filepath_hasext(const ucs2char_t* filename, const ucs2char_t* ext);

/**
 * Test if a filepath name represents a directory.
 *
 *	@param	path		The filepath name to be tested.
 *	@retval	int			True or false.
 */
FILEPATHAPI int filepath_is_dir(const ucs2char_t *path);

/**
 * Test if a filepath name represents a relative path.
 *
 *	@param	path		The filepath name to be tested.
 *	@retval	int			True or false.
 */
FILEPATHAPI int filepath_is_relative(const ucs2char_t* path);

/**
 * Test if two path names has a common prefix of a directory.
 *
 *	@param	path1		A filepath name.
 *	@param	path2		Another filepath name.
 *	@retval	int			True or false.
 */
FILEPATHAPI int filepath_is_same_root(const ucs2char_t* path1, const ucs2char_t* path2);

/**
 * Obtain the last-modified time of a file.
 *
 *	@param	filename	The filepath name.
 *	@retval	time_t		The time stamp when the file was updated most recently.
 */
FILEPATHAPI time_t filepath_mtime(const ucs2char_t *filename);

/** 
 * Compares the modification times of two files.
 * 
 *	This function compares the modification times of @p file1 and @p file2
 *	and returns the difference between them is returned.
 *	
 *	@param	file1		The name of file #1.
 *	@param	file2		The name of file #2.
 *	@retval int			The comparison result:
 *		- < 0, if file2 was modified more recently than file1;
 *		- = 0, if the modification times are the same;
 *		- > 0, if file1 was modified more recently than file2.
 */
FILEPATHAPI int filepath_compare_lastupdate(const ucs2char_t* file1, const ucs2char_t* file2);

/**
 * Obtain the size of a file.
 *
 *	@param	filename	The filepath name.
 *	@retval	uint32_t	The size, in bytes, of the file.
 */
FILEPATHAPI uint32_t filepath_size(const ucs2char_t *filename);

/** 
 * Remove a file.
 * 
 *	@param	file		The name of a file to be removed.
 *	@retval	int			True if succeeded, false otherwise.
 */
FILEPATHAPI int filepath_removefile(const ucs2char_t* file);

/** 
 * Copy a file.
 * 
 *	@param	src			The name of a source file to be copied.
 *	@param	dst			The name of a destination to be created.
 *	@retval	int			True if succeeded, false otherwise.
 */
FILEPATHAPI int filepath_copyfile(const ucs2char_t* src, const ucs2char_t* dst);

/** @} */


/** @} */

#ifdef	__cplusplus
}
#endif/*__cplusplus*/

#endif/*__FILEPATHUTIL_H__*/
