// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/Beam.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief Add a short analysis description here
  class TOPAZ_1989_I279575 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(TOPAZ_1989_I279575);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {


      // Initialise and register projections
      declare(FinalState(), "FS");

      // Book histograms
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal)) _sqs = en;
        bool offset(en == "59500"s);
        book(_h[en+"eec"],   1+offset, 1, 1);
        book(_h[en+"eecpi"], 1+offset, 1, 2);
        book(_h[en+"aeec"],  1+offset, 1, 3);
        book(_c[en], "TMP/weightSum_"+en);
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // First, veto on leptonic events by requiring at least 4 charged FS particles
      const FinalState& fs = apply<FinalState>(event, "FS");
      // Even if we only generate hadronic events, we still need a cut on numCharged >= 2.
      if ( fs.particles().size() < 2)  vetoEvent;
      _c[_sqs]->fill();

      const double Evis2 = sqr(sum(fs.particles(), Kin::E, 0.0));
      // (A)EEC
      // Need iterators since second loop starts at current outer loop iterator, i.e. no "foreach" here!
      for (Particles::const_iterator p_i = fs.particles().begin(); p_i != fs.particles().end(); ++p_i) {
        for (Particles::const_iterator p_j = p_i; p_j != fs.particles().end(); ++p_j) {
          const Vector3 mom3_i = p_i->mom().p3();
          const Vector3 mom3_j = p_j->mom().p3();
          const double energy_i = p_i->mom().E();
          const double energy_j = p_j->mom().E();
          const double thetaij = mom3_i.unit().angle(mom3_j.unit())/M_PI*180.;
          double eec = (energy_i*energy_j) / Evis2;
          if (p_i != p_j) eec *= 2.;
          if (thetaij < 90.) {
            _h[_sqs+"eec"]->fill(thetaij, eec);
            _h[_sqs+"aeec"]->fill(thetaij, -eec);
          }
          else {
            _h[_sqs+"eecpi"]->fill(180.-thetaij, eec);
            _h[_sqs+"aeec"]->fill(180.-thetaij, eec);
          }
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_c, crossSectionPerEvent());
      scale(_h, crossSectionPerEvent());
      for (const auto& item : _c) {
        if (isZero(item.second->sumW())) continue;
        scale(_h[item.first+"eec"],   180.0/M_PI/item.second->sumW());
        scale(_h[item.first+"eecpi"], 180.0/M_PI/item.second->sumW());
        scale(_h[item.first+"aeec"],   180.0/M_PI/item.second->sumW());
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    map<string,CounterPtr> _c;
    string _sqs = "";
    /// @}

  };


  RIVET_DECLARE_PLUGIN(TOPAZ_1989_I279575);
}
