//
// Syd: rock-solid application kernel
// src/fs.rs: Filesystem utilities
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
// Based in part upon uutils coreutils package's src/lib/features/fs.rs which is:
//   (c) Joseph Crail <jbcrail@gmail.com>
//   (c) Jian Zeng <anonymousknight96 AT gmail.com>
// Tests base based in part upon gnulib packages' tests/test-canonicalize.c which is:
//   (c) Free Software Foundation, Inc.
// {chdir,getdir}_long() functions are based in part upon zsh/Src/compat.c which is:
//   (c) 1992-1997 Paul Falstad
//   SPDX-License-Identifier: ZSH
//
// SPDX-License-Identifier: GPL-3.0

//! Set of functions to manage files and symlinks

use std::{
    borrow::Cow,
    collections::VecDeque,
    ffi::CStr,
    fmt,
    fs::{metadata, set_permissions, File},
    hash::Hash,
    io::{Read, Seek, SeekFrom, Write},
    ops::{Deref, RangeInclusive},
    os::{
        fd::{AsFd, AsRawFd, BorrowedFd, FromRawFd, OwnedFd, RawFd},
        unix::{ffi::OsStrExt, fs::PermissionsExt},
    },
    path::Path,
    sync::LazyLock,
};

use ahash::HashMapExt;
use bitflags::bitflags;
use btoi::btoi;
use libc::{
    c_char, c_int, c_long, c_uint, c_ulong, c_void, clone, dup3, flock, off64_t, openat,
    seccomp_notif, seccomp_notif_addfd, seccomp_notif_resp, siginfo_t, size_t, strncmp, syscall,
    SYS_close_range, SYS_execveat, SYS_faccessat2, SYS_ioctl, SYS_kcmp, SYS_pidfd_getfd,
    SYS_pidfd_open, SYS_pidfd_send_signal, SYS_tgkill, AT_EMPTY_PATH, AT_SYMLINK_NOFOLLOW,
    CLONE_PIDFD, EBADF, F_RDLCK, F_UNLCK, F_WRLCK, GRND_RANDOM, LOCK_EX, LOCK_NB, LOCK_SH, LOCK_UN,
    MFD_HUGETLB, MFD_HUGE_16GB, MFD_HUGE_16MB, MFD_HUGE_1GB, MFD_HUGE_1MB, MFD_HUGE_256MB,
    MFD_HUGE_2GB, MFD_HUGE_2MB, MFD_HUGE_32MB, MFD_HUGE_512MB, MFD_HUGE_8MB, O_NONBLOCK, SEEK_SET,
    S_IROTH, S_ISVTX, S_IWGRP, S_IWOTH, _IO, _IOR, _IOW, _IOWR,
};
use libseccomp::{ScmpFilterContext, ScmpSyscall};
use nix::{
    errno::Errno,
    fcntl::{
        fcntl, openat2, AtFlags, FallocateFlags, FcntlArg, FdFlag, OFlag, OpenHow, ResolveFlag,
        SealFlag, AT_FDCWD,
    },
    libc::{
        mode_t, DT_BLK, DT_CHR, DT_DIR, DT_FIFO, DT_LNK, DT_REG, DT_SOCK, S_IFBLK, S_IFCHR,
        S_IFDIR, S_IFIFO, S_IFLNK, S_IFMT, S_IFREG, S_IFSOCK,
    },
    pty::Winsize,
    sched::{CloneCb, CloneFlags},
    sys::{
        inotify::AddWatchFlags,
        resource::{getrlimit, Resource},
        signal::{SigSet, Signal},
        socket::{
            getsockopt,
            sockopt::{PeerCredentials, ReceiveTimeout, SendTimeout},
            UnixCredentials,
        },
        stat::Mode,
        statfs::fstatfs,
        time::TimeSpec,
    },
    unistd::{close, write, AccessFlags, Pid, UnlinkatFlags},
    NixPath,
};
use serde::{ser::SerializeMap, Serialize, Serializer};

use crate::{
    compat::{
        fstatx, getdents64, statx, FileStatx, STATX_BASIC_STATS, STATX_INO, STATX_MNT_ID,
        STATX_MNT_ID_UNIQUE, STATX_MODE, STATX_NLINK, STATX_SIZE, STATX_TYPE,
    },
    config::*,
    confine::CLONE_NEWTIME,
    cookie::{
        safe_memfd_create, safe_openat2, safe_unlinkat, SECCOMP_IOCTL_NOTIF_ADDFD_COOKIE_ARG3,
        SECCOMP_IOCTL_NOTIF_ADDFD_COOKIE_ARG4, SECCOMP_IOCTL_NOTIF_ADDFD_COOKIE_ARG5,
        SECCOMP_IOCTL_NOTIF_SEND_COOKIE_ARG3, SECCOMP_IOCTL_NOTIF_SEND_COOKIE_ARG4,
        SECCOMP_IOCTL_NOTIF_SEND_COOKIE_ARG5,
    },
    err::err2no,
    error,
    hash::SydHashMap,
    ioctl::{TIOCEXCL, TIOCGEXCL, TIOCGWINSZ, TIOCNXCL, TIOCSWINSZ},
    kernel::sandbox_path,
    magic::ProcMagic,
    path::{dotdot_with_nul, XPath, XPathBuf, PATH_MAX},
    proc::{proc_tgid, PROCMAP_QUERY},
    sandbox::{Capability, Sandbox},
};

/// SAFETY: AT_BADFD to be used a safe alternative to AT_FDCWD.
pub const AT_BADFD: BorrowedFd<'static> = unsafe { BorrowedFd::borrow_raw(-EBADF) };

/// MFD_CLOEXEC memfd_create(2) flag.
pub const MFD_CLOEXEC: c_uint = libc::MFD_CLOEXEC;
/// MFD_ALLOW_SEALING memfd_create(2) flag.
pub const MFD_ALLOW_SEALING: c_uint = libc::MFD_ALLOW_SEALING;
/// MFD_NOEXEC_SEAL memfd_create(2) flag.
pub const MFD_NOEXEC_SEAL: c_uint = libc::MFD_NOEXEC_SEAL;
/// MFD_EXEC memfd_create(2) flag.
pub const MFD_EXEC: c_uint = libc::MFD_EXEC;

// nix' MfdFlags does not include MFD_{EXEC,NOEXEC_SEAL} yet!
bitflags! {
    /// Flags for _memfd_create_(2)
    pub struct MfdFlags: c_uint {
        /// Close-on-exec
        const MFD_CLOEXEC = MFD_CLOEXEC;

        /// Allow sealing via _fcntl_(2).
        const MFD_ALLOW_SEALING = MFD_ALLOW_SEALING;

        /// Disallow exec (Linux >= 6.3).
        const MFD_NOEXEC_SEAL = MFD_NOEXEC_SEAL;

        /// Allow exec (Linux >= 6.3).
        const MFD_EXEC = MFD_EXEC;

        /// Use hugetlbfs.
        const MFD_HUGETLB = MFD_HUGETLB;

        /// Huge page size: 1MB.
        const MFD_HUGE_1MB = MFD_HUGE_1MB;

        /// Huge page size: 2MB.
        const MFD_HUGE_2MB = MFD_HUGE_2MB;

        /// Huge page size: 8MB.
        const MFD_HUGE_8MB = MFD_HUGE_8MB;

        /// Huge page size: 16MB.
        const MFD_HUGE_16MB = MFD_HUGE_16MB;

        /// Huge page size: 32MB.
        const MFD_HUGE_32MB = MFD_HUGE_32MB;

        /// Huge page size: 256MB.
        const MFD_HUGE_256MB = MFD_HUGE_256MB;

        /// Huge page size: 512MB.
        const MFD_HUGE_512MB = MFD_HUGE_512MB;

        /// Huge page size: 1GB.
        const MFD_HUGE_1GB = MFD_HUGE_1GB;

        /// Huge page size: 2GB.
        const MFD_HUGE_2GB = MFD_HUGE_2GB;

        /// Huge page size: 16GB.
        const MFD_HUGE_16GB = MFD_HUGE_16GB;
    }
}

/// Information to uniquely identify a file.
#[derive(Copy, Clone, Debug, Eq, PartialEq, Hash, Ord, PartialOrd)]
pub struct FileInfo {
    /// Inode number
    pub ino: u64,
    /// Mount id
    pub mnt: u64,
}

impl FileInfo {
    /// Get information for the current working directory.
    pub fn from_cwd() -> Result<Self, Errno> {
        statx(AT_FDCWD, XPath::empty(), AT_EMPTY_PATH, Self::mask()).map(Self::from_statx)
    }

    /// Get information for a given file descriptor.
    pub fn from_fd<Fd: AsFd>(fd: Fd) -> Result<Self, Errno> {
        fstatx(fd, Self::mask()).map(Self::from_statx)
    }

    /// Convert statx information to FileInfo.
    ///
    /// The statx(2) call must have been called with STATX_INO|STATX_MNT_ID.
    pub fn from_statx(stx: FileStatx) -> Self {
        Self {
            ino: stx.stx_ino,
            mnt: stx.stx_mnt_id,
        }
    }

    /// Return masks for statx(2) required to attain file information.
    #[inline]
    pub fn mask() -> c_uint {
        let mut mask = STATX_INO;
        mask |= if *HAVE_STATX_MNT_ID_UNIQUE {
            STATX_MNT_ID_UNIQUE
        } else {
            STATX_MNT_ID
        };
        mask
    }
}

bitflags! {
    /// Path canonicalization options
    #[derive(Clone, Copy, Debug, Eq, PartialEq)]
    pub struct FsFlags: u16 {
        /// All components of the path must exist.
        /// This is the default.
        /// Without this option, the behaviour is:
        /// Last component may exist, other components must exist.
        /// Conflicts with the option `MISS_LAST`.
        const MUST_PATH = 1 << 0;
        /// Last component must not exist, other components must exist.
        /// Without this option, the default is:
        /// Last component may exist, other components must exist.
        /// Conflicts with the option `MUST_PATH`.
        const MISS_LAST = 1 << 1;
        /// Do not follow symbolic links for the last path component.
        /// Symbolic links for parent components will be resolved.
        const NO_FOLLOW_LAST = 1 << 2;
        /// Do not permit the path resolution to succeed if any
        /// component of the resolution is not a descendant of the
        /// directory indicated by dirfd. This causes absolute symbolic
        /// links (and absolute values of pathname) to be rejected.
        /// Currently, this flag also disables magic-link resolution
        /// (see below). However, this may change in the future.
        /// Therefore, to ensure that magic links are not resolved, the
        /// caller should explicitly specify RESOLVE_NO_MAGICLINKS.
        const RESOLVE_BENEATH = 1 << 3;
        /// Do not resolve symbolic links for any of the path components.
        const NO_RESOLVE_PATH = 1 << 4;
        /// Do not resolve proc magic symbolic links.
        const NO_RESOLVE_PROC = 1 << 5;
        /// Do not traverse through mount points.
        const NO_RESOLVE_XDEV = 1 << 6;
        /// Do not traverse through `..` components.
        const NO_RESOLVE_DOTDOT = 1 << 7;
        /// The return file descriptor should be split to (parent, base),
        /// and a fd to the owning directory should be opened instead.
        ///
        /// Default is to use an `O_PATH` file descriptor directly to the file.
        /// Conflicts with the option `WANT_READ`.
        const WANT_BASE = 1 << 8;
        ///
        /// The return file descriptor should be made read-only.
        ///
        /// Default is to use an `O_PATH` file descriptor.
        /// Conflicts with the option `WANT_BASE`.
        const WANT_READ = 1 << 9;
    }
}

impl Default for FsFlags {
    fn default() -> Self {
        // All components of the path must exist.
        // This is the default.
        Self::MUST_PATH
    }
}

// Controls how missing components should be handled when canonicalizing a path.
#[derive(Clone, Copy, Debug, Eq, PartialEq)]
enum MissingHandling {
    // Last component may exist, other components must exist.
    Normal,

    // All components must exist.
    // Maps to FsFlags::MUST_PATH.
    Existing,

    // Last component must not exist, other components must exist.
    // Maps to FsFlags::MISS_LAST.
    Missing,
}

impl From<FsFlags> for MissingHandling {
    fn from(flag: FsFlags) -> Self {
        if flag.contains(FsFlags::MUST_PATH) {
            Self::Existing
        } else if flag.contains(FsFlags::MISS_LAST) {
            Self::Missing
        } else {
            Self::Normal
        }
    }
}

impl FsFlags {
    /// Return `Errno` corresponding to magic link restrictions.
    pub fn magic_errno(&self) -> Errno {
        if self.intersects(Self::RESOLVE_BENEATH | Self::NO_RESOLVE_XDEV) {
            Errno::EXDEV
        } else if self.intersects(Self::NO_RESOLVE_PATH | Self::NO_RESOLVE_PROC) {
            Errno::ELOOP
        } else {
            Errno::EACCES
        }
    }

    /// Return true if symbolic links in last path component should be followed.
    pub fn follow_last(&self) -> bool {
        !self.contains(Self::NO_FOLLOW_LAST)
    }

    /// Return true if symbolic links in path should be resolved.
    pub fn resolve_path(&self) -> bool {
        !self.contains(Self::NO_RESOLVE_PATH)
    }

    /// Return true if magic /proc symbolic links should be resolved.
    pub fn resolve_proc(&self) -> bool {
        !self.intersects(Self::NO_RESOLVE_PROC | Self::NO_RESOLVE_XDEV | Self::RESOLVE_BENEATH)
    }

    /// Return true if the path must exist.
    pub fn must_exist(&self) -> bool {
        self.contains(Self::MUST_PATH)
    }

    /// Return true if the file may be missing.
    pub fn missing(&self) -> bool {
        self.contains(Self::MISS_LAST)
    }
}

impl Serialize for FsFlags {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        let mut flags: Vec<&str> = vec![];

        if self.is_empty() {
            return serializer.collect_seq(flags);
        }

        if self.contains(Self::RESOLVE_BENEATH) {
            flags.push("resolve-beneath");
        }
        if self.contains(Self::NO_RESOLVE_PATH) {
            flags.push("resolve-no-symlinks");
        }
        if self.contains(Self::NO_RESOLVE_PROC) {
            flags.push("resolve-no-magiclinks");
        }
        if self.contains(Self::NO_RESOLVE_XDEV) {
            flags.push("resolve-no-xdev");
        }
        if self.contains(Self::NO_RESOLVE_DOTDOT) {
            flags.push("resolve-no-dotdot");
        }
        if self.contains(Self::MUST_PATH) {
            flags.push("must-path");
        }
        if self.contains(Self::MISS_LAST) {
            flags.push("miss-last");
        }
        if self.contains(Self::NO_FOLLOW_LAST) {
            flags.push("no-follow");
        }
        if self.contains(Self::WANT_BASE) {
            flags.push("want-base");
        }
        if self.contains(Self::WANT_READ) {
            flags.push("want-read");
        }

        serializer.collect_seq(flags)
    }
}

/// This function creates a flock(2) lock.
///
/// This function does NOT retry the system call on EINTR.
pub fn flock_fd<Fd: AsFd>(fd: Fd, exclusive: bool, wait: bool) -> Result<(), Errno> {
    let mut op = if exclusive { LOCK_EX } else { LOCK_SH };
    if !wait {
        op |= LOCK_NB;
    }

    // SAFETY: nix deprecated direct flock,
    // and we cannot use the type-safe interface here.
    Errno::result(unsafe { flock(fd.as_fd().as_raw_fd(), op) }).map(drop)
}

/// This function unlocks a flock(2) lock.
///
/// This function retries the system call on EINTR.
pub fn funlock_fd<Fd: AsFd>(fd: Fd) -> Result<(), Errno> {
    // SAFETY: nix deprecated direct flock,
    // and we cannot use the type-safe interface here.
    retry_on_eintr(|| Errno::result(unsafe { flock(fd.as_fd().as_raw_fd(), LOCK_UN) })).map(drop)
}

/// This function creates an Open File Description (OFD) lock.
///
/// This function does NOT retry the system call on EINTR.
#[allow(clippy::cast_possible_truncation)]
pub fn lock_fd<Fd: AsFd>(fd: Fd, write: bool, wait: bool) -> Result<(), Errno> {
    // SAFETY: struct flock has reserved fields on some architectures such as mipsel.
    let mut lock: flock = unsafe { std::mem::zeroed() };

    lock.l_type = if write { F_WRLCK } else { F_RDLCK } as i16;
    lock.l_whence = SEEK_SET as i16;
    lock.l_len = 1; // Lock the first byte.
                    // No need to set l_start and l_pid to zero.

    fcntl(
        &fd,
        if wait {
            FcntlArg::F_OFD_SETLKW(&lock)
        } else {
            FcntlArg::F_OFD_SETLK(&lock)
        },
    )
    .map(drop)
}

/// This function releases an Open File Description (OFD) lock.
///
/// This function retries the system call on EINTR.
#[allow(clippy::cast_possible_truncation)]
pub fn unlock_fd<Fd: AsFd>(fd: Fd) -> Result<(), Errno> {
    // SAFETY: struct flock has reserved fields on some architectures such as mipsel.
    let mut lock: flock = unsafe { std::mem::zeroed() };

    lock.l_type = F_UNLCK as i16;
    lock.l_whence = SEEK_SET as i16;
    lock.l_len = 1; // Lock the first byte.
                    // No need to set l_start and l_pid to zero.

    retry_on_eintr(|| fcntl(&fd, FcntlArg::F_OFD_SETLK(&lock))).map(drop)
}

/// Retries a closure on `EAGAIN` and `EINTR` errors.
///
/// This function will call the provided closure, and if the closure
/// returns `EAGAIN` or `EINTR` error, it will retry the operation until it
/// succeeds or fails with a different error.
#[inline]
pub fn retry_on_intr<F, T>(mut f: F) -> Result<T, Errno>
where
    F: FnMut() -> Result<T, Errno>,
{
    loop {
        match f() {
            Err(Errno::EAGAIN | Errno::EINTR) => continue,
            result => return result,
        }
    }
}

/// Retries a closure on `EINTR` errors.
///
/// This function will call the provided closure, and if the closure
/// returns `EINTR` error, it will retry the operation until it
/// succeeds or fails with a different error.
#[inline]
pub fn retry_on_eintr<F, T>(mut f: F) -> Result<T, Errno>
where
    F: FnMut() -> Result<T, Errno>,
{
    loop {
        match f() {
            Err(Errno::EINTR) => continue,
            result => return result,
        }
    }
}

/// Retries a closure on `EAGAIN` errors.
///
/// This function will call the provided closure, and if the closure
/// returns `EAGAIN` error, it will retry the operation until it
/// succeeds or fails with a different error.
#[inline]
pub fn retry_on_eagain<F, T>(mut f: F) -> Result<T, Errno>
where
    F: FnMut() -> Result<T, Errno>,
{
    loop {
        match f() {
            Err(Errno::EAGAIN) => continue,
            result => return result,
        }
    }
}

/// write! which retries on EINTR.
#[macro_export]
macro_rules! rwrite {
    ($dst:expr, $($arg:tt)*) => {{
        // retry missing / interrupted syscalls, map any IO error to Errno
        $crate::fs::retry_on_eintr(|| {
            $dst.write_fmt(format_args!($($arg)*))
                .map_err(|err| $crate::err::err2no(&err))
        })
    }};
}

/// writeln! which retries on EINTR.
#[macro_export]
macro_rules! rwriteln {
    // allow `rwriteln!(dst, "foo")` and `rwriteln!(dst)` (just newline)
    ($dst:expr $(, $($arg:tt)*)?) => {{
        $crate::fs::retry_on_eintr(|| {
            // first write the formatted stuff (if any).
            let () = $dst
                .write_fmt(format_args!($($($arg)*)?))
                .map_err(|err| $crate::err::err2no(&err))?;
            // then write a newline byte.
            $dst
                .write_all(b"\n")
                .map_err(|err| $crate::err::err2no(&err))
        })
    }};
}

/// Creates an anonymous, non-executable file that lives in memory, and
/// return an owned file-descriptor to it.
pub fn create_memfd(name: &[u8], flags: c_uint) -> Result<OwnedFd, Errno> {
    // Ensure the name is at most 255 bytes and null-terminated.
    if name.len() > 255 {
        return Err(Errno::ENAMETOOLONG);
    }
    // SAFETY: ^^ asserts arithmetic below cannot overflow.
    #[allow(clippy::arithmetic_side_effects)]
    let c_name = if name.last() == Some(&0) {
        Cow::Borrowed(name)
    } else {
        let mut c_name = Vec::with_capacity(name.len() + 1);
        c_name.extend_from_slice(name);
        c_name.push(0);
        Cow::Owned(c_name)
    };

    match safe_memfd_create(&c_name, flags | MFD_CLOEXEC) {
        Ok(fd) => Ok(fd),
        Err(Errno::EINVAL) => {
            // 1. Flags included unknown bits.
            // 2. Name was too long.
            // 3. Both MFD_HUGETLB and MFD_ALLOW_SEALING were
            //    specified in flags.
            // In our case only the first one is relevant and
            // indicates MFD_NOEXEC_SEAL is unsupported. To avoid
            // potential confusion, we return EOPNOTSUPP rather
            // EINVAL than in this case.
            Err(Errno::EOPNOTSUPP)
        }
        Err(errno) => Err(errno),
    }
}

/// Seals the memfd for writing.
pub fn seal_memfd<Fd: AsFd>(fd: Fd) -> Result<(), Errno> {
    // Seal memory fd.
    fcntl(
        fd,
        FcntlArg::F_ADD_SEALS(
            SealFlag::F_SEAL_SEAL
                | SealFlag::F_SEAL_WRITE
                | SealFlag::F_SEAL_SHRINK
                | SealFlag::F_SEAL_GROW,
        ),
    )
    .map(drop)
}

/// Sets or clears the append (O_APPEND) flag on a file descriptor.
pub fn set_append<Fd: AsFd>(fd: Fd, state: bool) -> Result<(), Errno> {
    let flags = fcntl(&fd, FcntlArg::F_GETFL)?;

    let mut new_flags = flags;
    if state {
        new_flags |= OFlag::O_APPEND.bits();
    } else {
        new_flags &= !OFlag::O_APPEND.bits();
    }

    fcntl(&fd, FcntlArg::F_SETFL(OFlag::from_bits_truncate(new_flags))).map(drop)
}

/// Returns `true` if the given file descriptor is set to non-blocking mode.
pub fn get_nonblock<Fd: AsFd>(fd: Fd) -> Result<bool, Errno> {
    fcntl(fd, FcntlArg::F_GETFL).map(|flags| flags & O_NONBLOCK != 0)
}

/// Sets or clears the non-blocking (O_NONBLOCK) flag on a file descriptor.
pub fn set_nonblock<Fd: AsFd>(fd: Fd, state: bool) -> Result<(), Errno> {
    let flags = fcntl(&fd, FcntlArg::F_GETFL)?;

    let mut new_flags = flags;
    if state {
        new_flags |= OFlag::O_NONBLOCK.bits();
    } else {
        new_flags &= !OFlag::O_NONBLOCK.bits();
    }

    fcntl(&fd, FcntlArg::F_SETFL(OFlag::from_bits_truncate(new_flags))).map(drop)
}

/// Sets or clears the close-on-exec (FD_CLOEXEC) flag on a file descriptor.
pub fn set_cloexec<Fd: AsFd>(fd: Fd, state: bool) -> Result<(), Errno> {
    let flags = fcntl(&fd, FcntlArg::F_GETFD)?;

    let mut new_flags = flags;
    if state {
        new_flags |= FdFlag::FD_CLOEXEC.bits();
    } else {
        new_flags &= !FdFlag::FD_CLOEXEC.bits();
    }

    fcntl(
        &fd,
        FcntlArg::F_SETFD(FdFlag::from_bits_truncate(new_flags)),
    )
    .map(drop)
}

/// Set pipe max size of the given pipe.
pub fn set_pipemax<Fd: AsFd>(fd: Fd, size: c_int) -> Result<usize, Errno> {
    #[allow(clippy::cast_sign_loss)]
    fcntl(fd, FcntlArg::F_SETPIPE_SZ(size)).map(|r| r as usize)
}

/// Get exclusive mode for the given terminal.
pub fn get_exclusive<Fd: AsFd>(fd: Fd) -> Result<bool, Errno> {
    let mut set: c_int = 0;
    let fd = fd.as_fd().as_raw_fd();
    let req = TIOCGEXCL.ok_or(Errno::ENOTTY)?;

    // SAFETY: TIOCGEXCL takes an int* to return 0 or nonzero.
    Errno::result(unsafe { syscall(SYS_ioctl, fd, req, std::ptr::addr_of_mut!(set)) })
        .map(|_| set != 0)
}

/// Set given terminal to exclusive mode, or disable exclusive mode.
pub fn set_exclusive<Fd: AsFd>(fd: Fd, enable: bool) -> Result<(), Errno> {
    let fd = fd.as_fd().as_raw_fd();
    let req = if enable { *TIOCEXCL } else { *TIOCNXCL }.ok_or(Errno::ENOTTY)?;

    // SAFETY: TIOC{E,N}XCL take no extra arguments.
    Errno::result(unsafe { syscall(SYS_ioctl, fd, req) }).map(drop)
}

const KCMP_FILE: c_long = 0;

/// Check if the given file descriptor is open for the given process.
pub fn is_open_fd(pid: Pid, fd: RawFd) -> Result<bool, Errno> {
    #[allow(clippy::cast_lossless)]
    #[allow(clippy::cast_possible_wrap)]
    #[allow(clippy::cast_sign_loss)]
    // SAFETY: There's no libc wrapper for kcmp.
    match Errno::result(unsafe {
        syscall(
            SYS_kcmp,
            pid.as_raw() as c_long,
            pid.as_raw() as c_long,
            KCMP_FILE,
            fd as c_ulong as c_long,
            fd as c_ulong as c_long,
        )
    }) {
        Ok(_) => Ok(true),
        Err(Errno::EBADF) => Ok(false),
        Err(errno) => Err(errno),
    }
}

/// Check two fds point to the same open file description for the given processes.
pub fn is_same_fd(pid1: Pid, pid2: Pid, fd1: RawFd, fd2: RawFd) -> Result<bool, Errno> {
    if pid1 == pid2 && fd1 == fd2 {
        // We do not check for open/valid FD in this function,
        // so we short-circuit here for efficiency.
        return Ok(true);
    }

    // SAFETY: There's no libc wrapper for kcmp.
    #[allow(clippy::cast_lossless)]
    #[allow(clippy::cast_possible_wrap)]
    #[allow(clippy::cast_sign_loss)]
    Ok(Errno::result(unsafe {
        syscall(
            SYS_kcmp,
            pid1.as_raw() as c_long,
            pid2.as_raw() as c_long,
            KCMP_FILE,
            fd1 as c_ulong as c_long,
            fd2 as c_ulong as c_long,
        )
    })? == 0)
}

/// Check two processes share the same address space.
pub fn is_same_vm(pid1: Pid, pid2: Pid) -> Result<bool, Errno> {
    const KCMP_VM: u64 = 1;
    // SAFETY: There's no libc wrapper for kcmp.
    Ok(Errno::result(unsafe { syscall(SYS_kcmp, pid1.as_raw(), pid2.as_raw(), KCMP_VM) })? == 0)
}

/// Check if file resides on a hugetlbfs (e.g. memfds with MFD_HUGETLB)
#[allow(clippy::cast_sign_loss)]
pub fn is_huge_file<Fd: AsFd>(fd: Fd) -> Result<bool, Errno> {
    // HUGETLBFS_MAGIC is 0x958458f6 which may be c_long or c_uint.
    fstatfs(fd.as_fd()).map(|st| st.filesystem_type().0 as u64 == 0x958458f6)
}

/// Check if file is the /dev/null character device.
pub fn is_dev_null<Fd: AsFd>(fd: Fd) -> Result<bool, Errno> {
    #[allow(clippy::cast_possible_truncation)]
    const S_IFCHR: u16 = libc::S_IFCHR as u16;
    const DEV_NULL_MAJOR: u32 = 1;
    const DEV_NULL_MINOR: u32 = 3;

    let statx = fstatx(fd, STATX_BASIC_STATS)?;

    // Check if file is a character device,
    // and its device major/minor numbers
    // match that of /dev/null.
    Ok(statx.stx_mode & S_IFCHR == S_IFCHR
        && statx.stx_rdev_major == DEV_NULL_MAJOR
        && statx.stx_rdev_minor == DEV_NULL_MINOR)
}

/// Check if file is the AMD KFD character device (/dev/kfd).
pub fn is_dev_kfd<Fd: AsFd>(fd: Fd) -> Result<bool, Errno> {
    #[allow(clippy::cast_possible_truncation)]
    const S_IFCHR: u16 = libc::S_IFCHR as u16;
    const KFD_MAJOR: u32 = 238;
    const KFD_MINOR: u32 = 0;

    let statx = fstatx(fd, STATX_BASIC_STATS)?;

    // Character device with the /dev/kfd major/minor (238:0).
    Ok(statx.stx_mode & S_IFCHR == S_IFCHR
        && statx.stx_rdev_major == KFD_MAJOR
        && statx.stx_rdev_minor == KFD_MINOR)
}

/// Checks if the given file descriptor has a send timeout set.
pub fn has_send_timeout<F: AsFd>(fd: &F) -> Result<bool, Errno> {
    let tv = getsockopt(fd, SendTimeout)?;
    Ok(tv.tv_sec() != 0 || tv.tv_usec() != 0)
}

/// Checks if the given file descriptor has a receive timeout set.
pub fn has_recv_timeout<F: AsFd>(fd: &F) -> Result<bool, Errno> {
    let tv = getsockopt(fd, ReceiveTimeout)?;
    Ok(tv.tv_sec() != 0 || tv.tv_usec() != 0)
}

/// Safe wrapper for inotify_add_watch.
pub fn inotify_add_watch<Fd: AsFd, P: ?Sized + NixPath>(
    fd: Fd,
    path: &P,
    mask: AddWatchFlags,
) -> Result<c_int, Errno> {
    // SAFETY: We need this because nix' `WatchDescriptor` is opaque...
    let res = path.with_nix_path(|cstr| unsafe {
        libc::inotify_add_watch(fd.as_fd().as_raw_fd(), cstr.as_ptr(), mask.bits())
    })?;

    Errno::result(res).map(|wd| wd as c_int)
}

/// Safe wrapper for fallocate64.
pub fn fallocate64<Fd: AsFd>(
    fd: Fd,
    mode: FallocateFlags,
    off: off64_t,
    len: off64_t,
) -> Result<(), Errno> {
    // SAFETY: nix does not have an interface for fallocate64 yet.
    Errno::result(unsafe { libc::fallocate64(fd.as_fd().as_raw_fd(), mode.bits(), off, len) })
        .map(drop)
}

/// Safe truncate64() wrapper.
pub fn truncate64<P: ?Sized + NixPath>(path: &P, len: off64_t) -> Result<(), Errno> {
    // SAFETY: nix does not have a truncate64 wrapper.
    Errno::result(path.with_nix_path(|cstr| unsafe { libc::truncate64(cstr.as_ptr(), len) })?)
        .map(drop)
}

/// Safe ftruncate64() wrapper.
pub fn ftruncate64<Fd: AsFd>(fd: Fd, len: off64_t) -> Result<(), Errno> {
    // SAFETY: nix does not have a ftruncate64 wrapper.
    Errno::result(unsafe { libc::ftruncate64(fd.as_fd().as_raw_fd(), len) }).map(drop)
}

/// Get window-size from the given FD.
pub fn winsize_get<Fd: AsFd>(fd: Fd) -> Result<Winsize, Errno> {
    let fd = fd.as_fd().as_raw_fd();
    let req = TIOCGWINSZ.ok_or(Errno::ENOTTY)?;
    let mut ws = Winsize {
        ws_row: 0,
        ws_col: 0,
        ws_xpixel: 0,
        ws_ypixel: 0,
    };

    // SAFETY: In libc we trust.
    Errno::result(unsafe { syscall(SYS_ioctl, fd, req, &mut ws) })?;

    Ok(ws)
}

/// Set window-size for the given FD.
pub fn winsize_set<Fd: AsFd>(fd: Fd, ws: Winsize) -> Result<(), Errno> {
    let fd = fd.as_fd().as_raw_fd();
    let req = TIOCSWINSZ.ok_or(Errno::ENOTTY)?;

    // SAFETY: In libc we trust.
    Errno::result(unsafe { syscall(SYS_ioctl, fd, req, &ws) }).map(drop)
}

/// Write all the data to the given file.
///
/// Returns `Errno::EPIPE` on EOF. NO-OP if data is empty.
pub fn write_all<Fd: AsFd>(fd: Fd, mut data: &[u8]) -> Result<(), Errno> {
    while !data.is_empty() {
        match retry_on_eintr(|| write(&fd, data))? {
            0 => return Err(Errno::EPIPE),
            n => data = &data[n..],
        }
    }
    Ok(())
}

// Description of one file extent.
#[repr(C)]
#[derive(Debug, Clone, Copy)]
struct fiemap_extent {
    // Byte offset of the extent in the file.
    fe_logical: u64,
    // Byte offset of the extent on disk.
    fe_physical: u64,
    // Length in bytes for this extent.
    fe_length: u64,
    _fe_reserved64: [u64; 2],
    // FIEMAP_EXTENT_* flags for this extent.
    fe_flags: u32,
    _fe_reserved: [u32; 3],
}

// File extent mappings.
//
// - `fm_start`: byte offset (inclusive) at which to start mapping (in)
// - `fm_length`: logical length of mapping which userspace wants (in)
// - `fm_flags`: FIEMAP_FLAG_* flags for request (in/out)
// - `fm_mapped_extents`: number of extents that were mapped (out)
// - `fm_extent_count`: size of `fm_extents` array (in)
// - `fm_reserved`: reserved
// - `fm_extents`: array of mapped extents (out)
#[repr(C)]
struct fiemap {
    fm_start: u64,
    fm_length: u64,
    fm_flags: u32,
    fm_mapped_extents: u32,
    fm_extent_count: u32,
    _fm_reserved: u32,
    // Flexible array of extents; actual length is `fm_extent_count`.
    fm_extents: [fiemap_extent; 0],
}

/// FS_IOC_FIEMAP ioctl(2) request.
pub const FS_IOC_FIEMAP: c_ulong = _IOWR::<fiemap>(b'f' as u32, 11) as c_ulong;

/// FIGETBSZ ioctl(2) request.
pub const FIGETBSZ: c_ulong = _IO(0x00, 2) as c_ulong;

// Information for a single dedupe operation on a destination file.
#[repr(C)]
#[derive(Debug, Clone, Copy)]
struct file_dedupe_range_info {
    // Destination file descriptor.
    dest_fd: i64,
    // Start offset of the extent in the destination file.
    dest_offset: u64,
    // Number of bytes successfully deduped.
    bytes_deduped: u64,
    // Status of this dedupe operation:
    // < 0 for error,
    // == FILE_DEDUPE_RANGE_SAME if dedupe succeeds,
    // == FILE_DEDUPE_RANGE_DIFFERS if data differs.
    status: i32,
    // Must be zero.
    reserved: u32,
}

// Arguments for a range of dedupe operations from a source file.
//
// - `src_offset`: start of the extent in the source file (in)
// - `src_length`: length of the extent (in)
// - `dest_count`: number of elements in the `info` array (in)
// - `reserved1`: must be zero
// - `reserved2`: must be zero
// - `info`: array of `file_dedupe_range_info` of length `dest_count` (out)
#[repr(C)]
struct file_dedupe_range {
    src_offset: u64,
    src_length: u64,
    dest_count: u16,
    reserved1: u16,
    reserved2: u32,
    // Flexible array member of length `dest_count`.
    info: [file_dedupe_range_info; 0],
}

/// FIDEDUPERANGE ioctl(2) request.
pub const FIDEDUPERANGE: c_ulong = _IOWR::<file_dedupe_range>(0x94, 54) as c_ulong;

// Filesystem UUID structure with fixed 16-byte buffer.
//
// - `len`: actual length of the UUID (≤16)
// - `uuid`: UUID bytes
#[repr(C)]
#[derive(Debug, Clone, Copy)]
struct fsuuid2 {
    len: u8,
    uuid: [u8; 16],
}

/// FS_IOC_GETFSUUID ioctl(2) request.
pub const FS_IOC_GETFSUUID: c_ulong = _IOR::<fsuuid2>(0x15, 0) as c_ulong;

// Filesystem sysfs path structure.
//
// - `len`: length of the returned name (≤128)
// - `name`: NUL-terminated path component under `/sys/fs/` or `/sys/kernel/debug/`
#[repr(C)]
#[derive(Debug, Clone, Copy)]
struct fs_sysfs_path {
    len: u8,
    name: [u8; 128],
}

/// FS_IOC_GETFSSYSFSPATH ioctl(2) request.
pub const FS_IOC_GETFSSYSFSPATH: c_ulong = _IOR::<fs_sysfs_path>(0x15, 1) as c_ulong;

/// FIBMAP ioctl(2) request.
pub const FIBMAP: c_ulong = _IO(0x00, 1) as c_ulong;

/// KDSETKEYCODE ioctl(2) request.
pub const KDSETKEYCODE: c_ulong = 0x4B4D;

/// KDSIGACCEPT ioctl(2) request.
pub const KDSIGACCEPT: c_ulong = 0x4B4E;

// File system extended attribute operations.
//
// Used with `FS_IOC_FSGETXATTR` and `FS_IOC_FSSETXATTR`.
#[repr(C)]
#[derive(Debug, Clone, Copy)]
struct fsxattr {
    // xflags field value (get/set)
    fsx_xflags: u32,
    // extsize field value (get/set)
    fsx_extsize: u32,
    // nextents field value (get)
    fsx_nextents: u32,
    // project identifier (get/set)
    fsx_projid: u32,
    // CoW extsize field value (get/set)
    fsx_cowextsize: u32,
    // Padding
    fsx_pad: [u8; 8],
}

/// FS_IOC_FSGETXATTR ioctl(2) request.
pub const FS_IOC_FSGETXATTR: c_ulong = _IOR::<fsxattr>(b'X' as u32, 31) as c_ulong;

/// FS_IOC_FSSETXATTR ioctl(2) request.
pub const FS_IOC_FSSETXATTR: c_ulong = _IOW::<fsxattr>(b'X' as u32, 32) as c_ulong;

/// FS_IOC_SETFLAGS ioctl(2) request.
pub const FS_IOC_SETFLAGS: c_ulong = _IOW::<c_long>(b'f' as u32, 2) as c_ulong;

/*
 * Seccomp constants
 */

/// '!' magic number for seccomp ioctls.
pub const SECCOMP_IOCTL_MAGIC: u32 = b'!' as u32;

/// SECCOMP_IOCTL_NOTIF_RECV ioctl(2) request.
pub const SECCOMP_IOCTL_NOTIF_RECV: c_ulong =
    _IOWR::<seccomp_notif>(SECCOMP_IOCTL_MAGIC, 0) as c_ulong;

/// SECCOMP_IOCTL_NOTIF_SEND ioctl(2) request.
pub const SECCOMP_IOCTL_NOTIF_SEND: c_ulong =
    _IOWR::<seccomp_notif_resp>(SECCOMP_IOCTL_MAGIC, 1) as c_ulong;

/// SECCOMP_IOCTL_NOTIF_ID_VALID ioctl(2) request.
pub const SECCOMP_IOCTL_NOTIF_ID_VALID: c_ulong = _IOW::<u64>(SECCOMP_IOCTL_MAGIC, 2) as c_ulong;

/// SECCOMP_IOCTL_NOTIF_ADDFD ioctl(2) request.
pub const SECCOMP_IOCTL_NOTIF_ADDFD: c_ulong =
    _IOW::<seccomp_notif_addfd>(SECCOMP_IOCTL_MAGIC, 3) as c_ulong;

/// SECCOMP_IOCTL_NOTIF_SET_FLAGS ioctl(2) request.
pub const SECCOMP_IOCTL_NOTIF_SET_FLAGS: c_ulong = _IOW::<u64>(SECCOMP_IOCTL_MAGIC, 4) as c_ulong;

pub(crate) const SECCOMP_IOCTL_NOTIF_LIST: &[c_ulong] = &[
    SECCOMP_IOCTL_NOTIF_RECV,
    SECCOMP_IOCTL_NOTIF_SEND,
    SECCOMP_IOCTL_NOTIF_ID_VALID,
    SECCOMP_IOCTL_NOTIF_ADDFD,
    SECCOMP_IOCTL_NOTIF_SET_FLAGS,
];

/// Flag to set synchronous mode for the seccomp notify fd.
pub(crate) const SECCOMP_USER_NOTIF_FD_SYNC_WAKE_UP: u32 = 1;

/// Export a seccomp filter in pretty-printed PFC.
///
/// Mostly equivalent to _seccomp_export_pfc(3) with
/// some convenience replacements for seccomp constants,
/// and returns a String rather than a FD.
///
/// The temporary file is created in `/tmp` directory
/// with O_TMPFILE|O_EXCL flags and empty/zero Mode.
pub(crate) fn seccomp_export_pfc(ctx: &ScmpFilterContext) -> Result<String, Errno> {
    // SAFETY: This runs at startup before confinement!
    #[allow(clippy::disallowed_methods)]
    let mut file = nix::fcntl::openat(
        AT_FDCWD,
        "/tmp",
        OFlag::O_TMPFILE | OFlag::O_EXCL | OFlag::O_RDWR,
        Mode::empty(),
    )
    .map(File::from)?;
    ctx.export_pfc(&mut file).or(Err(Errno::EFAULT))?;

    file.seek(SeekFrom::Start(0)).map_err(|err| err2no(&err))?;
    let mut buf = Vec::new();
    file.read_to_end(&mut buf).map_err(|err| err2no(&err))?;

    // from_utf8_lossy_to_owned() is nightly...
    let mut pfc = String::from_utf8_lossy(&buf).into_owned();
    for &(from, to) in &[
        ("0x7fc00000", "NOTIFY"),
        (
            &format!("{SECCOMP_IOCTL_NOTIF_RECV}"),
            "SECCOMP_IOCTL_NOTIF_RECV",
        ),
        (
            &format!("{SECCOMP_IOCTL_NOTIF_SEND}"),
            "SECCOMP_IOCTL_NOTIF_SEND",
        ),
        (
            &format!("{SECCOMP_IOCTL_NOTIF_ID_VALID}"),
            "SECCOMP_IOCTL_NOTIF_ID_VALID",
        ),
        (
            &format!("{SECCOMP_IOCTL_NOTIF_ADDFD}"),
            "SECCOMP_IOCTL_NOTIF_ADDFD",
        ),
        (
            &format!("{SECCOMP_IOCTL_NOTIF_SET_FLAGS}"),
            "SECCOMP_IOCTL_NOTIF_SET_FLAGS",
        ),
        (&format!("{PROCMAP_QUERY}"), "PROCMAP_QUERY"),
    ] {
        pfc = pfc.replace(from, to);
    }

    Ok(pfc)
}

/// Set seccomp notify fd flags, useful to set synchronous mode.
pub(crate) fn seccomp_notify_set_flags(fd: RawFd, flags: u32) -> Result<(), Errno> {
    if !*HAVE_SECCOMP_USER_NOTIF_FD_SYNC_WAKE_UP {
        return Err(Errno::ENOSYS);
    }

    retry_on_eintr(|| {
        // SAFETY: In libc we trust.
        Errno::result(unsafe {
            syscall(
                SYS_ioctl,
                fd,
                SECCOMP_IOCTL_NOTIF_SET_FLAGS as c_ulong,
                flags,
            )
        })
    })
    .map(drop)
}

/// Wrapper for SECCOMP_IOCTL_NOTIF_ID_VALID ioctl(2),
/// aka _seccomp_notify_id_valid_(3) of libseccomp.
///
/// Inlined for efficiency in hot-paths.
#[inline(always)]
pub(crate) fn seccomp_notify_id_valid(fd: RawFd, id: u64) -> Result<(), Errno> {
    retry_on_eintr(|| {
        // SAFETY: Validate request ID against the seccomp-notify fd:
        // 1. This function is a hot path where we don't want to run
        //    notify_supported() on each call.
        // 2. We want to reliably handle EAGAIN and EINTR.
        // 3. ENOENT means child died mid-way.
        // libseccomp::notify_id_valid(fd, id).is_ok().
        Errno::result(unsafe {
            syscall(SYS_ioctl, fd, SECCOMP_IOCTL_NOTIF_ID_VALID as c_ulong, &id)
        })
    })
    .map(drop)
}

/// Wrapper for SECCOMP_IOCTL_NOTIF_SEND ioctl(2),
/// aka _seccomp_notify_respond_(3) of libseccomp.
pub(crate) fn seccomp_notify_respond(
    fd: RawFd,
    response: *const seccomp_notif_resp,
) -> Result<(), Errno> {
    retry_on_eintr(|| {
        // SAFETY:
        // 1. libseccomp's version allocates needlessly, and
        // 2. libseccomp-sys's version requires a mutable pointer, and
        // 3. libseccomp does not export EINTR to user API, but returns EFAULT as catch-all:
        //    https://github.com/seccomp/libseccomp/blob/5491c4b931431bec489dd78247ef675fc1b49797/src/api.c#L92-L95
        //    So we use _ioctl_(2) directly.
        // 4. EINTR may mean `syd_int` thread misfired us.
        // 5. ENOENT means child-died mid-way.
        // 6. Ok() is all good!
        // 7. We protect SECCOMP_IOCTL_NOTIF_SEND with system call argument cookies,
        //    to raise the bar against an attacker who has compromised Syd and aims
        //    to inject the flag SECCOMP_USER_NOTIF_FLAG_CONTINUE to this response
        //    in order to pass-through a system call to the host Linux kernel.
        // 8. Randomizing the seccomp-fd at startup is another mitigation against this.
        Errno::result(unsafe {
            syscall(
                SYS_ioctl,
                fd,
                SECCOMP_IOCTL_NOTIF_SEND as c_ulong,
                response,
                *SECCOMP_IOCTL_NOTIF_SEND_COOKIE_ARG3,
                *SECCOMP_IOCTL_NOTIF_SEND_COOKIE_ARG4,
                *SECCOMP_IOCTL_NOTIF_SEND_COOKIE_ARG5,
            )
        })
    })
    .map(drop)
}

/// Wrapper for SECCOMP_IOCTL_NOTIF_ADDFD ioctl(2).
pub(crate) fn seccomp_notify_addfd(
    fd: RawFd,
    addfd: *const seccomp_notif_addfd,
) -> Result<(), Errno> {
    retry_on_eintr(|| {
        // SAFETY:
        // 1. libseccomp has no wrapper for ADDFD yet, and
        // 2. libseccomp does not export EINTR to user API, but returns EFAULT as catch-all:
        //    https://github.com/seccomp/libseccomp/blob/5491c4b931431bec489dd78247ef675fc1b49797/src/api.c#L92-L95
        //    So we use _ioctl_(2) directly.
        // 3. EINTR may mean `syd_int` thread misfired us.
        // 4. ENOENT means child-died mid-way.
        // 5. Ok() is all good!
        // 6. We protect SECCOMP_IOCTL_NOTIF_ADDFD with system call argument cookies,
        //    to raise the bar against an attacker who has compromised Syd and aims
        //    steal file descriptors.
        // 7. Randomizing the seccomp-fd at startup is another mitigation against this.
        Errno::result(unsafe {
            syscall(
                SYS_ioctl,
                fd,
                SECCOMP_IOCTL_NOTIF_ADDFD as c_ulong,
                addfd,
                *SECCOMP_IOCTL_NOTIF_ADDFD_COOKIE_ARG3,
                *SECCOMP_IOCTL_NOTIF_ADDFD_COOKIE_ARG4,
                *SECCOMP_IOCTL_NOTIF_ADDFD_COOKIE_ARG5,
            )
        })
    })
    .map(drop)
}

/// Returns file mode for the given file descriptor.
/// The file mode includes the file type.
pub fn fd_mode<Fd: AsFd>(fd: Fd) -> Result<mode_t, Errno> {
    retry_on_eintr(|| fstatx(&fd, STATX_MODE | STATX_TYPE)).map(|statx| statx.stx_mode.into())
}

/// Returns true if the given file descriptor is active.
pub fn is_active_fd<Fd: AsFd>(fd: Fd) -> bool {
    fcntl(fd, FcntlArg::F_GETFD).is_ok()
}

/// Returns true if the given file descriptor is syntactically valid.
///
/// Negative values, including AT_FDCWD, are not syntactically valid.
#[inline(always)]
pub fn is_valid_fd(fd: u64) -> bool {
    to_valid_fd(fd).map(|fd| fd >= 0).unwrap_or(false)
}

/// Converts a system call argument to a RawFd.
///
/// Negative values, excluding AT_FDCWD, return an error.
#[inline(always)]
#[allow(clippy::cast_possible_truncation)]
pub fn to_valid_fd(fd: u64) -> Result<RawFd, Errno> {
    let fd = fd as RawFd;

    if fd == libc::AT_FDCWD || fd >= 0 {
        Ok(fd)
    } else {
        Err(Errno::EBADF)
    }
}

/// Returns file access mode in status flags.
pub fn fd_status_flags<Fd: AsFd>(fd: Fd) -> Result<OFlag, Errno> {
    fcntl(fd, FcntlArg::F_GETFL).map(OFlag::from_bits_truncate)
}

/// Returns true if file is writable.
pub fn is_writable_fd<Fd: AsFd>(fd: Fd) -> Result<bool, Errno> {
    let flags = fd_status_flags(fd)?;

    Ok(flags.contains(OFlag::O_WRONLY) || flags.contains(OFlag::O_RDWR))
}

/// Get number of open file descriptors.
pub fn fd_count(pid: Option<Pid>) -> Result<u64, Errno> {
    let mut pfd = XPathBuf::from("/proc");
    if let Some(pid) = pid {
        pfd.push_pid(pid);
    } else {
        pfd.push(b"thread-self");
    }
    pfd.push(b"fd");

    if *HAVE_PROC_PID_FD_STAT_SIZE {
        let stx = statx(AT_BADFD, &pfd, 0, STATX_SIZE)?;
        return Ok(stx.stx_size);
    }

    #[allow(clippy::disallowed_methods)]
    let fd = nix::fcntl::openat(
        AT_BADFD,
        &pfd,
        OFlag::O_RDONLY | OFlag::O_DIRECTORY | OFlag::O_CLOEXEC,
        Mode::empty(),
    )?;
    let mut nfds: u64 = 0;
    loop {
        match getdents64(&fd, DIRENT_BUF_SIZE) {
            Ok(entries) => {
                nfds = nfds
                    .checked_add(entries.count() as u64)
                    .ok_or(Errno::ERANGE)?
            }
            Err(Errno::ECANCELED) => break, // EOF or empty directory.
            Err(errno) => return Err(errno),
        };
    }

    Ok(nfds.saturating_sub(2))
}

/// Represents UNIX file types
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum FileType {
    /// Represents regular file
    Reg,
    /// Represents directory
    Dir,
    /// Represents symbolic link
    ///
    /// Optionally, carries with it the symlink path it originates.
    Lnk,
    /// Represents a procfs magic symbolic link
    MagicLnk(ProcMagic),
    /// Represents a memory file descriptor.
    Mfd,
    /// Represents FIFO
    Fifo,
    /// Represents socket
    Sock,
    /// Represents character device
    Chr,
    /// Represents block device
    Blk,
    /// Represents unknown file
    Unk,
}

impl Serialize for FileType {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        // SAFETY: Both Lnk and MagicLnk have XPathBuf as saved data,
        // and its Display trait masks control characters.
        let repr = match self {
            Self::Reg => "reg".to_string(),
            Self::Dir => "dir".to_string(),
            Self::Lnk => "lnk".to_string(),
            Self::MagicLnk(magic) => format!("mgc@{}", &magic.link_path()),
            Self::Mfd => "memfd".to_string(),
            Self::Fifo => "fifo".to_string(),
            Self::Sock => "sock".to_string(),
            Self::Chr => "chr".to_string(),
            Self::Blk => "blk".to_string(),
            Self::Unk => "unk".to_string(),
        };

        serializer.serialize_str(&repr)
    }
}

impl FileType {
    /// Test whether file type represents a regular file.
    pub fn is_file(self) -> bool {
        self == Self::Reg
    }

    /// Test whether file type represents a directory.
    pub fn is_dir(self) -> bool {
        self == Self::Dir
    }

    /// Test whether file type represents a symlink.
    pub fn is_symlink(self) -> bool {
        self == Self::Lnk
    }

    /// Test whether file type represents a magic link.
    pub fn is_magic_link(self) -> bool {
        matches!(self, Self::MagicLnk(_))
    }

    /// Test whether file type represents a magic directory.
    ///
    /// SAFETY: Used to harden magic link open calls with O_DIRECTORY.
    pub fn is_magic_dir(self) -> bool {
        matches!(self, Self::MagicLnk(magic) if magic.want_dir())
    }

    /// Test whether file type represents a memory fd.
    pub fn is_mfd(self) -> bool {
        self == Self::Mfd
    }

    /// Test whether file type represents a block device.
    pub fn is_block_device(self) -> bool {
        self == Self::Blk
    }

    /// Test whether file type represents a character device.
    pub fn is_char_device(self) -> bool {
        self == Self::Chr
    }

    /// Test whether file type represents a FIFO.
    pub fn is_fifo(self) -> bool {
        self == Self::Fifo
    }

    /// Test whether file type represents a socket.
    pub fn is_socket(self) -> bool {
        self == Self::Sock
    }

    /// Test whether file type represents an unknown file.
    pub fn is_unknown(self) -> bool {
        self == Self::Unk
    }

    /// Returns the file type as `mode_t`.
    ///
    /// Returns `None` if file type can't be represented as `mode_t`.
    pub fn mode(self) -> Option<mode_t> {
        match self {
            Self::Blk => Some(S_IFBLK),
            Self::Chr => Some(S_IFCHR),
            Self::Dir => Some(S_IFDIR),
            Self::Fifo => Some(S_IFIFO),
            Self::Lnk => Some(S_IFLNK),
            Self::Reg => Some(S_IFREG),
            Self::Sock => Some(S_IFSOCK),
            _ => None,
        }
    }
}

impl From<u8> for FileType {
    fn from(dt_type: u8) -> Self {
        match dt_type {
            DT_DIR => Self::Dir,
            DT_REG => Self::Reg,
            DT_LNK => Self::Lnk,
            DT_CHR => Self::Chr,
            DT_BLK => Self::Blk,
            DT_FIFO => Self::Fifo,
            DT_SOCK => Self::Sock,
            _ => Self::Unk,
        }
    }
}

impl From<mode_t> for FileType {
    fn from(mode: mode_t) -> Self {
        match mode & S_IFMT {
            S_IFBLK => Self::Blk,
            S_IFCHR => Self::Chr,
            S_IFDIR => Self::Dir,
            S_IFIFO => Self::Fifo,
            S_IFLNK => Self::Lnk,
            S_IFREG => Self::Reg,
            S_IFSOCK => Self::Sock,
            _ => Self::Unk,
        }
    }
}

/// Return the file type of the given file descriptor, and optional path.
#[allow(clippy::cast_possible_truncation)]
pub fn file_type<Fd: AsFd>(
    fd: Fd,
    p: Option<&XPath>,
    follow_symlinks: bool,
) -> Result<FileType, Errno> {
    if let Some(p) = p {
        statx(
            fd,
            p,
            if p.is_empty() {
                AT_EMPTY_PATH
            } else if follow_symlinks {
                0
            } else {
                AT_SYMLINK_NOFOLLOW
            },
            STATX_TYPE,
        )
    } else {
        fstatx(fd, STATX_TYPE)
    }
    .map(|statx| FileType::from(mode_t::from(statx.stx_mode)))
}

/// Checks if the given file mode represents a sidechannel device.
///
/// A sidechannel device is defined as a character or block device that
/// is world-readable or world-writable. This function checks the file
/// mode to determine if it meets these criteria.
///
/// # Arguments
///
/// * `mode` - The mode of the file to check.
#[inline]
pub fn is_sidechannel_device(mode: mode_t) -> bool {
    mode & (S_IROTH | S_IWOTH) != 0 && matches!(mode & S_IFMT, S_IFCHR | S_IFBLK)
}

/// A safe version of clone that returns a PidFD,
/// and therefore is not subject to PID-recycling
/// races.
pub fn safe_clone(
    mut cb: CloneCb,
    stack: &mut [u8],
    flags: c_int,
    signal: Option<c_int>,
) -> Result<OwnedFd, Errno> {
    #[allow(clippy::cast_possible_truncation)]
    extern "C" fn callback(data: *mut CloneCb) -> c_int {
        // SAFETY: nix' version does not support CLONE_PIDFD.
        let cb: &mut CloneCb = unsafe { &mut *data };
        (*cb)() as c_int
    }

    let mut pid_fd: c_int = -1;
    let combined: c_int = flags | CLONE_PIDFD | signal.unwrap_or(0);
    // SAFETY: ditto.
    #[allow(clippy::missing_transmute_annotations)]
    let res = unsafe {
        let ptr = stack.as_mut_ptr().add(stack.len());
        let ptr_aligned = ptr.sub(ptr as usize % 16);
        clone(
            std::mem::transmute(callback as extern "C" fn(*mut Box<dyn FnMut() -> isize>) -> i32),
            ptr_aligned as *mut c_void,
            combined,
            std::ptr::addr_of_mut!(cb) as *mut c_void,
            &mut pid_fd,
        )
    };

    Errno::result(res).map(|_| {
        // SAFETY: clone with CLONE_PIDFD returns a valid FD.
        unsafe { OwnedFd::from_raw_fd(pid_fd) }
    })
}

/// Ensure the file we open is a regular file.
/// Ensure we do not block on a fifo if the file is one.
/// Returns (File, FileStatx) on success, Errno on failure.
pub fn safe_open_file<Fd: AsFd>(fd: Fd, base: &XPath) -> Result<(File, FileStatx), Errno> {
    // Step 1: Open the file with O_PATH.
    let fd = safe_open_path(fd, base, OFlag::O_NOFOLLOW, ResolveFlag::empty())?;

    // Step 2: Check the file type and bail if it's not a regular file.
    let statx = retry_on_eintr(|| fstatx(&fd, STATX_INO | STATX_TYPE | STATX_SIZE))?;
    let ftype = FileType::from(mode_t::from(statx.stx_mode));
    if !ftype.is_file() {
        return Err(Errno::ENOEXEC);
    }

    // Step 3: Reopen the file safely via /proc/thread-self/fd.
    // Note, we cannot use RESOLVE_NO_{MAGIC,SYM}LINKS or O_NOFOLLOW here.
    let pfd = XPathBuf::from_self_fd(fd.as_raw_fd());

    let flags = OFlag::O_RDONLY | OFlag::O_NOCTTY | OFlag::O_CLOEXEC;
    let file = safe_open_msym(PROC_FILE(), &pfd, flags, ResolveFlag::empty()).map(File::from)?;

    // Step 4: Return the File and FileStatx.
    Ok((file, statx))
}

/// Safely open and copy the given pathname into the file,
/// if the pathname exists. Returns the number of bytes copied.
pub fn safe_copy_if_exists<F: Write>(dst: &mut F, src: &XPath) -> Result<u64, Errno> {
    let how = safe_open_how(OFlag::O_PATH, ResolveFlag::empty());

    // SAFETY: This function is only called before sandboxing.
    #[allow(clippy::disallowed_methods)]
    let fd = if let Ok(fd) = retry_on_eintr(|| openat2(AT_FDCWD, src, how)) {
        fd
    } else {
        return Ok(0);
    };

    // Check the file type and bail if it's not a regular file.
    let statx = retry_on_eintr(|| fstatx(&fd, STATX_TYPE))?;
    let ftype = FileType::from(mode_t::from(statx.stx_mode));
    if !ftype.is_file() {
        return Err(Errno::ENOEXEC);
    }

    // Reopen the file safely via /proc/self/fd.
    // Note, we cannot use RESOLVE_NO_{MAGIC,SYM}LINKS or O_NOFOLLOW here.
    let mut pfd = XPathBuf::from("/proc/self");
    pfd.push(b"fd");
    pfd.push_fd(fd.as_raw_fd());

    let how = safe_open_how(OFlag::O_RDONLY | OFlag::O_NOCTTY, ResolveFlag::empty());

    // SAFETY: This function is only called before sandboxing.
    #[allow(clippy::disallowed_methods)]
    let mut src = if let Ok(src) = retry_on_eintr(|| openat2(AT_FDCWD, &pfd, how).map(File::from)) {
        src
    } else {
        return Ok(0);
    };

    #[allow(clippy::disallowed_methods)]
    std::io::copy(&mut src, dst).map_err(|e| err2no(&e))
}

/// Return a safe OpenHow structure.
pub fn safe_open_how(flags: OFlag, rflags: ResolveFlag) -> OpenHow {
    // Note we leave the caller to handle O_NOCTTY,
    // because its use is invalid with O_PATH.
    let mode = if flags.contains(OFlag::O_CREAT) || flags.contains(OFlag::O_TMPFILE) {
        Mode::from_bits_truncate(0o600)
    } else {
        Mode::empty()
    };
    OpenHow::new()
        .flags(flags | OFlag::O_CLOEXEC | OFlag::O_NOFOLLOW)
        .mode(mode)
        .resolve(
            rflags
                | ResolveFlag::RESOLVE_NO_MAGICLINKS
                | ResolveFlag::RESOLVE_NO_SYMLINKS
                | ResolveFlag::RESOLVE_BENEATH,
        )
}

/// Return a safe OpenHow structure without RESOLVE_BENEATH,
/// which allows for absolute pathnames.
pub fn safe_open_how_abs(flags: OFlag, rflags: ResolveFlag) -> OpenHow {
    // Note we leave the caller to handle O_NOCTTY,
    // because its use is invalid with O_PATH.
    let mode = if flags.contains(OFlag::O_CREAT) || flags.contains(OFlag::O_TMPFILE) {
        Mode::from_bits_truncate(0o600)
    } else {
        Mode::empty()
    };
    OpenHow::new()
        .flags(flags | OFlag::O_CLOEXEC | OFlag::O_NOFOLLOW)
        .mode(mode)
        .resolve(rflags | ResolveFlag::RESOLVE_NO_MAGICLINKS | ResolveFlag::RESOLVE_NO_SYMLINKS)
}

/// Return a safe OpenHow structure without O_NOFOLLOW,
/// RESOLVE_NO_MAGICLINKS and RESOLVE_NO_SYMLINKS.
#[inline]
pub fn safe_open_how_msym(flags: OFlag, rflags: ResolveFlag) -> OpenHow {
    // Note we leave the caller to handle O_NOCTTY,
    // because its use is invalid with O_PATH.
    OpenHow::new()
        .flags(flags | OFlag::O_CLOEXEC)
        .resolve(rflags)
}

/// Open a path safely using O_PATH and return an OwnedFd.
pub fn safe_open_path<Fd: AsFd>(
    fd: Fd,
    base: &XPath,
    flags: OFlag,
    rflags: ResolveFlag,
) -> Result<OwnedFd, Errno> {
    safe_open(fd, base, OFlag::O_PATH | flags, rflags)
}

/// Open a path safely and return an OwnedFd.
pub fn safe_open<Fd: AsFd>(
    fd: Fd,
    base: &XPath,
    flags: OFlag,
    rflags: ResolveFlag,
) -> Result<OwnedFd, Errno> {
    let how = safe_open_how(flags, rflags);

    let (fd, base) = if base.is_relative() {
        (fd.as_fd(), base)
    } else if base.is_rootfs() {
        unreachable!("BUG: Attempt to reopen /");
    } else {
        (
            ROOT_FILE(),
            XPath::from_bytes(&base.as_bytes()[b"/".len()..]),
        )
    };

    #[allow(clippy::disallowed_methods)]
    retry_on_eintr(|| safe_openat2(fd, base, how))
}

/// Open an absolute path safely using O_PATH and return an OwnedFd.
pub fn safe_open_path_abs(
    path: &XPath,
    flags: OFlag,
    rflags: ResolveFlag,
) -> Result<OwnedFd, Errno> {
    safe_open_abs(path, OFlag::O_PATH | flags, rflags)
}

/// Open an absolute path safely and return an OwnedFd.
pub fn safe_open_abs(path: &XPath, flags: OFlag, rflags: ResolveFlag) -> Result<OwnedFd, Errno> {
    let how = safe_open_how_abs(flags, rflags);

    // SAFETY: Ensure path is an absolute path.
    if path.is_relative() {
        return Err(Errno::EINVAL);
    }

    #[allow(clippy::disallowed_methods)]
    retry_on_eintr(|| openat2(AT_FDCWD, path, how))
}

/// Open a magic symlink safely using O_PATH and return an OwnedFd.
pub fn safe_open_path_msym<Fd: AsFd>(
    fd: Fd,
    base: &XPath,
    flags: OFlag,
    rflags: ResolveFlag,
) -> Result<OwnedFd, Errno> {
    safe_open_msym(fd, base, OFlag::O_PATH | flags, rflags)
}

/// Open a magic symlink safely and return an OwnedFd.
pub fn safe_open_msym<Fd: AsFd>(
    fd: Fd,
    base: &XPath,
    flags: OFlag,
    rflags: ResolveFlag,
) -> Result<OwnedFd, Errno> {
    let how = safe_open_how_msym(flags, rflags);

    let (fd, base) = if base.is_relative() {
        (fd.as_fd(), base)
    } else if base.is_rootfs() {
        unreachable!("BUG: Attempt to reopen /");
    } else {
        (
            ROOT_FILE(),
            XPath::from_bytes(&base.as_bytes()[b"/".len()..]),
        )
    };

    retry_on_eintr(|| safe_openat2(fd, base, how))
}

/// Get an extended attribute value.
///
/// # Arguments
/// - `fd`    - The file descriptor of the file.
/// - `name`  - The name of the extended attribute.
/// - `value` - The buffer to store the attribute value,
///   if None do a check run.
///
/// # Returns
/// - `Result<usize>` - The size of the extended attribute value.
///
/// # Errors
/// Returns an error if the operation fails.
///
/// # Safety
/// This function is unsafe as it directly interfaces with the libc function `fgetxattr`.
///
/// # See Also
/// [`fgetxattr`](https://man7.org/linux/man-pages/man2/fgetxattr.2.html)
pub fn fgetxattr<Fd: AsFd, P: ?Sized + NixPath>(
    fd: Fd,
    name: &P,
    value: Option<&mut [u8]>,
) -> Result<usize, Errno> {
    let (value, len) = match value {
        Some(v) => (v.as_mut_ptr() as *mut c_void, v.len() as size_t),
        None => (std::ptr::null_mut(), 0),
    };

    // SAFETY: nix lacks a wrapper for fgetxattr.
    let res = name.with_nix_path(|name_ptr| unsafe {
        libc::fgetxattr(fd.as_fd().as_raw_fd(), name_ptr.as_ptr(), value, len)
    })?;

    #[allow(clippy::cast_sign_loss)]
    Errno::result(res).map(|res| res as usize)
}

/// Set an extended attribute value.
///
/// # Arguments
/// - `fd`    - The file descriptor of the file.
/// - `name`  - The name of the extended attribute.
/// - `value` - The buffer containing the attribute value.
/// - `flags` - Flags to control the operation.
///
/// # Returns
/// - `Result<()>` - Returns an Ok result if the operation succeeds.
///
/// # Errors
/// Returns an error if the operation fails.
///
/// # Safety
/// This function is unsafe as it directly interfaces with the libc function `fsetxattr`.
///
/// # See Also
/// [`fsetxattr`](https://man7.org/linux/man-pages/man2/fsetxattr.2.html)
pub fn fsetxattr<Fd: AsFd, P: ?Sized + NixPath>(
    fd: Fd,
    name: &P,
    value: &[u8],
    flags: i32,
) -> Result<(), Errno> {
    // SAFETY: nix lacks a wrapper for fsetxattr.
    let res = name.with_nix_path(|name_ptr| unsafe {
        libc::fsetxattr(
            fd.as_fd().as_raw_fd(),
            name_ptr.as_ptr(),
            value.as_ptr() as *const c_void,
            value.len() as size_t,
            flags as c_int,
        )
    })?;

    Errno::result(res).map(drop)
}

/// Remove an extended attribute value.
///
/// # Arguments
/// - `fd`    - The file descriptor of the file.
/// - `name`  - The name of the extended attribute.
///
/// # Returns
/// - `Result<()>` - Returns an Ok result if the operation succeeds.
///
/// # Errors
/// Returns an error if the operation fails.
///
/// # Safety
/// This function is unsafe as it directly interfaces with the libc function `fremovexattr`.
///
/// # See Also
/// [`fremovexattr`](https://man7.org/linux/man-pages/man2/fremovexattr.2.html)
pub fn fremovexattr<Fd: AsFd, P: ?Sized + NixPath>(fd: Fd, name: &P) -> Result<(), Errno> {
    // SAFETY: nix lacks a wrapper for fremovexattr.
    let res = name.with_nix_path(|name_ptr| unsafe {
        libc::fremovexattr(fd.as_fd().as_raw_fd(), name_ptr.as_ptr())
    })?;

    Errno::result(res).map(drop)
}

const SEC_XATTR: &[u8] = b"security.";
const SYD_XATTR: &[u8] = b"user.syd.";
const XATTR_SEC: &[&[u8]] = &[SEC_XATTR, SYD_XATTR];

/// Deny access to the following extended attribute prefixes:
///
/// 1. security.*
/// 2. user.syd.*
///
/// # Safety
///
/// Dereferences name after a NULL check.
/// If name is not NULL, it must be a valid NUL-terminated C-String.
///
/// # Security
///
/// Denies with ENODATA for stealth.
pub unsafe fn denyxattr(name: *const c_char) -> Result<(), Errno> {
    if name.is_null() {
        return Ok(());
    }

    for prefix in XATTR_SEC {
        // SAFETY: The pointer from CStr is guaranteed to be
        // valid and null-terminated.
        if unsafe { strncmp(name, prefix.as_ptr().cast(), prefix.len()) } == 0 {
            return Err(Errno::ENODATA);
        }
    }

    Ok(())
}

/// Filters out the following extended attribute prefixes:
///
/// 1. security.*
/// 2. user.syd.*
///
/// # Arguments
///
/// * `buf` - A buffer containing the extended attribute names as
///   null-terminated strings.
/// * `n` - The length of valid data in the buffer.
pub fn filterxattr(buf: &[u8], n: usize) -> Result<Vec<u8>, Errno> {
    let mut filtered_buf = Vec::new();
    let mut start = 0;

    #[allow(clippy::arithmetic_side_effects)]
    while start < n {
        if let Some(end) = buf[start..].iter().position(|&c| c == 0) {
            let name = &buf[start..start + end + 1]; // +1 to include the null terminator.
            let cstr = CStr::from_bytes_with_nul(name).or(Err(Errno::E2BIG))?;

            let cstr = cstr.to_bytes();
            let mut filter = false;
            for prefix in XATTR_SEC {
                if cstr.starts_with(prefix) {
                    filter = true;
                    break;
                }
            }

            if !filter {
                filtered_buf.extend_from_slice(name);
            }

            start += end + 1;
        } else {
            break;
        }
    }

    Ok(filtered_buf)
}

/// Get secure bytes using the OS random number generator.
pub fn getrandom(size: usize) -> Result<Vec<u8>, Errno> {
    if size == 0 {
        // SAFETY:
        // Return EINVAL on zero length which is a common case of error.
        return Err(Errno::EINVAL);
    }

    let mut buf = Vec::new();
    if buf.try_reserve(size).is_err() {
        return Err(Errno::ENOMEM);
    }
    buf.resize(size, 0);

    fillrandom(&mut buf)?;
    Ok(buf)
}

/// Fill the given buffer using the OS random number generator.
pub fn fillrandom(buf: &mut [u8]) -> Result<(), Errno> {
    // SAFETY: Ensure buffer is not empty,
    // which is a common case of error.
    let siz = buf.len();
    if siz == 0 {
        return Err(Errno::EINVAL);
    }

    let mut n = 0;
    #[allow(clippy::arithmetic_side_effects)]
    #[allow(clippy::cast_sign_loss)]
    while n < siz {
        let ptr = &mut buf[n..];
        let ptr = ptr.as_mut_ptr().cast();

        n += retry_on_eintr(|| {
            // SAFETY: In libc we trust.
            Errno::result(unsafe { libc::getrandom(ptr, siz - n, GRND_RANDOM) })
        })? as usize;
    }

    Ok(())
}

/// Return a random `u64` within the given inclusive range using the OS random number generator.
#[allow(clippy::arithmetic_side_effects)]
pub fn randint(range: RangeInclusive<u64>) -> Result<u64, Errno> {
    let (start, end) = range.into_inner();

    if start > end {
        return Err(Errno::ERANGE);
    }

    let mut buf = [0u8; std::mem::size_of::<u64>()];
    fillrandom(&mut buf)?;

    Ok(start + (u64::from_le_bytes(buf) % (end - start + 1)))
}

/// Duplicate the file descriptor to a random fd.
///
/// Valid flags:
/// - O_EXCL: closes oldfd after successful duplication.
/// - All other flags are passed to dup3(2), ie O_CLOEXEC.
#[allow(clippy::arithmetic_side_effects)]
pub fn duprand(oldfd: RawFd, mut flags: OFlag) -> Result<RawFd, Errno> {
    let range_start = 7u64;
    let (range_end, _) = getrlimit(Resource::RLIMIT_NOFILE)?;
    #[allow(clippy::unnecessary_cast)]
    let range_end = range_end.saturating_sub(1) as u64;

    // SAFETY: Cap to a sane maximum because sufficiently big values
    // of the hard limit tend to return ENOMEM.
    let range_end = range_end.min(0x10000);
    if range_end <= range_start {
        return Err(Errno::EMFILE);
    }
    let range = range_start..=range_end;

    // Close old fd if O_EXCL is given,
    // pass the rest of the flags to dup3.
    let close_old = flags.contains(OFlag::O_EXCL);
    flags.remove(OFlag::O_EXCL);

    // SAFETY: To make this file descriptor harder to spot by an
    // attacker we duplicate it to a random fd number.
    for _ in range.clone() {
        #[allow(clippy::cast_possible_truncation)]
        let fd_rand = randint(range.clone())? as RawFd;

        // SAFETY: fd only used after validation.
        let fd_rand = unsafe { BorrowedFd::borrow_raw(fd_rand) };

        // Check if the slot is free.
        // This is arguably subject to race but since this is solely
        // used for fds at startup, we don't really care.
        if is_active_fd(fd_rand) {
            continue;
        }

        match retry_on_eintr(|| {
            // SAFETY: In libc we trust.
            Errno::result(unsafe { dup3(oldfd, fd_rand.as_raw_fd(), flags.bits()) })
        }) {
            Ok(_) => {
                if close_old {
                    let _ = close(oldfd);
                }
                return Ok(fd_rand.as_raw_fd());
            }
            Err(Errno::EMFILE) => return Err(Errno::EMFILE),
            Err(_) => {}
        }
    }

    Err(Errno::EBADF)
}

/// Return a random unprivileged port number using the OS random number
/// generator.
#[allow(clippy::cast_possible_truncation)]
#[inline]
pub fn randport() -> Result<u16, Errno> {
    randint(1025..=0xFFFF).map(|port| port as u16)
}

/// Create a unique temporary file in `dirfd` relative to `prefix`
/// unlink the file and return its file descriptor. Unlike libc's
/// mkstemp(3) function the template here does not have to end with any
/// number of `X` characters. The function appends an implementation
/// defined number of random characters after `prefix`. `prefix` must
/// not start with the `/` character and not be longer than `PATH_MAX`
/// characters long. It is OK for prefix to be empty.
/// If `dirfd` supports the `O_TMPFILE` operation, an unnamed temporary
/// file is created instead with `O_TMPFILE|O_EXCL`.
pub fn mkstempat<Fd: AsFd>(dirfd: Fd, prefix: &XPath) -> Result<OwnedFd, Errno> {
    const MAX_TCOUNT: usize = 8;
    const SUFFIX_LEN: usize = 128;
    const CHARSET: &[u8] = b"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";

    // Step 1: Attempt to use O_TMPFILE|O_EXCL which is safer.
    let mut flags = OFlag::O_TMPFILE | OFlag::O_EXCL | OFlag::O_RDWR;
    match safe_open(&dirfd, XPath::from_bytes(b"."), flags, ResolveFlag::empty()) {
        Ok(fd) => return Ok(fd),
        Err(Errno::EISDIR | Errno::ENOENT | Errno::EOPNOTSUPP) => {}
        Err(errno) => return Err(errno),
    }

    // Step 2: Fallback to random name generation.
    flags.remove(OFlag::O_TMPFILE);
    flags.insert(OFlag::O_CREAT);
    if prefix.is_absolute() {
        return Err(Errno::EINVAL);
    } else if prefix.len().saturating_sub(SUFFIX_LEN) > PATH_MAX {
        return Err(Errno::ENAMETOOLONG);
    }

    let mut attempts = 0;
    let mut rng_data = [0u8; SUFFIX_LEN];
    #[allow(clippy::arithmetic_side_effects)]
    loop {
        attempts += 1;
        if attempts > MAX_TCOUNT {
            // Too many collisions.
            return Err(Errno::EEXIST);
        }

        // Fill with random bytes.
        fillrandom(&mut rng_data)?;

        // Map bytes to characters.
        let mut base = XPathBuf::with_capacity(prefix.len() + SUFFIX_LEN);
        base.append_bytes(prefix.as_bytes());
        for &b in &rng_data {
            base.append_byte(CHARSET[(b as usize) % CHARSET.len()]);
        }

        match safe_open(&dirfd, &base, flags, ResolveFlag::empty()) {
            Ok(fd) => {
                safe_unlinkat(dirfd, &base, UnlinkatFlags::NoRemoveDir)?;
                return Ok(fd);
            }
            Err(Errno::EEXIST) => {
                // Try again with a new random sequence.
                continue;
            }
            Err(errno) => return Err(errno),
        }
    }
}

/// Safe wrapper for faccessat2(2) with AT_EMPTY_PATH.
pub fn fdaccess<Fd: AsFd>(fd: Fd, mode: AccessFlags, mut flags: AtFlags) -> Result<(), Errno> {
    // Remove AT_SYMLINK_NOFOLLOW and add AT_EMPTY_PATH to flags.
    flags.remove(AtFlags::AT_SYMLINK_NOFOLLOW);
    flags.insert(AtFlags::AT_EMPTY_PATH);

    // SAFETY: No libc wrapper for faccessat2 yet.
    Errno::result(unsafe {
        syscall(
            SYS_faccessat2,
            fd.as_fd().as_raw_fd(),
            c"".as_ptr(),
            mode.bits(),
            flags.bits(),
        )
    })
    .map(drop)
}

// execveat(2): Only perform a check if execution would be allowed.
// Requires Linux>=6.14.
pub(crate) const AT_EXECVE_CHECK: AtFlags = AtFlags::from_bits_retain(0x10000);

/// Return true if the given File is executable.
#[inline]
pub fn is_executable<Fd: AsFd>(file: Fd) -> bool {
    check_executable(file).is_ok()
}

/// Check if the given File is executable.
pub fn check_executable<Fd: AsFd>(file: Fd) -> Result<(), Errno> {
    if *HAVE_AT_EXECVE_CHECK {
        let argv: [*const c_char; 2] = [c"".as_ptr(), std::ptr::null()];
        let envp: [*const c_char; 1] = [std::ptr::null()];
        // SAFETY: In libc we trust.
        Errno::result(unsafe {
            syscall(
                SYS_execveat,
                file.as_fd().as_raw_fd(),
                c"".as_ptr(),
                argv.as_ptr(),
                envp.as_ptr(),
                (AT_EXECVE_CHECK | AtFlags::AT_EMPTY_PATH).bits(),
            )
        })
        .map(drop)
    } else {
        fdaccess(file, AccessFlags::X_OK, AtFlags::AT_EACCESS)
    }
}

/// PIDFD_THREAD flag for pidfd_open(2).
#[allow(clippy::cast_sign_loss)]
pub const PIDFD_THREAD: u32 = OFlag::O_EXCL.bits() as u32;

/// Safe wrapper for pidfd_open(2).
///
/// This function requires Linux 5.3+.
pub fn pidfd_open(pid: Pid, mut flags: u32) -> Result<OwnedFd, Errno> {
    // Use PIDFD_THREAD if available.
    // Pass-through PIDFD_NONBLOCK.
    let pid = if *HAVE_PIDFD_THREAD || flags & PIDFD_THREAD == 0 {
        pid
    } else {
        flags &= !PIDFD_THREAD;
        proc_tgid(pid)?
    };

    // SAFETY: libc does not have a pidfd_open(2) wrapper yet.
    #[allow(clippy::cast_possible_truncation)]
    Errno::result(unsafe { syscall(SYS_pidfd_open, pid.as_raw(), flags) }).map(|fd| {
        // SAFETY: pidfd_open(2) returned success, fd is valid.
        unsafe { OwnedFd::from_raw_fd(fd as RawFd) }
    })
}

/// Safe wrapper for pidfd_getfd(2).
///
/// This function requires Linux 5.6+.
pub fn pidfd_getfd<Fd: AsFd>(pid_fd: Fd, remote_fd: RawFd) -> Result<OwnedFd, Errno> {
    // SAFETY: libc does not have a pidfd_getfd(2) wrapper yet.
    #[allow(clippy::cast_possible_truncation)]
    Errno::result(unsafe { syscall(SYS_pidfd_getfd, pid_fd.as_fd().as_raw_fd(), remote_fd, 0) })
        .map(|fd| {
            // SAFETY: pidfd_getfd(2) returned success, fd is valid.
            unsafe { OwnedFd::from_raw_fd(fd as RawFd) }
        })
}

/// Safe wrapper for pidfd_send_signal(2).
///
/// This function requires Linux 5.1+.
pub fn pidfd_send_signal<Fd: AsFd>(pid_fd: Fd, sig: i32) -> Result<(), Errno> {
    // SAFETY: libc does not have a wrapper for pidfd_send_signal yet.
    Errno::result(unsafe { syscall(SYS_pidfd_send_signal, pid_fd.as_fd().as_raw_fd(), sig, 0, 0) })
        .map(drop)
}

/// Safe wrapper for pidfd_send_signal(2) with signal 0.
///
/// This function requires Linux 5.1+.
#[inline]
pub fn pidfd_is_alive<Fd: AsFd>(pid_fd: Fd) -> Result<(), Errno> {
    pidfd_send_signal(pid_fd, 0)
}

/// libc may not define process_mrelease yet (e.g. musl on riscv64).
static SYS_PROCESS_MRELEASE: LazyLock<Option<c_long>> = LazyLock::new(|| {
    match ScmpSyscall::from_name("process_mrelease")
        .map(i32::from)
        .map(c_long::from)
        .ok()
    {
        Some(n) if n < 0 => None,
        Some(n) => Some(n),
        None => None,
    }
});

/// Safe wrapper for process_mrelease(2).
///
/// This function requires Linux 5.15+.
pub fn process_mrelease<Fd: AsFd>(pid_fd: Fd) -> Result<(), Errno> {
    let sysnum = SYS_PROCESS_MRELEASE.ok_or(Errno::ENOSYS)?;

    // SAFETY:
    // 1. libc does not have a wrapper for process_mrelease yet.
    // 2. libc may not define SYS_process_mrelease yet.
    Errno::result(unsafe { syscall(sysnum, pid_fd.as_fd().as_raw_fd(), 0) }).map(drop)
}

/// Safe wrapper for tgkill(2).
pub fn tgkill(tgid: Pid, tid: Pid, sig: i32) -> Result<(), Errno> {
    // SAFETY: There's no libc wrapper for tgkill.
    Errno::result(unsafe { syscall(SYS_tgkill, tgid.as_raw(), tid.as_raw(), sig) }).map(drop)
}

/// Safe wrapper for sigwaitinfo(2).
pub fn sigwaitinfo(set: &SigSet, info: Option<&mut siginfo_t>) -> Result<i32, Errno> {
    let info = info.map(|si| si as *mut _).unwrap_or(std::ptr::null_mut());

    // SAFETY: In libc we trust.
    Errno::result(unsafe { libc::sigwaitinfo(set.as_ref(), info) })
}

/// Safe wrapper for sigtimedwait(2).
pub fn sigtimedwait(
    set: &SigSet,
    info: Option<&mut siginfo_t>,
    timeout: TimeSpec,
) -> Result<i32, Errno> {
    let info = info.map(|si| si as *mut _).unwrap_or(std::ptr::null_mut());

    // SAFETY: In libc we trust.
    Errno::result(unsafe { libc::sigtimedwait(set.as_ref(), info, timeout.as_ref()) })
}

/// Convenience wrapper for sigtimedwait with zero TimeSpec.
pub fn sigtimedpoll(set: &SigSet, info: Option<&mut siginfo_t>) -> Result<i32, Errno> {
    sigtimedwait(set, info, TimeSpec::new(0, 0))
}

/// Convenience wrapper to block a single Signal.
pub fn block_signal(sig: Signal) -> Result<(), Errno> {
    let mut mask = SigSet::empty();
    mask.add(sig);
    mask.thread_block()
}

/// Convenience wrapper to unblock a single Signal.
pub fn unblock_signal(sig: Signal) -> Result<(), Errno> {
    let mut mask = SigSet::empty();
    mask.add(sig);
    mask.thread_unblock()
}

/// Safe wrapper for close_range(2).
pub fn close_range(first: c_uint, last: c_uint, flags: c_uint) -> Result<(), Errno> {
    // SAFETY: nix does not have a close_range wrapper yet.
    Errno::result(unsafe { syscall(SYS_close_range, first, last, flags) }).map(drop)
}

/// Close all file descriptors >= `fd`, equivalent to BSD's closefrom(2).
///
/// # Errors
///
/// Propagates any error returned by `close_range`.
pub fn closefrom(fd: c_uint) -> Result<(), Errno> {
    close_range(fd, RawFd::MAX as c_uint, 0)
}

/// Close all file descriptors except those in `exceptions`.
///
/// `exceptions` must be sorted ascending and contain no duplicates;
/// otherwise returns `Err(Errno::EINVAL)`.
///
/// Uses `close_range(2)` under the hood to efficiently close the
/// non-exempt descriptors.
///
/// # Errors
///
/// Returns on the first syscall error encountered, or
/// `Err(Errno::EINVAL)` if `exceptions` is not strictly ascending.
pub fn closeexcept(exceptions: &[c_uint]) -> Result<(), Errno> {
    // Validate that `exceptions` is strictly ascending and unique.
    if exceptions.windows(2).any(|w| w[0] >= w[1]) {
        return Err(Errno::EINVAL);
    }

    // If no exceptions, close everything.
    if exceptions.is_empty() {
        return closefrom(0);
    }

    // Use a wider integer for range computations to avoid overflow.
    let mut next: u64 = 0;

    #[allow(clippy::arithmetic_side_effects)]
    #[allow(clippy::cast_possible_truncation)]
    for &ex_fd in exceptions {
        let ex_fd = u64::from(ex_fd);

        // Close [next .. ex_fd - 1], if non-empty.
        if next < ex_fd {
            let first = next as c_uint;
            // Safe: ex_fd >= next + 1 ensures no underflow.
            let last = (ex_fd - 1) as c_uint;
            close_range(first, last, 0)?;
        }

        // Skip the exception itself.
        next = ex_fd.saturating_add(1);
    }

    // Finally close [next .. MAX_FD], if any remain.
    #[allow(clippy::cast_possible_truncation)]
    if next <= RawFd::MAX as u64 {
        let first = next as c_uint;
        closefrom(first)?;
    }

    Ok(())
}

/// Get peer credentials for the given UNIX socket.
pub fn peer_cred<Fd: AsFd>(fd: Fd) -> Result<UnixCredentials, Errno> {
    getsockopt(&fd, PeerCredentials)
}

/// Parse a FD from a Path.
#[inline]
pub fn parse_fd(path: &XPath) -> Result<RawFd, Errno> {
    btoi::<RawFd>(path.as_bytes()).or(Err(Errno::EBADF))
}

/// Read a symbolic link and return a `XPathBuf`.
// TODO: Move to compat.rs
pub fn readlinkat<Fd: AsFd>(fd: Fd, base: &XPath) -> Result<XPathBuf, Errno> {
    nix::fcntl::readlinkat(fd, base).map(XPathBuf::from)
}

/// Read a symbolic link from FD and return a `XPathBuf`.
pub fn readlinkfd<Fd: AsFd>(fd: Fd) -> Result<XPathBuf, Errno> {
    let mut target = XPathBuf::with_capacity(PATH_MAX);

    // SAFETY: We don't want nix' extra handling around readlink here.
    #[allow(clippy::cast_sign_loss)]
    let n = Errno::result(unsafe {
        libc::readlinkat(
            fd.as_fd().as_raw_fd(),
            c"".as_ptr(),
            target.as_mut_ptr().cast(),
            target.capacity(),
        )
    })
    .map(|n| n as usize)
    .map_err(|errno| {
        // FD-only readlinkat(2) returns ENOENT,
        // when FD is not a symbolic link.
        if errno == Errno::ENOENT {
            Errno::EINVAL
        } else {
            errno
        }
    })?;

    if n >= target.capacity() {
        // Truncation occurred!
        return Err(Errno::ENAMETOOLONG);
    }

    // SAFETY: readlink returned success, target is populated.
    unsafe { target.set_len(n) };
    target.shrink_to_fit();

    Ok(target)
}

/// Create file and write the given content.
#[allow(clippy::disallowed_methods)]
pub fn cat<P: AsRef<Path>, T: AsRef<[u8]>>(path: P, content: T) -> std::io::Result<()> {
    let mut file = File::create(path)?;
    file.write_all(content.as_ref())?;
    Ok(())
}

/// Make a file executable.
pub fn chmod_x<P: AsRef<Path>>(path: P) -> std::io::Result<()> {
    // Set permissions to make path executable.
    let metadata = metadata(path.as_ref())?;
    let mut permissions = metadata.permissions();
    permissions.set_mode(0o700); // This sets the file executable for the owner (rwx).
    set_permissions(path.as_ref(), permissions)
}

#[derive(Debug, PartialEq)]
enum PathComponent {
    //We handle {Root,Cur}Dir transparently for efficiency.
    //RootDir,
    //CurDir,
    ParentDir,
    Normal(XPathBuf),
}

impl Serialize for PathComponent {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        // SAFETY: Convert to XPath to mask control characters in log output.
        let comp = match self {
            Self::Normal(p) => p.to_owned(),
            Self::ParentDir => XPathBuf::from(".."),
        };

        serializer.serialize_str(&comp.to_string())
    }
}

/// Generates a VecDeque of path components, optimized for performance and correctness.
/// - Assumes handling of the initial RootDir is done by the caller.
/// - Directly walks the path bytes to construct components, efficiently skipping multiple leading slashes.
/// - Returns an additional bool to check for trailing slash.
#[inline]
fn path_components(path: &XPath) -> Result<(VecDeque<PathComponent>, bool), Errno> {
    // Create a new searcher for the '/' character.
    let searcher = memchr::arch::all::memchr::One::new(b'/');

    let path_bytes = path.as_os_str().as_bytes();

    let mut components = VecDeque::new();
    let comp_count = searcher.iter(path_bytes).count().min(PATH_MAX_COMP);
    components.try_reserve(comp_count).or(Err(Errno::ENOMEM))?;

    let mut start = 0;
    #[allow(clippy::arithmetic_side_effects)]
    for pos in searcher.iter(path_bytes) {
        if pos > start {
            match &path_bytes[start..pos] {
                b"." => {} // Skip current directory component.
                b".." => components.push_back(PathComponent::ParentDir),
                component => {
                    if component.len() >= PATH_MAX {
                        return Err(Errno::ENAMETOOLONG);
                    }
                    components.push_back(PathComponent::Normal(component.into()));
                }
            }
        }
        start = pos + 1; // Move past the '/'.
    }

    // Handle the last component if it's not ended by a slash.
    #[allow(clippy::arithmetic_side_effects)]
    let slash = if start < path_bytes.len() {
        match &path_bytes[start..] {
            b"." => {
                // Skip current directory component.
                components.is_empty() || (start >= 1 && path_bytes[start - 1] == b'/')
            }
            b".." => {
                components.push_back(PathComponent::ParentDir);
                true
            }
            component => {
                if component.len() >= PATH_MAX {
                    return Err(Errno::ENAMETOOLONG);
                }
                components.push_back(PathComponent::Normal(component.into()));
                false
            }
        }
    } else {
        true
    };

    Ok((components, slash))
}

/// Adds to a VecDeque of (owning) path components, optimized for performance and correctness.
/// - Assumes handling of the initial RootDir is done by the caller.
/// - Directly walks the path bytes to construct components, efficiently skipping multiple leading slashes.
#[inline]
fn path_components2(path: &XPath, components: &mut VecDeque<PathComponent>) -> Result<(), Errno> {
    let path_bytes = path.as_os_str().as_bytes();

    // Create a new searcher for the '/' character.
    let searcher = memchr::arch::all::memchr::One::new(b'/');

    let mut last_pos = path_bytes.len();
    let mut last_component = true;
    #[allow(clippy::arithmetic_side_effects)]
    for pos in searcher.iter(path_bytes).rev() {
        match &path_bytes[pos + 1..last_pos] {
            b"" | b"." => {} // Skip current directory and empty components.
            b".." => {
                components.try_reserve(1).or(Err(Errno::ENOMEM))?;
                components.push_front(PathComponent::ParentDir);
            }
            component => {
                let mut component: XPathBuf = component.into();
                if last_component && path.ends_with(b"/") {
                    component.append_byte(b'/');
                }
                last_component = false;

                components.try_reserve(1).or(Err(Errno::ENOMEM))?;
                components.push_front(PathComponent::Normal(component));
            }
        }
        last_pos = pos;
    }

    // Handle the remaining component before the first slash (or the
    // only component if no slashes).
    match &path_bytes[..last_pos] {
        b"" | b"." => {} // Skip current directory and empty components.
        b".." => {
            components.try_reserve(1).or(Err(Errno::ENOMEM))?;
            components.push_front(PathComponent::ParentDir);
        }
        component => {
            let mut component: XPathBuf = component.into();
            if last_component && path.ends_with(b"/") {
                component.append_byte(b'/');
            }

            components.try_reserve(1).or(Err(Errno::ENOMEM))?;
            components.push_front(PathComponent::Normal(component));
        }
    }

    Ok(())
}

/// Gets current working directory handling arbitrarily long pathnames.
///
/// Safety: This function does not change the current working directory.
#[allow(clippy::arithmetic_side_effects)]
pub fn getdir_long<Fd: AsFd>(fd: Fd, max_components: usize) -> Result<XPathBuf, Errno> {
    // Record information on current directory.
    let mut pinfo = FileInfo::from_fd(&fd)?;

    let mut dir: Box<dyn AsFd> = Box::new(fd);
    let mut cwd = Vec::new();
    cwd.try_reserve(PATH_MAX).or(Err(Errno::ENOMEM))?;

    let flags = (OFlag::O_RDONLY
        | OFlag::O_CLOEXEC
        | OFlag::O_DIRECTORY
        | OFlag::O_LARGEFILE
        | OFlag::O_NOCTTY
        | OFlag::O_NOFOLLOW)
        .bits();

    let mut i = 0;
    while i < max_components {
        // Move one directory level up.
        let fd = retry_on_eintr(|| {
            // SAFETY: We do confine the string pointer here, so we cannot use nix.
            Errno::result(unsafe {
                openat(
                    dir.as_fd().as_raw_fd(),
                    dotdot_with_nul() as *const c_char,
                    flags,
                    0,
                )
            })
        })
        .map(|fd| {
            // SAFETY: openat returns a valid FD on success.
            unsafe { OwnedFd::from_raw_fd(fd) }
        })?;
        dir = Box::new(fd);

        let info = FileInfo::from_fd(&dir)?;
        if info == pinfo {
            // Current file information and parent are the same:
            // We have reached the root directory.
            let cwd = if !cwd.is_empty() {
                // Re-reverse the path and return.
                cwd.reverse();
                cwd.into()
            } else {
                // LOL: This was rootfs to begin with!
                XPathBuf::from("/")
            };
            return Ok(cwd);
        }

        let mut dot = 0u8;
        let mut found = false;
        let new_device = info.mnt != pinfo.mnt;
        'main: loop {
            let mut entries = match getdents64(&dir, DIRENT_BUF_SIZE) {
                Ok(entries) => entries,
                Err(Errno::ECANCELED) => break, // EOF or empty directory
                Err(errno) => return Err(errno),
            };

            for entry in &mut entries {
                if dot < 2 && entry.is_dot() {
                    dot += 1;
                    continue;
                } else if !new_device && entry.ino() != pinfo.ino {
                    // This is not our directory entry, continue.
                    continue;
                } else if pinfo
                    != statx(
                        &dir,
                        entry.as_xpath(),
                        AT_SYMLINK_NOFOLLOW,
                        FileInfo::mask(),
                    )
                    .map(FileInfo::from_statx)?
                {
                    // This is not our directory entry as verified by device & inode check.
                    continue;
                }

                // Found our entry!
                found = true;

                // Record parent information.
                pinfo = info;

                // Push the name reversed,
                // we'll re-reverse at the end.
                cwd.try_reserve(entry.name_bytes().len().saturating_add(1))
                    .or(Err(Errno::ENOMEM))?;
                cwd.extend(entry.name_bytes().iter().rev());
                cwd.push(b'/');

                break 'main;
            }
        }

        if found {
            i += 1;
        } else {
            return Err(Errno::ENOENT);
        }
    }

    Err(Errno::ERANGE)
}

/// An enum that may either be an OwnedFd or BorrowedFd.
pub enum MaybeFd {
    /// An `OwnedFd` which will be closed when `MaybeFd` is closed.
    Owned(OwnedFd),
    /// A `RawFd` that will remain open at least as long as `MaybeFd`.
    RawFd(RawFd),
}

impl Clone for MaybeFd {
    fn clone(&self) -> Self {
        match self {
            MaybeFd::Owned(fd) => MaybeFd::RawFd(fd.as_raw_fd()),
            MaybeFd::RawFd(fd) => MaybeFd::RawFd(*fd),
        }
    }
}

impl AsFd for MaybeFd {
    fn as_fd(&self) -> BorrowedFd<'_> {
        match self {
            MaybeFd::Owned(owned) => owned.as_fd(),
            // SAFETY: RawFd is alive as long as MaybeFd.
            MaybeFd::RawFd(fd) => unsafe { BorrowedFd::borrow_raw(*fd) },
        }
    }
}

impl AsRawFd for MaybeFd {
    fn as_raw_fd(&self) -> RawFd {
        match self {
            MaybeFd::Owned(owned) => owned.as_raw_fd(),
            MaybeFd::RawFd(fd) => *fd,
        }
    }
}

impl From<OwnedFd> for MaybeFd {
    fn from(fd: OwnedFd) -> Self {
        MaybeFd::Owned(fd)
    }
}

impl From<RawFd> for MaybeFd {
    fn from(fd: RawFd) -> Self {
        MaybeFd::RawFd(fd)
    }
}

impl fmt::Debug for MaybeFd {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            MaybeFd::Owned(_) => f.debug_tuple("OwnedFd").field(&self.as_raw_fd()).finish(),
            MaybeFd::RawFd(_) => f.debug_tuple("RawFd").field(&self.as_raw_fd()).finish(),
        }
    }
}

#[derive(Debug, Clone)]
pub(crate) struct FileMapEntry {
    fd: MaybeFd,
    f_type: Option<FileType>,
    f_mode: Option<u16>,
    mnt_id: Option<u64>,
    target: Option<Result<XPathBuf, Errno>>,
    nvisit: u8, // symlink visit count, used for loop detection.
}

#[derive(Debug, Clone)]
pub(crate) struct FileMap(pub(crate) SydHashMap<XPathBuf, FileMapEntry>);

impl FileMapEntry {
    pub(crate) fn new(
        fd: MaybeFd,
        f_type: Option<FileType>,
        f_mode: Option<u16>,
        mnt_id: Option<u64>,
        target: Option<Result<XPathBuf, Errno>>,
    ) -> Self {
        Self {
            fd,
            f_type,
            f_mode,
            mnt_id,
            target,
            nvisit: 0,
        }
    }

    #[allow(clippy::cognitive_complexity)]
    fn from_magic_link(
        magic: ProcMagic,
        sandbox: Option<&Sandbox>,
    ) -> Result<(Self, XPathBuf), Errno> {
        // Create path to symlink.
        let remote_sym = magic.link_path();

        // Step 1: Open directory using the magic symlink.
        let flags = if magic.want_dir() {
            OFlag::O_PATH | OFlag::O_DIRECTORY
        } else {
            OFlag::O_PATH
        };

        let fd = safe_open_msym(PROC_FILE(), &remote_sym, flags, ResolveFlag::empty())?;
        let sym = XPathBuf::from_self_fd(fd.as_raw_fd());

        // Step 2: Try to readlink().
        let (target, is_deleted, is_mfd) = match readlinkat(PROC_FILE(), &sym) {
            Ok(mut p) => {
                if p.is_relative() {
                    // /proc/1/fd/42 -> pipe:[4242], socket:[4242]
                    if magic.want_dir() {
                        return Err(Errno::EBADF);
                    }

                    // Magic link points to magic path.
                    // Paths such as [pipe:64], [socket:42] etc.
                    // 1. /proc/$pid/exe
                    // 2. /proc/$pid/fd/$fd
                    (Some(p), false, false)
                } else if p.ends_with(b" (deleted)") {
                    if p.starts_with(b"/memfd:") {
                        // Memory file descriptor:
                        // 1. Truncate deleted suffix
                        // 2. Change root slash to !
                        // This is used in access check as
                        // !memfd:* or !memfd-hugetlb:*
                        p.truncate(
                            p.len()
                                .checked_sub(b" (deleted)".len())
                                .ok_or(Errno::EOVERFLOW)?,
                        );
                        p.set(0, b'!');
                        (Some(p), false, true)
                    } else {
                        // /proc/1/cwd -> /path/to/foo (deleted)
                        //
                        // This may be a deleted file,
                        // or a file whose name funnily
                        // ends with a genuine " (deleted)" string.
                        //
                        // We will statx(2) and check nlink to tell.
                        (Some(p), true, false)
                    }
                } else {
                    // Genuine file descriptor that points to a valid directory.
                    (Some(p), false, false)
                }
            }
            Err(Errno::ENAMETOOLONG) => (None, false, false),
            Err(errno) => return Err(errno),
        };

        // Ensure magic symlink is /proc prefixed.
        let mut sym = XPathBuf::from("/proc");
        sym.push(remote_sym.as_bytes());

        if let Some(mut target) = target {
            // SAFETY: Prevent hidden path pass-through via magic symlinks.
            // Passing through root is OK.
            if !target.is_root() && target.is_absolute() {
                if let Some(sandbox) = sandbox {
                    sandbox_path(
                        None,
                        sandbox,
                        magic.pid(),
                        target.deref(),
                        Capability::CAP_WALK,
                        false,
                        "walk",
                    )?;
                }
            }

            // We managed to successfully readlink.
            //
            // Check for deleted entries.
            if is_deleted {
                let stx = retry_on_eintr(|| fstatx(&fd, STATX_TYPE | STATX_NLINK))?;
                let f_type = FileType::from(mode_t::from(stx.stx_mode));

                if stx.stx_nlink == 0 {
                    // File is deleted, strip the deleted suffix.
                    target.truncate(
                        target
                            .len()
                            .checked_sub(b" (deleted)".len())
                            .ok_or(Errno::EOVERFLOW)?,
                    );
                }

                let entry = Self {
                    fd: fd.into(),
                    f_type: Some(f_type),
                    f_mode: None,
                    mnt_id: None,
                    target: Some(Ok(target)),
                    nvisit: 0,
                };

                return Ok((entry, sym));
            }

            let f_type = if is_mfd {
                // Check for MFD_HUGETLB.
                if is_huge_file(&fd).unwrap_or(false) {
                    target.replace_prefix(MFD_NAME_PREFIX, MFD_HUGETLB_NAME_PREFIX)?;
                }
                Some(FileType::Mfd)
            } else if magic.want_dir() {
                Some(FileType::Dir)
            } else {
                file_type(&fd, None, false).ok()
            };

            let entry = Self {
                fd: fd.into(),
                f_type,
                f_mode: None,
                mnt_id: None,
                target: Some(Ok(target)),
                nvisit: 0,
            };

            return Ok((entry, sym));
        }

        // Step 3: Use getdir_long().

        // SAFETY:
        // 1. getdir_long() does not change CWD!
        // 2. We limit maximum dir components to PATH_MAX_COMP so the
        //    sandbox process cannot create arbitrarily long directories
        //    and crash Syd.
        let target = getdir_long(&fd, PATH_MAX_COMP)?;

        // SAFETY: Prevent hidden path pass-through via magic symlinks.
        // Part 2: Prevent bypass via too-long paths.
        // Passing through root is OK.
        if !target.is_root() && target.is_absolute() {
            if let Some(sandbox) = sandbox {
                sandbox_path(
                    None,
                    sandbox,
                    magic.pid(),
                    target.deref(),
                    Capability::CAP_WALK,
                    false,
                    "walk",
                )?;
            }
        }

        let entry = Self {
            fd: fd.into(),
            f_type: Some(FileType::Dir),
            f_mode: None,
            mnt_id: None,
            target: Some(Ok(target)),
            nvisit: 0,
        };

        Ok((entry, sym))
    }

    // Return the file information of an entry safely.
    // This is currently only used for symlinks post-readlink
    // for symbolic link detection.
    #[allow(clippy::cast_possible_truncation)]
    fn get_mode(&mut self) -> Result<u16, Errno> {
        // Return mode if we've saved it before.
        if let Some(mode) = self.f_mode {
            return Ok(mode);
        }

        // All done, fstat the fd of the entry.
        let stx = retry_on_eintr(|| fstatx(&self.fd, STATX_TYPE | STATX_MODE))?;

        // Keep magic link information for readlink to consume later.
        if !self.is_magic_link() {
            let f_type = FileType::from(mode_t::from(stx.stx_mode));
            self.f_type = Some(f_type);
        }

        // Save and return file mode.
        let mode = stx.stx_mode & !(S_IFMT as u16);
        self.f_mode = Some(mode);

        Ok(mode)
    }

    #[inline]
    fn is_magic_link(&self) -> bool {
        matches!(self.f_type, Some(FileType::MagicLnk(_)))
    }
}

impl FileMap {
    pub(crate) fn with_capacity(cap: usize) -> Self {
        Self(SydHashMap::with_capacity(cap))
    }

    /*
    fn len(&self) -> usize {
        self.0.len()
    }
    */

    fn remove(&mut self, path: &XPath) -> Option<FileMapEntry> {
        self.0.remove(path)
    }

    fn get(&self, path: &XPath) -> Option<&FileMapEntry> {
        self.0.get(path)
    }

    fn open(
        &mut self,
        pid: Pid,
        path: &XPath,
        options: FsFlags,
        filetyp: Option<FileType>,
        sandbox: Option<&Sandbox>,
    ) -> Result<&mut FileMapEntry, Errno> {
        // SAFETY: Opening a component which is hidden
        // would let us obtain metadata and then
        // resolve through it. Reject early.
        // Passing through root is OK.
        if !path.is_root() {
            if let Some(sandbox) = sandbox {
                sandbox_path(
                    None,
                    sandbox,
                    pid,
                    path,
                    Capability::CAP_WALK,
                    false,
                    "walk",
                )?;
            }
        }

        // SAFETY: rootfs and procfs views must be identical!
        let (parent, base) = path.split();
        let (parent_fd, base) = if let Some(parent_entry) = self.0.get(parent) {
            (parent_entry.fd.as_fd(), base)
        } else if path.starts_with(b"/proc/") {
            // Avoid multi-component resolution under /proc.
            // Open the immediate parent under /proc as O_PATH|O_DIRECTORY,
            // then open the last component relative to that FD.
            let pfd = if let Some(pfd) = self.0.get(parent) {
                pfd.fd.as_fd()
            } else {
                // Open /proc-relative parent in a single hop.
                let parent = XPath::from_bytes(&parent.as_bytes()[b"/proc/".len()..]);
                let dfd: MaybeFd = if parent.is_empty() {
                    PROC_FD().into()
                } else {
                    safe_open_path(
                        PROC_FILE(),
                        parent,
                        OFlag::O_DIRECTORY,
                        ResolveFlag::empty(),
                    )?
                    .into()
                };

                // Attempt to reserve memory or bail.
                self.0.try_reserve(1).or(Err(Errno::ENOMEM))?;

                // All good, insert entry and return FD.
                self.0.insert(
                    parent.to_owned(),
                    FileMapEntry::new(dfd, Some(FileType::Dir), None, None, None),
                );
                // We just inserted, unwrap must return Some.
                #[allow(clippy::disallowed_methods)]
                {
                    self.0.get(parent).unwrap().fd.as_fd()
                }
            };

            (pfd, base)
        } else {
            let base = XPath::from_bytes(&path.as_bytes()[1..]);
            (ROOT_FILE(), base)
        };

        // SAFETY:
        // 1. Force RESOLVE_NO_XDEV as necessary.
        // 2. RESOLVE_BENEATH | RESOLVE_NO_MAGICLINKS | RESOLVE_SYMLINKS
        //    are already enforced by safe_open_path.
        let rflags = if options.contains(FsFlags::NO_RESOLVE_XDEV) {
            ResolveFlag::RESOLVE_NO_XDEV
        } else {
            ResolveFlag::empty()
        };
        let fd = safe_open_path(parent_fd, base, OFlag::O_NOFOLLOW, rflags)?;

        // Attempt to reserve memory or bail.
        self.0.try_reserve(1).or(Err(Errno::ENOMEM))?;

        // All good, insert entry and return FD.
        let entry = FileMapEntry::new(fd.into(), filetyp, None, None, None);
        self.0.insert(path.to_owned(), entry);

        self.0.get_mut(path).ok_or(Errno::ENOENT)
    }

    // Resolve a symbolic link honouring magic proc links.
    #[allow(clippy::cognitive_complexity)]
    fn readlink(
        &mut self,
        path: &XPath,
        pid: Pid,
        options: FsFlags,
        filetyp: Option<FileType>,
        sandbox: Option<&Sandbox>,
    ) -> Result<XPathBuf, Errno> {
        if path.is_static() {
            // Static paths are not symlinks.
            return Err(Errno::EINVAL);
        }

        // Check if we have queried this file before.
        let entry = if let Some(entry) = self.0.get_mut(path) {
            match entry.target.as_ref() {
                Some(Ok(target)) => {
                    #[allow(clippy::arithmetic_side_effects)]
                    return if entry.nvisit > 16 {
                        // Symbolic link loop detected!
                        // TODO: This is deterministic and needs to be improved.
                        Err(Errno::ELOOP)
                    } else {
                        entry.nvisit += 1;
                        Ok(target.to_owned())
                    };
                }
                Some(Err(errno)) => {
                    // File is not a symbolic link.
                    return Err(*errno);
                }
                None => entry,
            }
        } else {
            self.open(pid, path, options, filetyp, sandbox)?
        };

        // All done, resolve symbolic link.
        let target = match readlinkfd(&entry.fd) {
            Ok(target) => target,
            Err(errno) => {
                // Mark resolve fail in target.
                entry.target = Some(Err(errno));
                return Err(errno);
            }
        };

        if !options.resolve_path() {
            // RESOLVE_NO_SYMLINKS -> ELOOP
            return Err(Errno::ELOOP);
        }

        if target.is_absolute() && options.contains(FsFlags::RESOLVE_BENEATH) {
            // RESOLVE_BENEATH -> EXDEV
            return Err(Errno::EXDEV);
        }

        // File is known to be a symlink, save target.
        entry.target = Some(Ok(target.clone()));

        // SAFETY:
        // Restrictions a la CONFIG_GRKERNSEC_LINK:
        // Do not follow symbolic links if the parent directory
        // has either one of the following:
        // 1. sticky-bit set.
        // 2. world-writable.
        // 3. group-writable.
        let restrict_symlinks = sandbox
            .map(|sb| !sb.flags.allow_unsafe_symlinks())
            .unwrap_or(false);
        if restrict_symlinks {
            // FIXME: This check is going to be bypassed
            // in case parent is not open yet, which is
            // indeed possible under some circumstances.
            #[allow(clippy::cast_possible_truncation)]
            if let Some(entry) = self.0.get_mut(path.parent()) {
                let mut err = None;
                let mode: u32 = entry.get_mode()?.into();
                if mode & S_ISVTX != 0 {
                    err = Some("parent directory has sticky bit set");
                } else if mode & S_IWOTH != 0 {
                    err = Some("parent directory is world writable");
                } else if mode & S_IWGRP != 0 {
                    err = Some("parent directory is group writable");
                }

                if let Some(msg) = err {
                    error!("ctx": "trusted_symlinks", "path": path,
                        "pid": pid.as_raw(), "rflags": options,
                        "msg": format!("follow for untrusted symlink blocked: {msg}"),
                        "tip": "fix parent directory permissions or use `trace/allow_unsafe_symlinks:1'");
                    return Err(Errno::ELOOP);
                }
            }
        }

        // RESOLVE_NO_MAGICLINKS is handled by ProcMagic::check_link() in canonicalize().
        Ok(target)
    }

    // Return the mount id and file type of a path safely.
    #[allow(clippy::cast_possible_truncation)]
    fn get_mnt_id(
        &mut self,
        path: &XPath,
        pid: Pid,
        options: FsFlags,
        filetyp: Option<FileType>,
        sandbox: Option<&Sandbox>,
    ) -> Result<u64, Errno> {
        let mut mask = STATX_TYPE;
        mask |= if *HAVE_STATX_MNT_ID_UNIQUE {
            STATX_MNT_ID_UNIQUE
        } else {
            STATX_MNT_ID
        };

        // If we have queried this file before, reuse information.
        let entry = if let Some(entry) = self.0.get_mut(path) {
            entry
        } else {
            self.open(pid, path, options, filetyp, sandbox)?
        };

        // Return mount id if we've saved it before.
        if let Some(mnt_id) = entry.mnt_id {
            return Ok(mnt_id);
        }

        // All done, fstat the fd of the entry.
        let stx = retry_on_eintr(|| fstatx(&entry.fd, mask))?;

        // Keep magic link information for readlink to consume later.
        if !entry.is_magic_link() {
            let f_type = FileType::from(mode_t::from(stx.stx_mode));
            entry.f_type = Some(f_type);
        }

        // Save and return mount id.
        let mnt_id = stx.stx_mnt_id;
        entry.mnt_id = Some(mnt_id);

        Ok(mnt_id)
    }
}

impl Default for FileMap {
    fn default() -> Self {
        FILEMAP.clone()
    }
}

/// Return value of `safe_canonicalize`.
#[derive(Debug)]
#[allow(dead_code)]
pub struct CanonicalPath<'a> {
    // Canonical, absolute form of path.
    //
    // SAFETY: This must solely used in access check,
    // and must never be passed as an argument to
    // system calls. Failing to do so will very likely
    // result in a TOCTTOU vulnerability.
    abs: XPathBuf,

    /// The final, base, component of the path.
    ///
    /// This is a reference into the `abs` element.
    /// This may be empty in which case `dir` must be used.
    pub base: &'a XPath,

    /// An `O_PATH` file descriptor to the owning directory.
    ///
    /// SAFETY: This is `Some` for all paths except the
    /// root path, ie `/`, which has no owning directory.
    pub dir: Option<MaybeFd>,

    /// File type information, if available.
    pub typ: Option<FileType>,
}

impl fmt::Display for CanonicalPath<'_> {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        // SAFETY: XPathBuf's Display masks control characters in path.
        write!(f, "{}", self.abs())
    }
}

impl Serialize for CanonicalPath<'_> {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        let mut map = serializer.serialize_map(Some(4))?;

        map.serialize_entry("abs", &self.abs)?;
        if self.base.is_empty() {
            map.serialize_entry("fd", &self.dir.as_ref().map(|fd| fd.as_raw_fd()))?;
        } else {
            map.serialize_entry("dir", &self.dir.as_ref().map(|fd| fd.as_raw_fd()))?;
            map.serialize_entry("base", self.base)?;
        }
        map.serialize_entry("type", &self.typ)?;

        let owned = matches!(self.dir, Some(MaybeFd::Owned(_)));
        map.serialize_entry("owned", &owned)?;

        map.end()
    }
}

impl<'a> CanonicalPath<'a> {
    /// Test whether file type represents a regular file.
    pub fn is_file(&self) -> bool {
        self.typ.as_ref().map(|typ| typ.is_file()).unwrap_or(false)
    }

    /// Test whether file type represents a directory.
    pub fn is_dir(&self) -> bool {
        self.typ.as_ref().map(|typ| typ.is_dir()).unwrap_or(false)
    }

    /// Test whether file type represents a symlink.
    pub fn is_symlink(&self) -> bool {
        self.typ
            .as_ref()
            .map(|typ| typ.is_symlink())
            .unwrap_or(false)
    }

    /// Test whether file type represents a magic link.
    pub fn is_magic_link(&self) -> bool {
        self.typ
            .as_ref()
            .map(|typ| typ.is_magic_link())
            .unwrap_or(false)
    }

    /// Test whether file type represents a magic directory.
    ///
    /// SAFETY/TODO: Used to harden magic link calls with O_DIRECTORY.
    pub fn is_magic_dir(&self) -> bool {
        self.typ
            .as_ref()
            .map(|typ| typ.is_magic_dir())
            .unwrap_or(false)
    }

    /// Test whether file type represents a block device.
    pub fn is_block_device(&self) -> bool {
        self.typ
            .as_ref()
            .map(|typ| typ.is_block_device())
            .unwrap_or(false)
    }

    /// Test whether file type represents a character device.
    pub fn is_char_device(&self) -> bool {
        self.typ
            .as_ref()
            .map(|typ| typ.is_char_device())
            .unwrap_or(false)
    }

    /// Test whether file type represents a FIFO.
    pub fn is_fifo(&self) -> bool {
        self.typ.as_ref().map(|typ| typ.is_fifo()).unwrap_or(false)
    }

    /// Test whether file type represents a socket.
    pub fn is_socket(&self) -> bool {
        self.typ
            .as_ref()
            .map(|typ| typ.is_socket())
            .unwrap_or(false)
    }

    /// Test whether file type represents an unknown file.
    pub fn is_unknown(&self) -> bool {
        self.typ
            .as_ref()
            .map(|typ| typ.is_unknown())
            .unwrap_or(false)
    }

    /// Returns a reference to the absolute form of path.
    ///
    /// SAFETY: This must solely used in access check,
    /// and must never be passed as an argument to
    /// system calls. Failing to do so will very likely
    /// result in a TOCTTOU vulnerability.
    #[inline]
    pub fn abs(&self) -> &XPath {
        XPath::from_bytes(self.abs.as_bytes())
    }

    /// Consume the `CanonicalPath` and return the absolute path.
    pub fn take(self) -> XPathBuf {
        self.abs
    }

    /// Create a new `CanonicalPath`.
    pub fn new(abs: XPathBuf, typ: FileType, options: FsFlags) -> Result<CanonicalPath<'a>, Errno> {
        // SAFETY: rootfs, devfs, procfs and sysfs views must be identical!
        if abs.is_rootfs() {
            return Ok(Self::new_root());
        } else if abs.is_procfs() {
            return Ok(Self::new_proc());
        } else if abs.is_equal(b"/dev/null") {
            return Ok(Self::new_null());
        }

        // SAFETY:
        // 1. Use NONBLOCK with non-path fds to avoid blocking on pipes, FIFOs etc.
        // 2. Use NOCTTY to avoid acquiring controlling terminal.
        let flags = if options.contains(FsFlags::WANT_READ) {
            OFlag::O_RDONLY | OFlag::O_NONBLOCK | OFlag::O_NOCTTY
        } else if typ.is_dir() {
            // SAFETY: Assert known directory with O_DIRECTORY.
            OFlag::O_PATH | OFlag::O_DIRECTORY
        } else {
            OFlag::O_PATH
        };

        // SAFETY: safe_open does not follow symlinks.
        let fd = safe_open(AT_FDCWD, &abs, flags, ResolveFlag::empty())?;

        Ok(Self {
            abs,
            base: XPath::from_bytes(b""),
            dir: Some(fd.into()),
            typ: Some(typ),
        })
    }

    /// Create a new `CanonicalPath` for (saved) `/`.
    ///
    /// This fd is opened at startup.
    pub fn new_root() -> CanonicalPath<'a> {
        Self {
            abs: XPathBuf::from("/"),
            base: XPath::from_bytes(b""),
            dir: Some(ROOT_FD().into()),
            typ: Some(FileType::Dir),
        }
    }

    /// Create a new `CanonicalPath` for (saved) `/proc`.
    ///
    /// This fd is opened at startup.
    pub fn new_proc() -> CanonicalPath<'a> {
        Self {
            abs: XPathBuf::from("/proc"),
            base: XPath::from_bytes(b""),
            dir: Some(PROC_FD().into()),
            typ: Some(FileType::Dir),
        }
    }

    /// Create a new `CanonicalPath` for a magic file.
    ///
    /// Magic files are virtual files that do not have a filesystem part,
    /// and therefore are not subject to TOCTTOU.
    pub fn new_magic(virtual_path: XPathBuf) -> CanonicalPath<'a> {
        Self {
            abs: virtual_path,
            base: XPath::from_bytes(b""),
            dir: Some(NULL_FD().into()),
            typ: Some(FileType::Chr),
        }
    }

    /// Create a new `CanonicalPath` for a `/dev/null'.
    pub fn new_null() -> CanonicalPath<'a> {
        Self {
            abs: XPathBuf::from("/dev/null"),
            base: XPath::from_bytes(b""),
            dir: Some(NULL_FD().into()),
            typ: Some(FileType::Chr),
        }
    }

    /// Create a new `CanonicalPath` for the given mask path.
    pub fn new_mask(mask: &XPath, abs: &XPath) -> Result<CanonicalPath<'a>, Errno> {
        let base = XPath::from_bytes(&mask.as_bytes()[b"/".len()..]);
        let fd = safe_open_path(ROOT_FILE(), base, OFlag::O_NOFOLLOW, ResolveFlag::empty())?;
        let typ = file_type(&fd, None, false)?;

        Ok(Self {
            abs: abs.to_owned(), // Points to the original path.
            base: XPath::from_bytes(b""),
            dir: Some(fd.into()), // Points to the mask path.
            typ: Some(typ),
        })
    }

    /// Create a new `CanonicalPath` from an absolute tty name.
    pub fn new_tty(abs: XPathBuf) -> Result<CanonicalPath<'a>, Errno> {
        assert!(
            abs.is_dev(),
            "BUG: non /dev path passed to CanonicalPath::new_tty!"
        );

        let base = XPath::from_bytes(&abs.as_bytes()[b"/".len()..]);
        let fd = safe_open_path(ROOT_FILE(), base, OFlag::O_NOFOLLOW, ResolveFlag::empty())?;

        Ok(Self {
            abs,
            base: XPath::empty(),
            dir: Some(fd.into()),
            typ: Some(FileType::Chr),
        })
    }

    /// Create a new `CanonicalPath` from a file descriptor.
    ///
    /// The return value has `abs` field populated and
    /// may be used in access checks.
    pub fn new_fd(mut fd: MaybeFd, pid: Pid) -> Result<CanonicalPath<'a>, Errno> {
        // Step 1: Open CWD using the magic symlink.
        let cwd = fd.as_raw_fd() == libc::AT_FDCWD;
        if cwd {
            let flags = OFlag::O_PATH | OFlag::O_DIRECTORY;

            let mut sym = XPathBuf::from_pid(pid);
            sym.push(b"cwd");

            fd = safe_open_msym(PROC_FILE(), &sym, flags, ResolveFlag::empty())?.into();
        }
        let sym = XPathBuf::from_self_fd(fd.as_raw_fd());

        // Step 2: Try to readlink().
        let (target, is_deleted, is_mfd) = match readlinkat(PROC_FILE(), &sym) {
            Ok(mut p) => {
                if p.is_relative() {
                    // /proc/1/fd/42 -> pipe:[4242], socket:[4242]
                    if cwd {
                        // /proc/$pid/cwd
                        // This can never happen.
                        return Err(Errno::ENOTDIR);
                    }

                    // SAFETY: /proc fd symlink does not refer to a
                    // filesystem path.
                    let mut sym = XPathBuf::from("/proc");
                    sym.push_pid(pid);

                    sym.push(b"fd");
                    sym.push(p.as_bytes());

                    (Some(sym), false, false)
                } else if p.ends_with(b" (deleted)") {
                    if p.starts_with(b"/memfd:") {
                        // Memory file descriptor:
                        // 1. Truncate deleted suffix
                        // 2. Change root slash to !
                        // This is used in access check as
                        // !memfd:* or !memfd-hugetlb:*
                        p.truncate(
                            p.len()
                                .checked_sub(b" (deleted)".len())
                                .ok_or(Errno::EOVERFLOW)?,
                        );
                        p.set(0, b'!');
                        (Some(p), false, true)
                    } else {
                        // /proc/1/cwd -> /path/to/foo (deleted)
                        //
                        // This may be a deleted file,
                        // or a file whose name funnily
                        // ends with a genuine " (deleted)" string.
                        //
                        // We will statx(2) and check nlink to tell.
                        (Some(p), true, false)
                    }
                } else {
                    // Genuine file descriptor that points to a valid directory.
                    (Some(p), false, false)
                }
            }
            Err(Errno::ENOENT) => return Err(Errno::EBADF),
            Err(Errno::ENAMETOOLONG) => (None, false, false),
            Err(errno) => return Err(errno),
        };

        if let Some(mut path) = target {
            // We managed to successfully readlink.
            //
            // Check for deleted entries.
            if is_deleted {
                // Check if path is really deleted.
                let stx = retry_on_eintr(|| fstatx(&fd, STATX_TYPE | STATX_NLINK))?;
                let f_type = FileType::from(mode_t::from(stx.stx_mode));

                if stx.stx_nlink == 0 {
                    // File is deleted, strip the deleted suffix.
                    #[allow(clippy::arithmetic_side_effects)]
                    path.truncate(path.len() - b" (deleted)".len());
                }

                return Ok(Self {
                    abs: path,
                    base: XPath::from_bytes(b""),
                    dir: Some(fd),
                    typ: Some(f_type),
                });
            }

            let file_type = if is_mfd {
                // Check for MFD_HUGETLB.
                if is_huge_file(&fd).unwrap_or(false) {
                    path.replace_prefix(MFD_NAME_PREFIX, MFD_HUGETLB_NAME_PREFIX)?;
                }
                Some(FileType::Mfd)
            } else if cwd {
                Some(FileType::Dir)
            } else {
                file_type(&fd, None, false).ok()
            };

            return Ok(Self {
                abs: path,
                base: XPath::from_bytes(b""),
                dir: Some(fd),
                typ: file_type,
            });
        }

        // Step 3: Use getdir_long().

        // SAFETY:
        // 1. getdir_long() does not change CWD!
        // 2. We limit maximum dir components to PATH_MAX_COMP so the sandbox
        //    process cannot create arbitrarily long directories and
        //    crash Syd.
        let path = getdir_long(&fd, PATH_MAX_COMP)?;

        Ok(Self {
            abs: path,
            base: XPath::from_bytes(b""),
            dir: Some(fd),
            typ: Some(FileType::Dir),
        })
    }

    // Create a `CanonicalPath` using a `FileMap`.
    #[allow(clippy::cognitive_complexity)]
    fn new_map(
        mut abs: XPathBuf,
        mut typ: Option<FileType>,
        options: FsFlags,
        mut filemap: FileMap,
        restrict_mkbdev: bool,
    ) -> Result<CanonicalPath<'a>, Errno> {
        // SAFETY: rootfs, devfs, procfs and sysfs views must be identical!
        if abs.is_rootfs() {
            return Ok(Self::new_root());
        } else if abs.is_procfs() {
            return Ok(Self::new_proc());
        } else if abs.is_equal(b"/dev/null") {
            return Ok(Self::new_null());
        }

        let follow_last = options.follow_last();
        let (mut is_magic_link, magic_base) = match typ {
            Some(FileType::MagicLnk(ref magic)) => {
                (true, if abs.is_proc() { magic.base() } else { None })
            }
            _ => (false, None),
        };

        // During statx and openat2 we do not want trailing slash,
        // or we'll get unexpected ELOOP on symbolic links.
        let has_trailing_slash = abs.ends_with_slash();
        if has_trailing_slash {
            #[allow(clippy::arithmetic_side_effects)]
            abs.truncate(abs.len() - 1);
        }

        // Determine file type.
        // SAFETY: rootfs and procfs views must be identical!
        let entry = filemap.remove(&abs);
        let mut magic_parent: XPathBuf;
        let (parent, mut base) = abs.split();

        // SAFETY: magic links are _always_ prefixed with `/proc`.
        // Hence the `unwrap` in the second branch is fine.
        #[allow(clippy::disallowed_methods)]
        let (parent_fd, has_parent) = if let Some(ref entry) = entry {
            // SAFETY: Use the O_PATH fd to determine file type
            // to avoid TOCTTOU. The file type is going to be used
            // during sandbox access check and therefore it's
            // security critical.
            typ = if let Some(f_type) = entry.f_type {
                Some(f_type)
            } else {
                file_type(&entry.fd, None, false).ok()
            };

            // SAFETY: The arguments we return here are only going to be
            // used in case we're re-opening the file for read. In this case,
            // we fall-back to `/proc/thread-self/fd` indirection to ensure
            // this is TOCTTOU free.
            //
            magic_parent = XPathBuf::from_self_fd(entry.fd.as_raw_fd());
            base = &magic_parent;
            is_magic_link = true; // Avoids ELOOP.

            (PROC_FD().into(), false)
        } else if let Some(entry_parent) = filemap.remove(parent) {
            if let Some(ref magic_base) = magic_base {
                base = magic_base;
            } else {
                typ = if let Some(Some(f_type)) = entry.as_ref().map(|e| e.f_type) {
                    Some(f_type)
                } else {
                    file_type(&entry_parent.fd, Some(base), false).ok()
                };
            }
            (entry_parent.fd, true)
        } else if let Some(ref magic_base) = magic_base {
            magic_parent = parent.strip_prefix(b"/proc").unwrap().to_owned();
            magic_parent.push(magic_base.as_bytes());
            base = &magic_parent;

            (PROC_FD().into(), false)
        } else if abs.is_proc() {
            base = XPath::from_bytes(&abs.as_bytes()[b"/proc/".len()..]);

            if !is_magic_link {
                typ = if let Some(Some(f_type)) = entry.as_ref().map(|e| e.f_type) {
                    Some(f_type)
                } else {
                    file_type(PROC_FILE(), Some(base), false).ok()
                };
            }

            (PROC_FD().into(), false)
        } else {
            base = XPath::from_bytes(&abs.as_bytes()[1..]);

            if !is_magic_link {
                typ = if let Some(Some(f_type)) = entry.as_ref().map(|e| e.f_type) {
                    Some(f_type)
                } else {
                    file_type(ROOT_FILE(), Some(base), false).ok()
                };
            }

            (ROOT_FD().into(), false)
        };

        crate::debug!("ctx": "resolve_path", "op": "open_last",
            "path": &abs,
            "base": &base,
            "type": &typ,
            "options": format!("{options:?}"),
            "open_files": filemap.0.len());

        // Do we want to resolve symbolic links for the last component?
        if follow_last {
            match typ {
                None if options.must_exist() => {
                    // SAFETY: last component must exist but it does not!
                    return Err(Errno::ENOENT);
                }
                Some(FileType::Lnk) => {
                    // SAFETY: symlink appeared out-of-nowhere, deny!
                    return Err(Errno::ELOOP);
                }
                _ => {}
            }
        } else if has_trailing_slash && typ.is_none() && options.must_exist() {
            // SAFETY: Attempt to follow dangling symbolic link.
            return Err(Errno::ENOENT);
        }

        // Do we want to open the file or the owning directory?
        let open_parent = options.intersects(FsFlags::MISS_LAST | FsFlags::WANT_BASE)
            || (typ.is_none() && !options.must_exist());

        if open_parent {
            // Open an `O_PATH` file descriptor to the owning directory.
            // Use parent fd as a reference if available.
            // SAFETY: rootfs, devfs, procfs and sysfs views must be identical!
            let parent_fd = if has_parent {
                parent_fd
            } else if parent.starts_with(b"/proc") {
                let fd: MaybeFd = if parent.len() == b"/proc".len() {
                    PROC_FD().into()
                } else {
                    let parent_base = XPath::from_bytes(&parent.as_bytes()[b"/proc/".len()..]);

                    safe_open_path(
                        PROC_FILE(),
                        parent_base,
                        OFlag::O_DIRECTORY,
                        ResolveFlag::empty(),
                    )?
                    .into()
                };

                fd
            } else {
                let parent_base = XPath::from_bytes(&parent.as_bytes()[1..]);

                let fd: MaybeFd = if parent_base.is_empty() {
                    ROOT_FD().into()
                } else {
                    safe_open_path(
                        ROOT_FILE(),
                        parent_base,
                        OFlag::O_DIRECTORY,
                        ResolveFlag::empty(),
                    )?
                    .into()
                };

                fd
            };

            // Calculate parent length early here, as `parent' which is
            // a reference into `abs' can potentially get invalidated in
            // the next trailing slash check.
            let parent_len = parent.len();

            // Preserve trailing slash to assert directory
            // after sandbox path hide check.
            if has_trailing_slash {
                abs.append_byte(b'/');
            };

            // Ensure we have the correct base path, because e.g. for
            // /proc/self/fd, `base' atm points not the actual base but
            // $pid/fd.  This happens due to the special parent handling
            // above and here we correct base.
            #[allow(clippy::arithmetic_side_effects)]
            {
                // When parent is "/", parent_len == 1 and we must NOT skip another byte.
                // For all other parents we skip the separating '/'.
                let offset = parent_len + usize::from(parent_len > 1);
                base = XPath::from_bytes(&abs.as_bytes()[offset..]);
            }

            // SAFETY: Extend the lifetime of `base` to `'a`
            let base = unsafe { std::mem::transmute::<&XPath, &'a XPath>(base) };

            return Ok(Self {
                abs,
                base,
                dir: Some(parent_fd),
                typ,
            });
        }

        // SAFETY:
        // 1. Use NONBLOCK with non-path fds to avoid blocking on pipes, FIFOs etc.
        // 2. Use NOCTTY to avoid acquiring controlling terminal.
        let (mut flags, is_read) = if options.contains(FsFlags::WANT_READ) {
            (OFlag::O_RDONLY | OFlag::O_NOCTTY | OFlag::O_NONBLOCK, true)
        } else {
            (OFlag::O_PATH, false)
        };

        // If the original path had a trailing slash,
        // require a directory on the final open
        // including magic symlinks.
        if has_trailing_slash {
            flags.insert(OFlag::O_DIRECTORY);
        }

        // SAFETY:
        // Do not open block devices without O_PATH,
        // which can have unintended side-effects. Note,
        // we already pass O_NONBLOCK so as not to block on FIFOs,
        // and pass O_NOCTTY so as not to acquire a controlling terminal.
        //
        // This restriction may be relaxed using trace/allow_unsafe_mkbdev:1.
        if restrict_mkbdev && is_read && typ.as_ref().map(|t| t.is_block_device()).unwrap_or(false)
        {
            return Err(Errno::ENOENT);
        }

        // If read is not requested return the O_PATH fd we opened.
        if !is_read {
            if let Some(entry) = entry {
                if has_trailing_slash {
                    // SAFETY: Preserve trailing slash to assert
                    // directory after sandbox path hide check.
                    abs.append_byte(b'/');
                }

                return Ok(Self {
                    abs,
                    base: XPath::from_bytes(b""),
                    dir: Some(entry.fd),
                    typ,
                });
            }
        }

        // If we must NOT resolve the last component and it is a regular symlink,
        // we must open the link object with O_PATH (not O_RDONLY).
        if !follow_last && matches!(typ, Some(FileType::Lnk)) {
            flags.remove(OFlag::O_RDONLY | OFlag::O_NONBLOCK | OFlag::O_NOCTTY);
            flags.insert(OFlag::O_PATH | OFlag::O_NOFOLLOW);
            if has_trailing_slash {
                flags.insert(OFlag::O_DIRECTORY);
            }
        }

        // SAFETY: safe_open does not follow symlinks.
        // Exception: Final component is a (magic) symlink,
        // and NO_FOLLOW_LAST is not set.
        let fd = if is_magic_link {
            if !follow_last {
                // NO_FOLLOW_LAST set, insert O_NOFOLLOW.
                flags.insert(OFlag::O_NOFOLLOW);
            }
            safe_open_msym(parent_fd, base, flags, ResolveFlag::empty())
        } else {
            // O_NOFOLLOW already set here.
            safe_open(parent_fd, base, flags, ResolveFlag::empty())
        }?;

        // Preserve trailing slash to assert directory
        // after sandbox path hide check.
        if has_trailing_slash {
            abs.append_byte(b'/');
        }

        Ok(Self {
            abs,
            base: XPath::from_bytes(b""),
            dir: Some(fd.into()),
            typ,
        })
    }
}

/// Return the canonical, absolute form of a path safely as a
/// file descriptor to the owning file descriptor together with
/// the base component and optional file type.
///
/// The `flag` parameter determines the sandbox restrictions to apply.
///
/// The `miss_mode` parameter controls how missing components are handled.
///
/// The `resolve` is a boolean parameter which controls whether the last
/// component should be resolved or not. Remaining components are always
/// resolved.
#[allow(clippy::cognitive_complexity)]
pub fn safe_canonicalize<'a>(
    pid: Pid,
    fd: Option<RawFd>,
    path: &XPath,
    mut options: FsFlags,
    sandbox: Option<&Sandbox>,
) -> Result<CanonicalPath<'a>, Errno> {
    // Safely initialize flags.
    let flags = sandbox.map(|sb| *sb.flags).unwrap_or_default();

    // SAFETY: File map is a hash table with paths as keys. The
    // values are the file descriptor, the file type, and optional
    // symbolic link target. This map is used throughout
    // canonicalization to ensure:
    // 1. We never reopen the same file.
    // 2. We never follow the same link.
    // 3. We never recheck the file type or mount id.
    let mut filemap: FileMap = FileMap::default();
    let mut file_type = None;

    let resolve_beneath = options.contains(FsFlags::RESOLVE_BENEATH);
    let (mut result, cwd) = if path.is_relative() {
        if let Some(fd) = fd {
            let magic = if fd == libc::AT_FDCWD {
                ProcMagic::Cwd { pid }
            } else if fd < 0 {
                return Err(Errno::EBADF);
            } else {
                ProcMagic::Fd { pid, fd }
            };
            let (mut entry, sym) = FileMapEntry::from_magic_link(magic, sandbox).map_err(|e| {
                // SAFETY: Return bad-fd rather than no-such-file for genuine fds.
                if !magic.want_dir() && e == Errno::ENOENT {
                    Errno::EBADF
                } else {
                    e
                }
            })?;

            // SAFETY: fd_entry's target member is always Some,
            // when FileMapEntry::from_magic_link returns success
            // with want_dir parameter set to true.
            #[allow(clippy::disallowed_methods)]
            let dir = entry.target.clone().unwrap().unwrap();

            let entry_sym = FileMapEntry::new(
                MaybeFd::RawFd(entry.fd.as_raw_fd()),
                Some(FileType::MagicLnk(ProcMagic::Fd { pid, fd })),
                entry.f_mode,
                entry.mnt_id,
                entry.target.take(),
            );

            entry.f_type = Some(FileType::Dir);
            filemap.0.try_reserve(2).or(Err(Errno::ENOMEM))?;
            filemap.0.insert(dir.clone(), entry);
            filemap.0.insert(sym, entry_sym);

            (dir.clone(), Some(dir))
        } else if !path.is_empty() {
            let magic = ProcMagic::Cwd { pid };
            let (mut entry, sym) = FileMapEntry::from_magic_link(magic, sandbox)?;

            // SAFETY: cwd_entry's target member is always Some,
            // when FileMapEntry::from_magic_link returns success
            // with want_dir parameter set to true.
            #[allow(clippy::disallowed_methods)]
            let dir = entry.target.clone().unwrap().unwrap();

            let entry_sym = FileMapEntry::new(
                MaybeFd::RawFd(entry.fd.as_raw_fd()),
                Some(FileType::MagicLnk(ProcMagic::Cwd { pid })),
                entry.f_mode,
                entry.mnt_id,
                entry.target.take(),
            );

            entry.f_type = Some(FileType::Dir);
            filemap.0.try_reserve(2).or(Err(Errno::ENOMEM))?;
            filemap.0.insert(dir.clone(), entry);
            filemap.0.insert(sym, entry_sym);

            (dir.clone(), Some(dir))
        } else {
            return Err(Errno::ENOENT);
        }
    } else if resolve_beneath {
        // RESOLVE_BENEATH rejects absolute values of pathname.
        return Err(Errno::EXDEV);
    } else if path.is_rootfs() {
        // Special case, `/` is never a symlink.
        // This must be done after the RESOLVE_BENEATH check!
        return Ok(CanonicalPath::new_root());
    } else if path.is_procfs() {
        // Special case, `/proc` is never a symlink.
        // This must be done after the RESOLVE_BENEATH check!
        return Ok(CanonicalPath::new_proc());
    } else {
        // Absolute path, CWD is ignored.
        (XPathBuf::from("/"), Some(XPathBuf::from("/")))
    };

    // What do we do when a component is missing?
    let miss_mode = MissingHandling::from(options);
    // Do we want to resolve magic /proc symbolic links?
    let resolve_proc = options.resolve_proc();
    // Do we want to traverse through mount points?
    //
    // As of version 3.39.0 this may be enforced
    // using the trace/force_no_xdev option.
    let resolve_xdev = if flags.force_no_xdev() {
        options.insert(FsFlags::NO_RESOLVE_XDEV);
        false
    } else {
        !options.contains(FsFlags::NO_RESOLVE_XDEV)
    };

    // `..` restriction for path traversal for chdir and open* family calls.
    let deny_dotdot = options.contains(FsFlags::NO_RESOLVE_DOTDOT);

    // Restriction for symlinks in sticky-bit, group/world-writable dirs.
    let restrict_symlinks = !flags.allow_unsafe_symlinks();

    // Restriction for /proc magic links.
    let restrict_magiclinks = !flags.allow_unsafe_magiclinks();

    // Restriction for block devices.
    let restrict_mkbdev = !flags.allow_unsafe_mkbdev();

    // Do we want to resolve symbolic links for the last component?
    let no_follow_last = !options.follow_last();

    // Options to open flags. O_NOFOLLOW is omitted here on purpose.
    let is_split = options.intersects(FsFlags::MISS_LAST | FsFlags::WANT_BASE);
    let mut open_flags = if !is_split && options.contains(FsFlags::WANT_READ) {
        OFlag::O_RDONLY | OFlag::O_NONBLOCK | OFlag::O_NOCTTY
    } else {
        OFlag::O_PATH
    };

    // SAFETY: Set close-on-exec.
    open_flags |= OFlag::O_CLOEXEC;

    // Split path into components, record trailing slash.
    let (mut parts, mut has_to_be_directory) = path_components(path)?;

    // Assert directory requirement with O_DIRECTORY.
    if has_to_be_directory {
        open_flags.insert(OFlag::O_DIRECTORY);
    }

    crate::debug!("ctx": "resolve_path", "op": "loop_init",
        "pid": pid.as_raw(),
        "path": &result,
        "root": &cwd,
        "is_dir": has_to_be_directory,
        "parts": &parts,
        "options": format!("{options:?}"),
        "flags": format!("{flags:?}"));

    // Used by ProcMagic::check_link:
    //  Deny regardless of PID if RESOLVE_NO_MAGICLINKS.
    // `magic_errno` determines between EXDEV and ELOOP.
    let pid_errno = options.magic_errno();

    // Determine reference mount id for RESOLVE_NO_XDEV.
    let mnt_id = if resolve_xdev {
        // Do nothing if RESOLVE_NO_XDEV is not set.
        None
    } else if let Some(ref cwd) = cwd {
        Some(filemap.get_mnt_id(cwd, pid, options, file_type, sandbox)?)
    } else if result.is_rootfs() {
        // Jumping to "/" is ok, but later components cannot cross.
        None
    } else {
        return Err(Errno::EXDEV);
    };

    let mut last;
    let mut loop_first = true;
    let mut no_resolve_symlinks;
    while let Some(part) = parts.pop_front() {
        // SAFETY: Limit maximum dir components to PATH_MAX_COMP so the
        // sandbox process cannot create arbitrarily long directories
        // and crash Syd.
        if result.len() >= PATH_MAX && parts.len() >= PATH_MAX_COMP {
            return Err(Errno::ENAMETOOLONG);
        }

        // Check if this is the last component.
        //
        // It may not necessarily be the last iteration,
        // in case the last component points to a symlink.
        last = parts.is_empty();

        // Do we want to resolve symbolic links in this path component?
        //
        // The answer is YES for all path components but the final
        // component. The resolution of the final component depends
        // on the following conditions:
        //
        // The answer is NO _if_ NO_FOLLOW_LAST flag is set, YES otherwise.
        //
        // _Unless_ the final component has a trailing slash which asserts a directory,
        // in which case we _DO_ resolve symbolic links in the final component regardless
        // of the NO_FOLLOW_LAST flag.
        //
        // _However_, if the MISS_LAST flag is also set, the trailing slash
        // is irrelevant, and we do _NOT_ resolve symlinks in the final
        // component.
        no_resolve_symlinks = last
            && no_follow_last
            && (!has_to_be_directory || miss_mode == MissingHandling::Missing);

        crate::debug!("ctx": "resolve_path", "op": "loop_iter",
            "pid": pid.as_raw(),
            "path": &result,
            "type": &file_type,
            "root": &cwd,
            "options": format!("{options:?}"),
            "part": &part,
            "parts": &parts,
            "open_files": filemap.0.len(),
            "resolve_beneath": resolve_beneath,
            "resolve_proc": resolve_proc,
            "resolve_xdev": resolve_xdev,
            "is_last": last,
            "is_dir": has_to_be_directory,
            "follow_last": !no_follow_last,
            "is_split": is_split,
            "oflags": format_oflags(open_flags),
            "rflags": options,
            "miss_mode": format!("{miss_mode:?}"),
            "deny_dotdot": deny_dotdot,
            "restrict_symlinks": restrict_symlinks,
            "restrict_magiclinks": restrict_magiclinks);

        if deny_dotdot && part == PathComponent::ParentDir {
            // SAFETY: Traversing through `..` is forbidden with
            // trace/deny_dotdot:1.
            return Err(Errno::EACCES);
        }

        // The first iteration of checks for RESOLVE_BENEATH,
        // and RESOLVE_NO_XDEV was done before the loop so
        // we skip them here for one turn.
        if !loop_first {
            // Check for RESOLVE_BENEATH and RESOLVE_NO_XDEV.
            // Temporary jumps are not allowed!
            if resolve_beneath {
                let cwd = cwd.as_ref().ok_or(Errno::EXDEV)?;

                if !result.deref().descendant_of(cwd.as_bytes()) {
                    return Err(Errno::EXDEV);
                }
            }

            if let Some(mnt_id) = mnt_id {
                // Jumping to "/" is ok, but later components cannot cross.
                if !result.is_rootfs() {
                    let my_mnt_id =
                        filemap.get_mnt_id(&result, pid, options, file_type, sandbox)?;
                    if my_mnt_id != mnt_id {
                        return Err(Errno::EXDEV);
                    }
                }
            }
        } else {
            loop_first = false;
        }

        match part {
            PathComponent::Normal(ref p) => {
                result.try_reserve(p.len()).or(Err(Errno::ENAMETOOLONG))?;
                result.push(p.as_bytes());

                // Invalidate file type.
                file_type = None;
            }
            PathComponent::ParentDir => {
                // Set file type to directory.
                file_type = Some(FileType::Dir);

                // Step 1: Check if file map has parent.
                if filemap.get(result.parent()).is_some() {
                    // We have seen parent before, pop and continue.
                    //
                    // SAFETY:
                    // 1. `result` is a normalized absolute path.
                    // 2. `result` does not have a trailing slash.
                    unsafe { result.pop_unchecked() };
                    continue;
                }

                // Step 2: Check file map for current entry.
                let entry = filemap.0.get_mut(&result).ok_or(Errno::ENOENT)?;

                // Check for directory if file type is known.
                if let Some(f_type) = entry.f_type {
                    if f_type != FileType::Dir {
                        // Walking up is only allowed for directories.
                        return Err(Errno::ENOTDIR);
                    }
                }

                // Pop result to create parent directory name.
                // SAFETY:
                // 1. `result` is a normalized absolute path.
                // 2. `result` does not have a trailing slash.
                unsafe { result.pop_unchecked() };

                // Open parent using a dotdot reference.
                //
                // This will fail with ENOTDIR if current entry
                // is not a directory which is what we want.
                //
                // Remove RESOLVE_BENEATH from resolve flags for dotdot to work.
                let how = safe_open_how(OFlag::O_PATH | OFlag::O_DIRECTORY, ResolveFlag::empty())
                    .resolve(ResolveFlag::RESOLVE_NO_MAGICLINKS | ResolveFlag::RESOLVE_NO_SYMLINKS);
                let fd = retry_on_eintr(|| safe_openat2(&entry.fd, XPath::from_bytes(b".."), how))?;

                // Mark current entry as a directory.
                // This is now known because dotdot worked.
                entry.f_type = Some(FileType::Dir);

                // Add new entry to file map and continue.
                let entry = FileMapEntry::new(fd.into(), Some(FileType::Dir), None, None, None);

                filemap.0.try_reserve(1).or(Err(Errno::ENOMEM))?;
                filemap.0.insert(result.clone(), entry);

                continue;
            }
        }

        // SAFETY:
        // 1. NO_MAGICLINKS unless trace/allow_unsafe_magiclinks:1.
        //    Continue in case:
        //    (a) pipe:42 socket:42 etc, special paths.
        //    (b) file descriptors that point to deleted paths.
        // 2. Handle /proc/self and /proc/thread-self indirections.
        let result_magic = match ProcMagic::check_link(pid, result.deref(), restrict_magiclinks)? {
            Some(_) if pid_errno == Errno::ELOOP && (!last || !no_follow_last) => {
                // PROC_NO_SYMLINKS || PROC_NO_MAGICLINKS.
                return Err(pid_errno);
            }
            Some(
                magic @ (ProcMagic::Fd { .. }
                | ProcMagic::Cwd { .. }
                | ProcMagic::Root { .. }
                | ProcMagic::Exe { .. }),
            ) => {
                // Handle /proc/$pid/{fd/<n>,cwd,root,exe} indirection unless
                // this is the final component and NO_FOLLOW_LAST is set.
                let sym = magic.link_path();
                file_type = Some(FileType::MagicLnk(magic));

                let target = if let Some(entry) = filemap.get(&sym) {
                    // SAFETY: Magic link target member is always Some.
                    #[allow(clippy::disallowed_methods)]
                    entry.target.clone().unwrap().unwrap()
                } else {
                    let (mut entry, _) = FileMapEntry::from_magic_link(magic, sandbox)?;

                    // SAFETY: Magic link target member is always Some.
                    #[allow(clippy::disallowed_methods)]
                    let target = entry.target.take().unwrap().unwrap();

                    let entry_sym = FileMapEntry::new(
                        MaybeFd::RawFd(entry.fd.as_raw_fd()),
                        Some(FileType::MagicLnk(magic)),
                        entry.f_mode,
                        entry.mnt_id,
                        Some(Ok(target.clone())),
                    );
                    filemap.0.try_reserve(1).or(Err(Errno::ENOMEM))?;
                    filemap.0.insert(sym, entry_sym);

                    if target.is_absolute() {
                        // entry.target is intentionally None here.
                        filemap.0.try_reserve(1).or(Err(Errno::ENOMEM))?;
                        filemap.0.insert(target.clone(), entry);
                    }

                    target
                };

                if last && !no_resolve_symlinks {
                    if target.is_relative() {
                        // Magic link points to magic path.
                        // Paths such as [pipe:64], [socket:42] etc.
                        // Append to result so the user can filter.

                        // SAFETY:
                        // 1. `result` is a normalized absolute path.
                        // 2. `result` does not have a trailing slash.
                        unsafe { result.pop_unchecked() };
                        result.push(target.as_bytes());
                    } else {
                        // Magic link points to genuine path.
                        // Set the result to target, reset file type.
                        result = target;
                        file_type = None;
                    }

                    if has_to_be_directory {
                        result.push(b"");
                    }

                    // Nothing left to do, break out.
                    break;
                }

                Some(Ok(target))
            }
            Some(magic @ ProcMagic::Ns { .. }) => {
                // readlias_ref().nk() yields a token "<name>:[id]" which is not a pathname.
                // Keep magic typing so new_map will open via safe_open_msym.
                file_type = Some(FileType::MagicLnk(magic));
                Some(Err(Errno::EINVAL))
            }
            None if result.is_proc_self(false) => {
                // Handle /proc/self indirection unless
                // this is the final component and NO_FOLLOW_LAST is set.
                if !no_resolve_symlinks {
                    file_type = Some(FileType::Dir);

                    // SAFETY:
                    // 1. `result` is a normalized absolute path.
                    // 2. `result` does not have a trailing slash.
                    unsafe { result.pop_unchecked() };

                    result.push_pid(pid);
                } else if last {
                    // Set file type to symlink.
                    // TODO: Do we want to set to magic link here?
                    file_type = Some(FileType::Lnk);
                }

                Some(Err(Errno::EINVAL))
            }
            None if result.is_proc_self(true) => {
                // Handle /proc/thread-self indirection unless
                // this is the final component and NO_FOLLOW_LAST is set.
                if !no_resolve_symlinks {
                    file_type = Some(FileType::Dir);

                    // SAFETY:
                    // 1. `result` is a normalized absolute path.
                    // 2. `result` does not have a trailing slash.
                    unsafe { result.pop_unchecked() };

                    let tgid = proc_tgid(pid)?;
                    result.push_pid(tgid);
                    result.push(b"task");
                    result.push_pid(pid);
                } else if last {
                    // Set file type to symlink.
                    // TODO: Do we want to set to magic link here?
                    file_type = Some(FileType::Lnk);
                }

                Some(Err(Errno::EINVAL))
            }
            None => None,
        };

        // We're resolving symbolic links for all path components but last.
        // The resolution of the last component depends on the NO_FOLLOW_LAST option.
        if no_resolve_symlinks {
            // Handle trailing slash as part of a symlink target.
            if result.ends_with(b"/") {
                has_to_be_directory = true;
                open_flags.insert(OFlag::O_DIRECTORY);
            }

            // SAFETY: Invalidate file type since we are not going to
            // resolve the final component. One exception is (magic)
            // symbolic links whose information we keep for later
            // hardening.
            if !file_type
                .as_ref()
                .map(|typ| typ.is_symlink() || typ.is_magic_link())
                .unwrap_or(false)
            {
                file_type = None;
            }

            // Nothing left to do, break out.
            break;
        }

        let resolve_result = if let Some(result_magic) = result_magic {
            result_magic
        } else {
            filemap.readlink(&result, pid, options, file_type, sandbox)
        };

        crate::debug!("ctx": "resolve_path", "op": "read_symlink",
            "pid": pid.as_raw(),
            "ret": format!("{resolve_result:?}"),
            "path": &result,
            "type": &file_type,
            "root": &cwd,
            "part": &part,
            "parts": &parts,
            "options": format!("{options:?}"),
            "open_files": filemap.0.len(),
            "resolve_beneath": resolve_beneath,
            "resolve_proc": resolve_proc,
            "resolve_xdev": resolve_xdev,
            "is_last": last,
            "follow_last": !no_follow_last,
            "is_split": is_split,
            "oflags": format_oflags(open_flags),
            "rflags": options,
            "miss_mode": format!("{miss_mode:?}"),
            "deny_dotdot": deny_dotdot,
            "restrict_symlinks": restrict_symlinks,
            "restrict_magiclinks": restrict_magiclinks);

        match resolve_result {
            Ok(target) => {
                // Invalidate file type before the next round.
                file_type = None;

                if target.is_relative() {
                    // SAFETY:
                    // 1. `result` is a normalized absolute path.
                    // 2. `result` does not have a trailing slash.
                    unsafe { result.pop_unchecked() };
                } else {
                    // 1. RootDir as part of resolving a symlink to an absolute path!
                    // 2. `result` is an absolute path so truncating to 1 effectively
                    //    leaves behind `/` only.
                    result.truncate(1);
                }
                path_components2(target.deref(), &mut parts)?;
            }
            Err(Errno::EINVAL) => {
                // Handle trailing slash as part of a symlink target.
                if last && !has_to_be_directory && result.ends_with(b"/") {
                    has_to_be_directory = true;
                    open_flags.insert(OFlag::O_DIRECTORY);
                } else if !file_type.map(|t| t.is_magic_link()).unwrap_or(false) {
                    // Invalidate only non-magic types;
                    // keep MagicLnk so we open via magicsym.
                    file_type = None;
                }
            }
            Err(errno) => match miss_mode {
                MissingHandling::Existing => return Err(errno),
                MissingHandling::Normal if !parts.is_empty() => return Err(errno),
                _ => {
                    // Invalidate file type.
                    file_type = None;
                }
            },
        }
    }

    crate::debug!("ctx": "resolve_path", "op": "loop_done",
        "pid": pid.as_raw(),
        "path": &result,
        "type": &file_type,
        "root": &cwd,
        "options": format!("{options:?}"),
        "open_files": filemap.0.len(),
        "resolve_beneath": resolve_beneath,
        "resolve_proc": resolve_proc,
        "resolve_xdev": resolve_xdev,
        "is_dir": has_to_be_directory,
        "follow_last": !no_follow_last,
        "is_split": is_split,
        "oflags": format_oflags(open_flags),
        "rflags": options,
        "miss_mode": format!("{miss_mode:?}"),
        "deny_dotdot": deny_dotdot,
        "restrict_symlinks": restrict_symlinks,
        "restrict_magiclinks": restrict_magiclinks);

    // SAFETY: Deny symbolic links at final component,
    // unless NO_FOLLOW_LAST is set explicitly.
    if options.follow_last()
        && file_type
            .as_ref()
            .map(|typ| typ.is_symlink())
            .unwrap_or(false)
    {
        return Err(Errno::ELOOP);
    }

    // Check for RESOLVE_BENEATH and RESOLVE_NO_XDEV.
    if resolve_beneath {
        let cwd = cwd.as_ref().ok_or(Errno::EXDEV)?;

        if !result.deref().descendant_of(cwd.as_bytes()) {
            return Err(Errno::EXDEV);
        }
    }

    // SAFETY: Handle trailing slash.
    // 1. Keep trailing slash as necessary.
    // 2. Add trailing slash for directories to assert file type.
    if has_to_be_directory {
        let dir_ok = match file_type {
            Some(FileType::Dir) => true,
            Some(FileType::MagicLnk(ProcMagic::Cwd { .. } | ProcMagic::Root { .. })) => true,
            // Everything else (Exe/Fd/Ns/other magics) is not directory-typed.
            Some(FileType::MagicLnk(_)) => false,
            // Unknown type: defer to the open() with O_DIRECTORY to decide.
            None => true,
            _ => false,
        };

        if matches!(
            miss_mode,
            MissingHandling::Existing | MissingHandling::Normal
        ) && !dir_ok
        {
            return Err(Errno::ENOTDIR);
        }
        result.append_byte(b'/');
    }

    if let Some(mnt_id) = mnt_id {
        // Open last component.
        let my_mnt_id = match (
            filemap.get_mnt_id(&result, pid, options, file_type, sandbox),
            miss_mode,
        ) {
            (Ok(mnt_id), _) => mnt_id,
            (Err(Errno::ELOOP), _) => {
                // SAFETY: symlink appeared out-of-nowhere, deny!
                return Err(Errno::ELOOP);
            }
            (Err(Errno::ENOENT), MissingHandling::Existing) => {
                return Err(Errno::ENOENT);
            }
            (Err(Errno::ENOTDIR), MissingHandling::Existing | MissingHandling::Normal)
                if has_to_be_directory =>
            {
                return Err(Errno::ENOTDIR);
            }
            _ => return Err(Errno::EXDEV),
        };

        if my_mnt_id != mnt_id {
            return Err(Errno::EXDEV);
        }
    }

    // SAFETY: Symbolic link resolution is no-longer needed/wanted.
    open_flags |= OFlag::O_NOFOLLOW;

    // SAFETY: Open a file descriptor to the canonical path, without
    // resolving symbolic links anymore. This brings safety against
    // TOCTTOU attacks.
    let flen = filemap.0.len();
    let path = CanonicalPath::new_map(result, file_type, options, filemap, restrict_mkbdev)?;

    crate::debug!("ctx": "resolve_path", "op": "resolve_done",
        "pid": pid.as_raw(),
        "path": &path,
        "root": &cwd,
        "options": format!("{options:?}"),
        "open_files": flen,
        "resolve_beneath": resolve_beneath,
        "resolve_proc": resolve_proc,
        "resolve_xdev": resolve_xdev,
        "is_dir": has_to_be_directory,
        "is_split": is_split,
        "follow_last": !no_follow_last,
        "oflags": format_oflags(open_flags),
        "rflags": options,
        "miss_mode": format!("{miss_mode:?}"),
        "deny_dotdot": deny_dotdot,
        "restrict_symlinks": restrict_symlinks,
        "restrict_magiclinks": restrict_magiclinks);

    Ok(path)
}

/// Format open(2) flags for serialization.
pub fn format_oflags(flags: OFlag) -> Vec<String> {
    let count = flags.into_iter().count();
    if count == 0 {
        return vec![];
    }

    let mut fmt = Vec::with_capacity(count);
    for flag in flags.iter() {
        fmt.push(format_oflag(flag));
    }

    fmt
}

/// Format a single open(2) flag for serialization.
pub fn format_oflag(flag: OFlag) -> String {
    let flag = format!("{flag:?}");

    if !flag.starts_with("OFlag(O_") || !flag.ends_with(')') {
        return "?".to_string();
    }

    #[allow(clippy::arithmetic_side_effects)]
    String::from_utf8_lossy(&flag.as_bytes()[8..flag.len() - 1]).to_ascii_lowercase()
}

/// Format unshare(2) flags for serialization.
pub fn format_clone_flags(flags: CloneFlags) -> Vec<&'static str> {
    let mut names = vec![];

    if flags.is_empty() {
        return names;
    }
    if flags.contains(CloneFlags::CLONE_NEWUSER) {
        names.push("user");
    }
    if flags.contains(CloneFlags::CLONE_NEWNS) {
        names.push("mount");
    }
    if flags.contains(CloneFlags::CLONE_NEWUTS) {
        names.push("uts");
    }
    if flags.contains(CloneFlags::CLONE_NEWIPC) {
        names.push("ipc");
    }
    if flags.contains(CloneFlags::CLONE_NEWPID) {
        names.push("pid");
    }
    if flags.contains(CloneFlags::CLONE_NEWNET) {
        names.push("net");
    }
    if flags.contains(CloneFlags::CLONE_NEWCGROUP) {
        names.push("cgroup");
    }
    if flags.contains(CLONE_NEWTIME) {
        names.push("time");
    }

    names
}

/// Format the return vector from `format_clone_flags` into a `String`.
pub fn format_clone_names(clone_names: &[&str]) -> String {
    match clone_names.len() {
        0 => "no namespaces".to_string(),
        1 => format!("{} namespace", clone_names[0]),
        2 => format!("{} and {} namespaces", clone_names[0], clone_names[1]),
        _ => {
            let mut s = clone_names.join(", ");
            #[allow(clippy::arithmetic_side_effects)]
            if let Some(pos) = s.rfind(", ") {
                s.replace_range(pos..pos + 2, ", and ");
            }
            format!("{s} namespaces")
        }
    }
}

/// Searches for a name within a directory.
///
/// `name` is matched literally and exactly against file names.
/// Directory entries are appended a trailing slash before matching.
/// Symlink entries are appended an `@` character before matching.
#[allow(clippy::disallowed_methods)]
pub fn grep(dir: &XPath, name: &[u8]) -> Option<XPathBuf> {
    let dir = File::open(dir.as_path()).ok()?;
    let name = XPath::from_bytes(name);
    loop {
        let mut entries = getdents64(&dir, 128).ok()?;
        for entry in &mut entries {
            let mut path = XPathBuf::from(entry.name_bytes());
            // Append a trailing slash for directories.
            if entry.is_dir() {
                path.append_byte(b'/');
            } else if entry.is_symlink() {
                path.append_byte(b'@');
            } else if entry.is_block_device() {
                path.append_byte(b'!');
            } else if entry.is_char_device() {
                path.append_byte(b'$');
            } else if entry.is_fifo() {
                path.append_byte(b'|');
            } else if entry.is_socket() {
                path.append_byte(b'~');
            }
            if *path == *name || (name.len() == 1 && path.ends_with(name.as_bytes())) {
                return Some(path);
            }
        }
    }
}

#[cfg(test)]
mod tests {
    use std::{
        fs::{self, OpenOptions},
        os::{
            fd::{AsRawFd, IntoRawFd},
            unix::{
                fs::{symlink, OpenOptionsExt},
                net::UnixStream,
            },
        },
        process::{exit, Command},
        thread::sleep,
        time::{Duration, SystemTime},
    };

    use bitflags::Flags;
    use nix::{
        fcntl::open,
        sched::{unshare, CloneFlags},
        sys::{
            signal::{kill, Signal},
            stat::Mode,
            wait::waitpid,
        },
        unistd::{chdir, close, dup, fchdir, fork, getcwd, mkdir, pause, pipe, ForkResult},
    };
    use tempfile::NamedTempFile;

    use super::*;
    use crate::xpath;

    fn setup() -> bool {
        let _ = crate::log::log_init_simple(crate::syslog::LogLevel::Warn);

        if let Err(error) = crate::config::proc_init() {
            eprintln!("Failed to initialize proc: {error:?}");
            return false;
        }

        true
    }

    // A helper function to create a deep directory structure.
    fn setup_deep_directory_structure(name: &XPath, depth: usize) -> Result<(), nix::Error> {
        // Tests may run in parallel, so ensure we don't share CWD.
        unshare(CloneFlags::CLONE_FS).unwrap();

        // Save the current directory by opening it
        let fd = safe_open_path(
            AT_FDCWD,
            XPath::from_bytes(b"."),
            OFlag::O_DIRECTORY,
            ResolveFlag::empty(),
        )?;

        // Create a deep directory structure starting from the current directory
        for _ in 0..depth {
            mkdir(name, Mode::S_IRWXU)?;
            chdir(name)?;
        }

        // After setup, restore the original directory
        fchdir(fd)?;

        Ok(())
    }

    // A helper function to get the current atime of a file
    fn get_atime<P: AsRef<Path>>(path: P) -> SystemTime {
        let metadata = fs::metadata(path).expect("Failed to get metadata");
        metadata.accessed().expect("Failed to get accessed time")
    }

    // Helper function to assert that the atime of a file or directory has not changed
    fn assert_atime_unchanged<'a, P: AsRef<Path>, F>(path: P, func: F)
    where
        F: FnOnce() -> Result<CanonicalPath<'a>, Errno>,
    {
        let original_atime_f = get_atime(&path);
        let original_atime_p = get_atime(path.as_ref().parent().unwrap());
        sleep(Duration::from_secs(7));
        assert!(
            func().is_ok(),
            "canonicalize {} failed",
            path.as_ref().display()
        );
        // We allow a 1-second tolerance since some filesystems do not have nanosecond precision.
        let new_atime_f = get_atime(&path);
        let new_atime_p = get_atime(path.as_ref().parent().unwrap());
        assert!(
            new_atime_f <= original_atime_f + Duration::new(1, 0),
            "The atime of the file should not have significantly changed."
        );
        assert!(
            new_atime_p <= original_atime_p + Duration::new(1, 0),
            "The atime of the parent dir should not have significantly changed."
        );
    }

    // std::fs::remove_dir_all stops on the first error.
    // we need something more forceful.
    fn remove_dir_all<P: AsRef<Path>>(path: P) -> std::io::Result<()> {
        let status = Command::new("rm")
            .arg("-rf")
            .arg(path.as_ref().to_string_lossy().to_string())
            .status()?;
        if status.success() {
            Ok(())
        } else {
            Err(std::io::Error::new(
                std::io::ErrorKind::Other,
                format!("Failed to remove directory: {}", path.as_ref().display()),
            ))
        }
    }

    // Helper function to create a symlink chain that eventually loops back to an earlier link
    fn setup_symlink_loop(tmp_dir: &XPath, links: &[(&str, &str)]) {
        for &(src, dst) in links {
            let src_path = tmp_dir.join(src.as_bytes());

            // Check and create parent directory for the source if necessary
            let src_parent = src_path.parent();
            if !src_parent.exists(false) {
                fs::create_dir_all(src_parent.as_path()).unwrap();
            }

            // The destination is given relative to the source
            let dst_path = XPath::from_bytes(&dst.as_bytes());

            // Avoid creating a symlink if the source already exists
            if src_path.exists(false) {
                fs::remove_file(src_path.as_path()).unwrap();
            }

            // If the destination is an absolute path or starts with '/', we strip the '/' and prefix with tmp_dir
            let full_dst_path = if dst_path.is_absolute() {
                tmp_dir.join(dst_path.strip_prefix(b"/").unwrap().as_bytes())
            } else {
                src_parent.join(dst_path.as_bytes()).into()
            };

            // Create parent directories for the destination if they don't exist
            let dst_parent = full_dst_path.parent();
            if !dst_parent.exists(false) {
                fs::create_dir_all(dst_parent.as_path()).unwrap();
            }

            // Create the symlink
            symlink(full_dst_path.as_path(), src_path.as_path())
                .expect(&format!("Unable to symlink {src_path} -> {full_dst_path}",));
        }
    }

    fn tempdir() -> Result<XPathBuf, Box<dyn std::error::Error>> {
        let tmp = tempfile::Builder::new()
            .disable_cleanup(true)
            .tempdir_in(".")?;
        let _ = OpenOptions::new()
            .write(true)
            .create(true)
            .mode(0o600)
            .open(tmp.path().join("test"))?;
        Ok(tmp
            .path()
            .to_path_buf()
            .file_name()
            .unwrap()
            .as_bytes()
            .into())
    }

    // Changes the current directory to `dir` handling arbitrarily long pathnames.
    fn chdir_long(dir: &XPath) -> Result<(), Errno> {
        let mut path_buf = dir.as_bytes();
        let mut current_dir_fd = -2;

        loop {
            // Attempt to change directory.
            let dir = XPath::from_bytes(path_buf);
            if path_buf.is_empty() || chdir(dir).is_ok() {
                if current_dir_fd >= 0 {
                    let _ = close(current_dir_fd);
                }
                return Ok(());
            }

            if !matches!(Errno::last(), Errno::ENAMETOOLONG | Errno::ENOMEM)
                || dir.len() < PATH_MAX - 1
            {
                break;
            }

            // Handle long path by trying to split at a directory boundary
            let mut boundary = path_buf.len().min(PATH_MAX - 1);
            while boundary > 0 && path_buf[boundary] != b'/' {
                boundary -= 1;
            }

            if boundary == 0 {
                break;
            }

            // Attempt to temporarily change to the subdirectory.
            if current_dir_fd == -2 {
                // Open current directory.
                current_dir_fd = open(".", OFlag::O_PATH | OFlag::O_DIRECTORY, Mode::empty())
                    .map(|fd| fd.into_raw_fd())?;
            }

            let dir = XPath::from_bytes(&path_buf[..boundary]);
            if chdir(dir).is_err() {
                break;
            }

            // Move to the next component.
            path_buf = &path_buf[boundary + 1..];
        }

        // Save last errno to return from the function.
        let errno = match Errno::last() {
            Errno::UnknownErrno => Errno::ENAMETOOLONG,
            errno => errno,
        };

        // Attempt to restore the original directory if there was a failure.
        if current_dir_fd >= 0 {
            let result = fchdir(unsafe { BorrowedFd::borrow_raw(current_dir_fd) });
            let _ = close(current_dir_fd);

            return if result.is_ok() {
                Err(errno)
            } else {
                Err(Errno::ENOTRECOVERABLE)
            };
        }

        Err(if current_dir_fd == -2 {
            errno
        } else {
            Errno::ENOTRECOVERABLE
        })
    }

    #[test]
    fn test_canonicalize_empty_path() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &XPath::from_bytes(b""),
            FsFlags::empty(),
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ENOENT)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &XPath::from_bytes(b""),
            FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ENOENT)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &XPath::from_bytes(b""),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ENOENT)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &XPath::from_bytes(b""),
            FsFlags::NO_FOLLOW_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ENOENT)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &XPath::from_bytes(b""),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ENOENT)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &XPath::from_bytes(b""),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ENOENT)), "{result:?}");
    }

    #[test]
    fn test_canonicalize_repetitive_root() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let root = safe_canonicalize(
            Pid::this(),
            None,
            &XPath::from_bytes(b"//"),
            FsFlags::empty(),
            Some(&sandbox),
        )
        .unwrap();
        assert_eq!(root.abs.as_bytes(), b"/");
        assert_eq!(root.typ, Some(FileType::Dir));
    }

    #[test]
    fn test_canonicalize_repetitive_slashes() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let result_test = safe_canonicalize(
            Pid::this(),
            None,
            &XPath::from_bytes(b"/etc/passwd"),
            FsFlags::empty(),
            Some(&sandbox),
        )
        .unwrap()
        .abs;
        let paths = vec![
            "/etc/passwd",
            "/etc//passwd",
            "/etc///passwd",
            "//etc/passwd",
            "//etc//passwd",
            "//etc///passwd",
            "///etc/passwd",
            "///etc//passwd",
            "///etc///passwd",
        ];
        for path in &paths {
            let path = XPathBuf::from(path.to_string());
            let result = safe_canonicalize(
                Pid::this(),
                None,
                &path.deref(),
                FsFlags::empty(),
                Some(&sandbox),
            )
            .unwrap()
            .abs;
            assert_eq!(result, result_test);
        }
    }

    #[test]
    fn test_canonicalize_dots_slashes() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let cwd = XPathBuf::from(std::env::current_dir().unwrap());
        let tmp = tempdir().unwrap();

        let path = xpath!("{tmp}//./..//{tmp}/test");
        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::empty(),
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{path}->{result:?}");
        let result1 = result.unwrap().abs;
        let path = xpath!("{cwd}/{tmp}//./..//{tmp}/test");
        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{path}->{result:?}");
        let result2 = result.unwrap().abs;

        assert!(!result1.is_empty(), "result:{result1}");
        assert!(!result2.is_empty(), "result:{result2}");
        assert_eq!(result1, result2);
    }

    #[test]
    fn test_canonicalize_non_directory_with_slash() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let path = tempdir().unwrap();
        let test = xpath!("{path}/test/");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &test.deref(),
            FsFlags::empty(),
            Some(&sandbox),
        );
        //SAFETY: File existence check is handled by read_path*
        //assert!(matches!(result, Err(Errno::ENOTDIR)), "{result:?}");
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &test.deref(),
            FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        //SAFETY: File existence check is handled by read_path*
        //assert!(matches!(result, Err(Errno::ENOTDIR)), "{result:?}");
        assert!(result.is_ok(), "{result:?}");

        //SAFETY: Missing is handled by read_path*.
        assert!(safe_canonicalize(
            Pid::this(),
            None,
            &test.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        )
        .is_ok());

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &test.deref(),
            FsFlags::NO_FOLLOW_LAST,
            Some(&sandbox),
        );
        //SAFETY: File existence check is handled by read_path*
        //assert!(matches!(result, Err(Errno::ENOTDIR)), "{result:?}");
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &test.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        //SAFETY: File existence check is handled by read_path*
        //assert!(matches!(result, Err(Errno::ENOTDIR)), "{result:?}");
        assert!(result.is_ok(), "{result:?}");

        //SAFETY: Missing is handled by read_path*.
        assert!(safe_canonicalize(
            Pid::this(),
            None,
            &test.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        )
        .is_ok());
    }

    /// FIXME: The asserts return success rather than failure.
    /// Bug or feature.unwrap()
    #[test]
    fn test_canonicalize_missing_directory_returns_enoent() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &XPath::from_bytes(b"/zzz/.."),
            FsFlags::empty(),
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ENOENT)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &XPath::from_bytes(b"/zzz/.."),
            FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ENOENT)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &XPath::from_bytes(b"/zzz/.."),
            FsFlags::NO_FOLLOW_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ENOENT)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &XPath::from_bytes(b"/zzz/.."),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ENOENT)), "{result:?}");
    }

    #[test]
    fn test_relative_symlink_resolution() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        // Setup
        let root_test_dir = &XPath::from_bytes(b"test_root_relative_symlink_resolution");
        let deep_dir = root_test_dir.join(b"a/b/c");
        let _ = remove_dir_all(&root_test_dir);
        fs::create_dir_all(&root_test_dir.join(b"d")).unwrap();
        fs::create_dir_all(&deep_dir).unwrap();

        // Create a symlink in "b" that points upwards to "a"
        let rel_link = root_test_dir.join(b"a/b/rel_link");
        symlink("../..", &rel_link).unwrap();

        // Append /proc/self/cwd to get an absolute path to our symlinked path
        let abs_link_path = &XPath::from_bytes(b"/proc/self/cwd")
            .join(root_test_dir.join(b"a/b/rel_link/d").as_bytes());

        // Call canonicalize
        let result = safe_canonicalize(
            Pid::this(),
            None,
            &abs_link_path.deref(),
            FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(
            result.is_ok(),
            "canonicalize:{abs_link_path} result:{result:?}",
        );
        let resolved_path = result.unwrap().abs;

        // We expect the path to be resolved to "test_root/a/d", but we need to canonicalize it
        let expected_path = fs::canonicalize(
            &XPath::from_bytes(b"/proc/self/cwd").join(root_test_dir.join(b"d").as_bytes()),
        )
        .unwrap();

        // Cleanup:
        // Note, remove_dir_all cannot remove a directory
        // tree if it spots a symbolic link loop unlike
        // `rm -rf`.
        // let _ = remove_dir_all(&root_test_dir);
        Command::new("/bin/rm")
            .arg("-rf")
            .arg(&root_test_dir)
            .status()
            .expect("rm -rf tmpdir");

        assert_eq!(resolved_path, XPathBuf::from(expected_path));
    }

    // FIXME: This test broke after we removed normalize()
    // The question: Is the test incorrect or is canonicalize().unwrap()
    #[ignore]
    #[test]
    fn test_complex_interplay_symlinks_dots() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        // Setup
        let cwd = XPathBuf::from(Path::new("/proc/self/cwd").canonicalize().unwrap());
        let root_test_dir = cwd.join(b"test_root_complex_interplay_symlinks_dots");
        let _ = remove_dir_all(&root_test_dir);
        fs::create_dir_all(root_test_dir.join(b"a/b/c")).unwrap();
        fs::create_dir(root_test_dir.join(b"d")).unwrap();
        fs::create_dir(root_test_dir.join(b"e")).unwrap();
        fs::create_dir(root_test_dir.join(b"x")).unwrap();

        // Create several symlinks
        symlink("./a", root_test_dir.join(b"link_to_a")).unwrap();
        symlink("e", root_test_dir.join(b"link_to_e")).unwrap();
        symlink("a/b", root_test_dir.join(b"link_to_b")).unwrap();
        symlink("../../x", root_test_dir.join(b"a/b/rel_link")).unwrap();

        let path = root_test_dir.join(b"link_to_a/../link_to_b/rel_link/../..");
        let resolved_path = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::MUST_PATH,
            Some(&sandbox),
        )
        .unwrap()
        .abs;

        // Cleanup
        let _ = remove_dir_all(&root_test_dir);

        // Assertion
        assert_eq!(resolved_path, XPathBuf::from(root_test_dir));
    }

    #[test]
    fn test_trailing_slash_handling() {
        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        // Spawn a new process which will inherit the fds.
        // Note we cannot use the current process here,
        // as XPath::check will think it's Syd and deny
        // access.
        let child = match unsafe { fork() } {
            Ok(ForkResult::Parent { child }) => child,
            Ok(ForkResult::Child) => {
                pause();
                exit(127);
            }
            Err(errno) => exit(errno as i32),
        };

        if !setup() {
            kill(child, Signal::SIGKILL).unwrap();
            waitpid(child, None).unwrap();
            return;
        }

        let path = XPath::from_bytes(b"/usr/");
        let pexp = path;
        let path = safe_canonicalize(child, None, &path, FsFlags::empty(), Some(&sandbox)).unwrap();
        assert_eq!(pexp, path.abs.deref(), "{pexp} != {path:?}");
        eprintln!("ok 1");

        let parg = XPath::from_bytes(b"/proc/self/");
        let pexp = xpath!("/proc/{child}/");

        let path = safe_canonicalize(child, None, &parg, FsFlags::empty(), Some(&sandbox)).unwrap();
        assert_eq!(path.abs, pexp, "{pexp} != {path:?}");
        eprintln!("ok 2 step 1");

        let path =
            safe_canonicalize(child, None, &parg, FsFlags::NO_FOLLOW_LAST, Some(&sandbox)).unwrap();
        assert_eq!(path.abs, pexp, "{pexp} != {path:?}");
        eprintln!("ok 2 step 2");

        kill(child, Signal::SIGKILL).unwrap();
        waitpid(child, None).unwrap();
    }

    #[test]
    fn test_canonicalize_no_atime_change_normal() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let cdir = XPathBuf::from(std::env::current_dir().unwrap());
        let base = cdir.join(tempdir().unwrap().as_bytes());
        let path = base.join(b"file");
        fs::File::create(&path).unwrap();

        assert_atime_unchanged(&path, || {
            safe_canonicalize(
                Pid::this(),
                None,
                &path.deref(),
                FsFlags::empty(),
                Some(&sandbox),
            )
        });

        let _ = remove_dir_all(&base);
    }

    #[test]
    fn test_canonicalize_no_atime_change_existing() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let cdir = XPathBuf::from(std::env::current_dir().unwrap());
        let base = cdir.join(&tempdir().unwrap().as_bytes());
        let path = base.join(b"file");
        fs::File::create(&path).unwrap();

        assert_atime_unchanged(&path, || {
            safe_canonicalize(
                Pid::this(),
                None,
                &path.deref(),
                FsFlags::MUST_PATH,
                Some(&sandbox),
            )
        });

        let _ = remove_dir_all(&base);
    }

    #[test]
    fn test_canonicalize_symlink_loop() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let tmp_dir = tempfile::Builder::new()
            .disable_cleanup(true)
            .tempdir()
            .expect("Failed to create temp dir");
        let dir_path = XPathBuf::from(tmp_dir.path().to_path_buf());

        // Create a symlink loop: link_a -> link_b -> link_a
        let mut link_a = dir_path.join(b"link_a");
        let mut link_b = dir_path.join(b"link_b");
        symlink(&link_b, &link_a).expect("Failed to create symlink a");
        symlink(&link_a, &link_b).expect("Failed to create symlink b");

        // Now check that canonicalize detects the loop correctly
        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link_a.deref(),
            FsFlags::NO_FOLLOW_LAST,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link_a.deref(),
            FsFlags::empty(),
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link_a.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link_a.deref(),
            FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link_a.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        //SAFETY: Missing is handled by read_path*.
        //assert_eq!(result, Err(Errno::EEXIST));
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link_a.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        // Add a trailing slash and retest.
        link_a.push(b"");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link_a.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link_a.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        // Add a trailing slash and retest.
        link_b.push(b"");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link_b.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link_b.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");
    }

    #[test]
    fn test_canonicalize_nonexistent_final_component() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let tmp_dir = tempfile::Builder::new()
            .disable_cleanup(true)
            .tempdir()
            .expect("Failed to create temp dir");
        let dir_path = XPathBuf::from(tmp_dir.path().to_path_buf());

        // Create a valid symlink to a non-existent final component
        let mut valid_link = dir_path.join(b"valid_link");
        let nonexistent_target = dir_path.join(b"nonexistent");
        symlink(&nonexistent_target, &valid_link)
            .expect("Failed to create symlink to non-existent target");

        // Now check that canonicalize handles the non-existent final component correctly.
        let result = safe_canonicalize(
            Pid::this(),
            None,
            &valid_link.deref(),
            FsFlags::NO_FOLLOW_LAST,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &valid_link.deref(),
            FsFlags::empty(),
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &valid_link.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &valid_link.deref(),
            FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ENOENT)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &valid_link.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        //SAFETY: Missing is handled by read_path*.
        //assert_eq!(result, Err(Errno::EEXIST));
        assert!(result.is_ok(), "{result:?}");

        // FIXME: Why oh why.unwrap()
        let result = safe_canonicalize(
            Pid::this(),
            None,
            &valid_link.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");
        // should be: assert_eq!(result, Err(Errno::EEXIST));

        // Add a trailing slash and retest.
        valid_link.push(b"");

        // FIXME: Why oh why.unwrap()
        let result = safe_canonicalize(
            Pid::this(),
            None,
            &valid_link.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");
        // should be: assert_eq!(result, Err(Errno::EEXIST));

        // FIXME: Why oh why?
        let result = safe_canonicalize(
            Pid::this(),
            None,
            &valid_link.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");
        // should be: assert_eq!(result, Err(Errno::EEXIST));
    }

    #[test]
    fn test_canonicalize_self_referential_symlink() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let tmp_dir = tempfile::Builder::new()
            .disable_cleanup(true)
            .tempdir()
            .expect("Failed to create temp dir");
        let dir_path = XPathBuf::from(tmp_dir.path().to_path_buf());
        let mut symlink_path = dir_path.join(b"self_link");
        symlink(symlink_path.as_path(), symlink_path.as_path())
            .expect("Failed to create self-referential symlink");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &symlink_path.deref(),
            FsFlags::NO_FOLLOW_LAST,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &symlink_path.deref(),
            FsFlags::empty(),
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &symlink_path.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &symlink_path.deref(),
            FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &symlink_path.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        //SAFETY: Missing is handled by read_path*.
        //assert_eq!(result, Err(Errno::EEXIST));
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &symlink_path.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        // Add a trailing slash and retest.
        symlink_path.push(b"");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &symlink_path.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        //SAFETY: Missing is handled by read_path*.
        //assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &symlink_path.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");
    }

    #[test]
    fn test_canonicalize_broken_symlink() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let tmp_dir = tempfile::Builder::new()
            .disable_cleanup(true)
            .tempdir()
            .expect("Failed to create temp dir");
        let dir_path = XPathBuf::from(tmp_dir.path().to_path_buf());
        let mut broken_link = dir_path.join(b"broken_link");
        let nonexistent_target = dir_path.join(b"nonexistent_target");
        symlink(nonexistent_target.as_path(), broken_link.as_path())
            .expect("Failed to create broken symlink");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &broken_link.deref(),
            FsFlags::NO_FOLLOW_LAST,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");
        eprintln!("ok 1");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &broken_link.deref(),
            FsFlags::empty(),
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");
        eprintln!("ok 2");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &broken_link.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");
        eprintln!("ok 3");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &broken_link.deref(),
            FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ENOENT)), "{result:?}");
        eprintln!("ok 4");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &broken_link.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        //SAFETY: Missing is handled by read_path*.
        //assert_eq!(result, Err(Errno::EEXIST));
        assert!(result.is_ok(), "{result:?}");
        eprintln!("ok 5");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &broken_link.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");
        eprintln!("ok 6");

        // Add a trailing slash and retest.
        broken_link.push(b"");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &broken_link.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");
        eprintln!("ok 7");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &broken_link.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{result:?}");
        eprintln!("ok 8");
    }

    #[test]
    fn test_canonicalize_symlink_to_directory() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let tmp_dir = tempfile::Builder::new()
            .disable_cleanup(true)
            .tempdir()
            .expect("Failed to create temp dir");
        let tmp_path = XPathBuf::from(tmp_dir.path().to_path_buf());
        let dir = tmp_path.join(b"dir");
        fs::create_dir(&dir).expect("Failed to create directory");

        let symlink_path = tmp_path.join(b"dir_link");
        symlink(dir.as_path(), symlink_path.as_path())
            .expect("Failed to create symlink to directory");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &symlink_path.deref(),
            FsFlags::NO_FOLLOW_LAST,
            Some(&sandbox),
        );
        let result_repr = format!("{result:?}");
        assert!(result.is_ok(), "{result_repr}");
        assert!(result.unwrap().typ.unwrap().is_symlink(), "{result_repr}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &symlink_path.deref(),
            FsFlags::empty(),
            Some(&sandbox),
        );
        let result_repr = format!("{result:?}");
        assert!(result.is_ok(), "{result_repr}");
        assert!(result.unwrap().typ.unwrap().is_dir(), "{result_repr}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &symlink_path.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        let result_repr = format!("{result:?}");
        assert!(result.is_ok(), "{result_repr}");
        assert!(result.unwrap().typ.unwrap().is_symlink(), "{result_repr}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &symlink_path.deref(),
            FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        let result_repr = format!("{result:?}");
        assert!(result.is_ok(), "{result_repr}");
        assert!(result.unwrap().typ.unwrap().is_dir(), "{result_repr}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &symlink_path.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        //SAFETY: Missing is handled by read_path*.
        //assert_eq!(result, Err(Errno::EEXIST));
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &symlink_path.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        //SAFETY: Missing is handled by read_path*.
        //assert_eq!(result, Err(Errno::EEXIST));
        assert!(result.is_ok(), "{result:?}");
    }

    #[test]
    fn test_canonicalize_symlink_chain() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let tmp_dir = tempfile::Builder::new()
            .disable_cleanup(true)
            .tempdir()
            .expect("Failed to create temp dir");
        let dir_path = XPathBuf::from(tmp_dir.path().to_path_buf());
        let link1 = dir_path.join(b"link1");
        let link2 = dir_path.join(b"link2");
        let link3 = dir_path.join(b"link3");
        let file = dir_path.join(b"file");
        fs::write(file.as_path(), "content").expect("Failed to write file");

        // Create a chain of symlinks: link1 -> link2 -> link3 -> file
        symlink(link2.as_path(), link1.as_path()).expect("Failed to create link1");
        symlink(link3.as_path(), link2.as_path()).expect("Failed to create link2");
        symlink(file.as_path(), link3.as_path()).expect("Failed to create link3");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link1.deref(),
            FsFlags::NO_FOLLOW_LAST,
            Some(&sandbox),
        );
        let result_repr = format!("{result:?}");
        assert!(result.is_ok(), "{result_repr}");
        assert!(result.unwrap().typ.unwrap().is_symlink(), "{result_repr}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link1.deref(),
            FsFlags::empty(),
            Some(&sandbox),
        );
        let result_repr = format!("{result:?}");
        assert!(result.is_ok(), "{result_repr}");
        assert!(result.unwrap().typ.unwrap().is_file(), "{result_repr}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link1.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        let result_repr = format!("{result:?}");
        assert!(result.is_ok(), "{result_repr}");
        assert!(result.unwrap().typ.unwrap().is_symlink(), "{result_repr}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link1.deref(),
            FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        let result_repr = format!("{result:?}");
        assert!(result.is_ok(), "{result_repr}");
        assert!(result.unwrap().typ.unwrap().is_file(), "{result_repr}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link1.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        //SAFETY: Missing is handled by read_path*.
        //assert_eq!(result, Err(Errno::EEXIST));
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &link1.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        //SAFETY: Missing is handled by read_path*.
        //assert_eq!(result, Err(Errno::EEXIST));
        assert!(result.is_ok(), "{result:?}");
    }

    #[test]
    fn test_safe_canonicalize_complex_symlink_loop_with_intermediate_components() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let tmp_dir = tempfile::Builder::new()
            .disable_cleanup(true)
            .tempdir()
            .expect("Failed to create temp dir");
        let dir_path = XPathBuf::from(tmp_dir.path().to_path_buf());

        // Setting up a complex symlink scenario
        setup_symlink_loop(
            &dir_path.deref(),
            &[("a", "b/c"), ("b/c", "d"), ("b/d", "../e"), ("e", "f/../a")],
        );

        let mut path = dir_path.join(b"a");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::NO_FOLLOW_LAST,
            Some(&sandbox),
        );
        let result_repr = format!("{result:?}");
        assert!(result.is_ok(), "{result_repr}");
        assert!(result.unwrap().typ.unwrap().is_symlink(), "{result_repr}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::empty(),
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        let result_repr = format!("{result:?}");
        assert!(result.is_ok(), "{result_repr}");
        assert!(result.unwrap().typ.unwrap().is_symlink(), "{result_repr}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        //SAFETY: Missing is handled by read_path*.
        //assert_eq!(result, Err(Errno::EEXIST));
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        // Add a trailing slash and retest.
        path.push(b"");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        //SAFETY: Missing is handled by read_path*.
        //assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        // Add a final component and retest.
        path.push(b"foo");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");
    }

    #[test]
    fn test_safe_canonicalize_symlinks_with_dot_and_dotdot_components() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let tmp_dir = tempfile::Builder::new()
            .disable_cleanup(true)
            .tempdir()
            .expect("Failed to create temp dir");

        // Create a complex directory structure with dots and symlinks
        fs::create_dir_all(tmp_dir.path().join("b")).expect("Failed to create directory b");
        symlink("b", tmp_dir.path().join("a")).expect("Failed to create symlink a -> b");
        symlink("..///e", tmp_dir.path().join("b").join("d"))
            .expect("Failed to create symlink b/d -> ../e");
        symlink("b/.///./d", tmp_dir.path().join("e")).expect("Failed to create symlink e -> b/d");

        let mut path = XPathBuf::from(tmp_dir.path().join("a").join(".").join("d"));

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::NO_FOLLOW_LAST,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{path}->{result:?}");
        let result = result.unwrap();
        assert!(result.abs.exists(false), "{path}->{result:?}");
        assert!(!result.abs.exists(true), "{path}->{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::empty(),
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(result.is_ok(), "{path}->{result:?}");
        let result = result.unwrap();
        assert!(result.abs.exists(false), "{path}->{result:?}");
        assert!(!result.abs.exists(true), "{path}->{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::MUST_PATH,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        //SAFETY: Missing is handled by read_path*.
        //assert_eq!(result, Err(Errno::EEXIST));
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        // Add a trailing slash and retest.
        path.push(b"");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        // Add a final component and retest.
        path.push(b"foo");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");

        let result = safe_canonicalize(
            Pid::this(),
            None,
            &path.deref(),
            FsFlags::MISS_LAST,
            Some(&sandbox),
        );
        assert!(matches!(result, Err(Errno::ELOOP)), "{result:?}");
    }

    #[test]
    fn test_canonicalize_proc_self() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        let pid = Pid::this();
        let path = safe_canonicalize(
            pid,
            None,
            &XPath::from_bytes(b"/proc/self"),
            FsFlags::NO_FOLLOW_LAST,
            Some(&sandbox),
        )
        .expect("canonicalize /proc/self");
        assert!(
            path.typ
                .as_ref()
                .map(|typ| typ.is_symlink())
                .unwrap_or(false),
            "path:{path:?}"
        );
        assert_eq!(
            path.abs.deref(),
            XPath::from_bytes(b"/proc/self"),
            "path:{path:?}"
        );
        match path.typ {
            Some(FileType::Lnk) => {}
            _ => panic!("path:{path:?}"),
        }
    }

    #[test]
    fn test_canon_glob_std() {
        if !setup() {
            return;
        }

        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        // Note we cannot assume the FDs {0,1,2} are open here.
        let fd = open("/dev/null", OFlag::O_RDONLY, Mode::empty()).unwrap();
        let fd = xpath!("/proc/self/fd/{}", fd.as_raw_fd());
        let pid = Pid::this();

        let result = safe_canonicalize(pid, None, &fd.deref(), FsFlags::empty(), Some(&sandbox));
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(pid, None, &fd.deref(), FsFlags::MUST_PATH, Some(&sandbox));
        assert!(result.is_ok(), "{result:?}");

        let result = safe_canonicalize(pid, None, &fd.deref(), FsFlags::MISS_LAST, Some(&sandbox));
        //SAFETY: Missing is handled by read_path*.
        //assert_eq!(result, Err(Errno::EEXIST));
        assert!(result.is_ok(), "{result:?}");
    }

    #[test]
    fn test_canon_glob_pipe() {
        let mut sandbox = Sandbox::default();
        sandbox.config("allow/lpath+/***").unwrap();
        sandbox.flags.clear();

        // Create a socket pair.
        let (read_end, write_end) = UnixStream::pair().unwrap();

        // Spawn a new process which will inherit the fds.
        // Note we cannot use the current process here,
        // as XPath::check will think it's Syd and deny
        // access.
        let child = match unsafe { fork() } {
            Ok(ForkResult::Parent { child }) => child,
            Ok(ForkResult::Child) => {
                pause();
                exit(127);
            }
            Err(errno) => exit(errno as i32),
        };

        if !setup() {
            kill(child, Signal::SIGKILL).unwrap();
            waitpid(child, None).unwrap();
            return;
        }

        let fd = read_end.as_raw_fd();
        let path = xpath!("/proc/{child}/fd/{fd}");
        let result =
            safe_canonicalize(child, None, &path.deref(), FsFlags::empty(), Some(&sandbox));
        assert!(result.is_ok(), "{path}->{result:?}");

        let fd = write_end.as_raw_fd();
        let path = xpath!("/proc/{child}/fd/{fd}");
        let result =
            safe_canonicalize(child, None, &path.deref(), FsFlags::empty(), Some(&sandbox));
        assert!(result.is_ok(), "{path}->{result:?}");

        kill(child, Signal::SIGKILL).unwrap();
        waitpid(child, None).unwrap();
    }

    #[test]
    fn test_path_components_empty_path() {
        let path = XPath::from_bytes(b"");
        let (components, has_trailing_slash) = path_components(&path).unwrap();
        assert_eq!(components, VecDeque::new());
        assert!(has_trailing_slash);
    }

    #[test]
    fn test_path_components_only_parent_dir() {
        let path = XPath::from_bytes(b"..");
        let (components, has_trailing_slash) = path_components(&path).unwrap();
        assert_eq!(components, VecDeque::from([PathComponent::ParentDir]));
        assert!(has_trailing_slash);
    }

    #[test]
    fn test_path_components_multiple_parent_dir() {
        let path = XPath::from_bytes(b"../..");
        let (components, has_trailing_slash) = path_components(&path).unwrap();
        assert_eq!(
            components,
            VecDeque::from([PathComponent::ParentDir, PathComponent::ParentDir])
        );
        assert!(has_trailing_slash);
    }

    #[test]
    fn test_path_components_parent_dir_with_normal_components() {
        let path = XPath::from_bytes(b"../foo/../bar");
        let (components, has_trailing_slash) = path_components(&path).unwrap();
        assert_eq!(
            components,
            VecDeque::from([
                PathComponent::ParentDir,
                PathComponent::Normal(XPathBuf::from("foo")),
                PathComponent::ParentDir,
                PathComponent::Normal(XPathBuf::from("bar")),
            ])
        );
        assert!(!has_trailing_slash);
    }

    #[test]
    fn test_path_components_trailing_slash_with_parent_dir() {
        let path = XPath::from_bytes(b"../foo/..");
        let (components, has_trailing_slash) = path_components(&path).unwrap();
        assert_eq!(
            components,
            VecDeque::from([
                PathComponent::ParentDir,
                PathComponent::Normal(XPathBuf::from("foo")),
                PathComponent::ParentDir,
            ])
        );
        assert!(has_trailing_slash);
    }

    #[test]
    fn test_path_components_leading_slashes_are_skipped() {
        let path = XPath::from_bytes(b"////..////bar");
        let (components, has_trailing_slash) = path_components(&path).unwrap();
        assert_eq!(
            components,
            VecDeque::from([
                PathComponent::ParentDir,
                PathComponent::Normal(XPathBuf::from("bar")),
            ])
        );
        assert!(!has_trailing_slash);
    }

    #[test]
    fn test_path_components_path_with_mixed_components_and_trailing_slash() {
        let path = XPath::from_bytes(b"../foo/../bar/");
        let (components, has_trailing_slash) = path_components(&path).unwrap();
        assert_eq!(
            components,
            VecDeque::from([
                PathComponent::ParentDir,
                PathComponent::Normal(XPathBuf::from("foo")),
                PathComponent::ParentDir,
                PathComponent::Normal(XPathBuf::from("bar")),
            ])
        );
        assert!(has_trailing_slash);
    }

    #[test]
    fn test_path_components_complex_path_with_leading_parent_dir() {
        let path = XPath::from_bytes(b"../foo/bar/../../baz/..");
        let (components, has_trailing_slash) = path_components(&path).unwrap();
        assert_eq!(
            components,
            VecDeque::from([
                PathComponent::ParentDir,
                PathComponent::Normal(XPathBuf::from("foo")),
                PathComponent::Normal(XPathBuf::from("bar")),
                PathComponent::ParentDir,
                PathComponent::ParentDir,
                PathComponent::Normal(XPathBuf::from("baz")),
                PathComponent::ParentDir,
            ])
        );
        assert!(has_trailing_slash);
    }

    #[test]
    fn test_path_components_root_path_is_handled_externally() {
        let path = XPath::from_bytes(b"/..");
        let (components, has_trailing_slash) = path_components(&path).unwrap();
        assert_eq!(components, VecDeque::from([PathComponent::ParentDir]));
        assert!(has_trailing_slash);
    }

    #[test]
    fn test_chdir_long() {
        // Create a deep directory structure.
        const MAX_DEPTH: usize = 128;
        let o = "o".repeat(200);
        let name = XPathBuf::from(format!("syd_test_chdir_l{o}ng.dir"));
        setup_deep_directory_structure(&name, MAX_DEPTH).unwrap();

        // Save current working directory.
        let fd = open(".", OFlag::O_PATH | OFlag::O_DIRECTORY, Mode::empty()).unwrap();

        // Attempt to change dir into it.
        let mut lname = name.clone();
        for _ in 0..MAX_DEPTH - 1 {
            lname.push(name.as_bytes());
        }
        let result = chdir_long(&lname);

        // Restore current working directory.
        fchdir(fd).unwrap();

        // Clean up after the test.
        Command::new("rm").arg("-rf").arg(name).status().unwrap();

        assert!(result.is_ok(), "chdir_long failed: {result:?}");
    }

    #[test]
    fn test_getdir_long_with_deep_structure() {
        // Create a deep directory structure.
        const MAX_DEPTH: usize = 128;
        let o = "o".repeat(200);
        let name = XPathBuf::from(format!("syd_test_getdir_l{o}ng_with_deep_structure.dir"));
        setup_deep_directory_structure(&name, MAX_DEPTH).unwrap();

        // Save current working directory.
        let fd = open(".", OFlag::O_PATH | OFlag::O_DIRECTORY, Mode::empty()).unwrap();
        let mut cwd = getcwd().map(XPathBuf::from).unwrap();

        // Attempt to change dir into it.
        let max = cwd.depth() + MAX_DEPTH + 1;
        for _ in 0..MAX_DEPTH {
            cwd.push(name.as_bytes());
        }
        let result = chdir_long(&cwd);

        // Get current working directory.
        let result_cwd = if result.is_ok() {
            let cwd_fd = open(".", OFlag::O_PATH | OFlag::O_DIRECTORY, Mode::empty()).unwrap();
            Some(getdir_long(cwd_fd, max))
        } else {
            None
        };

        // Restore current working directory.
        fchdir(fd).unwrap();

        // Clean up after the test.
        Command::new("rm").arg("-rf").arg(name).status().unwrap();

        // Check results.
        assert!(result.is_ok(), "chdir_long failed: {result:?}");

        let result_cwd = result_cwd.unwrap();
        assert!(result_cwd.is_ok(), "getdir_long failed: {result_cwd:?}");

        let result_cwd = result_cwd.unwrap();
        assert_eq!(cwd, result_cwd, "getdir_long returned incorrect directory");
    }

    #[test]
    fn test_getdir_long_limit_max_components() {
        // Create a deep directory structure.
        const MAX_DEPTH: usize = 128;
        let o = "o".repeat(200);
        let name = XPathBuf::from(format!("syd_test_getdir_l{o}ng_limit_max_components.dir"));
        setup_deep_directory_structure(&name, MAX_DEPTH).unwrap();

        // Save current working directory.
        let fd = open(".", OFlag::O_PATH | OFlag::O_DIRECTORY, Mode::empty()).unwrap();
        let mut cwd = getcwd().map(XPathBuf::from).unwrap();

        // Attempt to change dir into it.
        let max = cwd.depth() + MAX_DEPTH;
        for _ in 0..MAX_DEPTH {
            cwd.push(name.as_bytes());
        }
        let result = chdir_long(&cwd);

        // Get current working directory.
        let result_cwd = if result.is_ok() {
            let cwd_fd = open(".", OFlag::O_PATH | OFlag::O_DIRECTORY, Mode::empty()).unwrap();
            Some(getdir_long(cwd_fd, max))
        } else {
            None
        };

        // Restore current working directory.
        fchdir(fd).unwrap();

        // Clean up after the test.
        Command::new("rm").arg("-rf").arg(name).status().unwrap();

        // Check results.
        assert!(result.is_ok(), "chdir_long failed: {result:?}");

        let result_cwd = result_cwd.unwrap();
        assert_eq!(
            result_cwd,
            Err(Errno::ERANGE),
            "getdir_long did not fail as expected: {result_cwd:?}"
        );
    }

    #[test]
    fn test_fillrandom() {
        assert_eq!(fillrandom(&mut []), Err(Errno::EINVAL));

        assert_eq!(fillrandom(&mut [0u8; 257]), Ok(()));
    }

    #[test]
    fn test_getrandom() {
        assert_eq!(getrandom(0), Err(Errno::EINVAL));

        let result = getrandom(257);
        assert!(result.is_ok(), "result:{result:?}");
    }

    #[test]
    fn test_fd_status_flags_file_read_only() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().read(true).open(temp.path()).unwrap();
        let flags = fd_status_flags(&file).unwrap();

        assert!(!flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_fd_status_flags_file_write_only() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().write(true).open(temp.path()).unwrap();
        let flags = fd_status_flags(&file).unwrap();

        assert!(flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_fd_status_flags_file_read_write() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .read(true)
            .write(true)
            .open(temp.path())
            .unwrap();
        let flags = fd_status_flags(&file).unwrap();

        assert!(flags.contains(OFlag::O_RDWR));
        assert!(!flags.contains(OFlag::O_WRONLY));
    }

    #[test]
    fn test_fd_status_flags_owned_fd_read_only() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().read(true).open(temp.path()).unwrap();
        let owned_fd = unsafe { OwnedFd::from_raw_fd(file.as_raw_fd()) };
        std::mem::forget(file);

        let flags = fd_status_flags(&owned_fd).unwrap();
        assert!(!flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_fd_status_flags_owned_fd_write_only() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().write(true).open(temp.path()).unwrap();
        let owned_fd = unsafe { OwnedFd::from_raw_fd(file.as_raw_fd()) };
        std::mem::forget(file);

        let flags = fd_status_flags(&owned_fd).unwrap();
        assert!(flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_fd_status_flags_owned_fd_read_write() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .read(true)
            .write(true)
            .open(temp.path())
            .unwrap();
        let owned_fd = unsafe { OwnedFd::from_raw_fd(file.as_raw_fd()) };
        std::mem::forget(file);

        let flags = fd_status_flags(&owned_fd).unwrap();
        assert!(flags.contains(OFlag::O_RDWR));
        assert!(!flags.contains(OFlag::O_WRONLY));
    }

    #[test]
    fn test_fd_status_flags_borrowed_fd_read_only() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().read(true).open(temp.path()).unwrap();
        let borrowed_fd = file.as_fd();

        let flags = fd_status_flags(borrowed_fd).unwrap();
        assert!(!flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_fd_status_flags_borrowed_fd_write_only() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().write(true).open(temp.path()).unwrap();
        let borrowed_fd = file.as_fd();

        let flags = fd_status_flags(borrowed_fd).unwrap();
        assert!(flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_fd_status_flags_borrowed_fd_read_write() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .read(true)
            .write(true)
            .open(temp.path())
            .unwrap();
        let borrowed_fd = file.as_fd();

        let flags = fd_status_flags(borrowed_fd).unwrap();
        assert!(flags.contains(OFlag::O_RDWR));
        assert!(!flags.contains(OFlag::O_WRONLY));
    }

    #[test]
    fn test_fd_status_flags_dev_null_read() {
        let file = OpenOptions::new().read(true).open("/dev/null").unwrap();
        let flags = fd_status_flags(&file).unwrap();

        assert!(!flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_fd_status_flags_dev_null_write() {
        let file = OpenOptions::new().write(true).open("/dev/null").unwrap();
        let flags = fd_status_flags(&file).unwrap();

        assert!(flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_fd_status_flags_dev_null_read_write() {
        let file = OpenOptions::new()
            .read(true)
            .write(true)
            .open("/dev/null")
            .unwrap();
        let flags = fd_status_flags(&file).unwrap();

        assert!(flags.contains(OFlag::O_RDWR));
        assert!(!flags.contains(OFlag::O_WRONLY));
    }

    #[test]
    fn test_fd_status_flags_pipe_read_end() {
        let (read_fd, _) = pipe().unwrap();

        let flags = fd_status_flags(&read_fd).unwrap();
        assert!(!flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_fd_status_flags_pipe_write_end() {
        let (_, write_fd) = pipe().unwrap();

        let flags = fd_status_flags(&write_fd).unwrap();
        assert!(flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_fd_status_flags_append_mode() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .write(true)
            .append(true)
            .open(temp.path())
            .unwrap();
        let flags = fd_status_flags(&file).unwrap();

        assert!(flags.contains(OFlag::O_WRONLY));
        assert!(flags.contains(OFlag::O_APPEND));
    }

    #[test]
    fn test_fd_status_flags_create_mode() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .write(true)
            .create(true)
            .open(temp.path())
            .unwrap();
        let flags = fd_status_flags(&file).unwrap();

        assert!(flags.contains(OFlag::O_WRONLY));
    }

    #[test]
    fn test_fd_status_flags_truncate_mode() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .write(true)
            .truncate(true)
            .open(temp.path())
            .unwrap();
        let flags = fd_status_flags(&file).unwrap();

        assert!(flags.contains(OFlag::O_WRONLY));
    }

    #[test]
    fn test_fd_status_flags_read_append_mode() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .read(true)
            .append(true)
            .open(temp.path())
            .unwrap();
        let flags = fd_status_flags(&file).unwrap();

        assert!(flags.contains(OFlag::O_RDWR));
        assert!(flags.contains(OFlag::O_APPEND));
    }

    #[test]
    fn test_fd_status_flags_create_new_mode() {
        let temp = NamedTempFile::new().unwrap();
        std::fs::remove_file(temp.path()).unwrap();
        let file = OpenOptions::new()
            .write(true)
            .create_new(true)
            .open(temp.path())
            .unwrap();
        let flags = fd_status_flags(&file).unwrap();

        assert!(flags.contains(OFlag::O_WRONLY));
    }

    #[test]
    fn test_fd_status_flags_reference_to_file() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().read(true).open(temp.path()).unwrap();
        let file_ref = &file;

        let flags = fd_status_flags(file_ref).unwrap();
        assert!(!flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_fd_status_flags_mutable_reference_to_file() {
        let temp = NamedTempFile::new().unwrap();
        let mut file = OpenOptions::new().write(true).open(temp.path()).unwrap();
        let file_ref = &mut file;

        let flags = fd_status_flags(file_ref).unwrap();
        assert!(flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_fd_status_flags_box_file() {
        let temp = NamedTempFile::new().unwrap();
        let file = Box::new(OpenOptions::new().read(true).open(temp.path()).unwrap());

        let flags = fd_status_flags(&file).unwrap();
        assert!(!flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_fd_status_flags_arc_file() {
        use std::sync::Arc;
        let temp = NamedTempFile::new().unwrap();
        let file = Arc::new(OpenOptions::new().read(true).open(temp.path()).unwrap());

        let flags = fd_status_flags(&file).unwrap();
        assert!(!flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_fd_status_flags_rc_file() {
        use std::rc::Rc;
        let temp = NamedTempFile::new().unwrap();
        let file = Rc::new(OpenOptions::new().read(true).open(temp.path()).unwrap());

        let flags = fd_status_flags(&file).unwrap();
        assert!(!flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_fd_status_flags_invalid_fd() {
        let result = fd_status_flags(AT_BADFD);

        assert!(result.is_err());
        assert_eq!(result.unwrap_err(), Errno::EBADF);
    }

    #[test]
    fn test_fd_status_flags_multiple_calls_consistency() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .read(true)
            .write(true)
            .open(temp.path())
            .unwrap();

        let flags1 = fd_status_flags(&file).unwrap();
        let flags2 = fd_status_flags(&file).unwrap();
        let flags3 = fd_status_flags(&file).unwrap();

        assert_eq!(flags1, flags2);
        assert_eq!(flags2, flags3);
    }

    #[test]
    fn test_fd_status_flags_different_file_types() {
        let temp = NamedTempFile::new().unwrap();
        let file1 = OpenOptions::new().write(true).open(temp.path()).unwrap();
        let file2 = OpenOptions::new().write(true).open("/dev/null").unwrap();

        let flags1 = fd_status_flags(&file1).unwrap();
        let flags2 = fd_status_flags(&file2).unwrap();

        assert!(flags1.contains(OFlag::O_WRONLY));
        assert!(flags2.contains(OFlag::O_WRONLY));
    }

    #[test]
    fn test_fd_status_flags_dup_file_descriptor() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().read(true).open(temp.path()).unwrap();
        let duped_fd = dup(&file).unwrap();

        let flags = fd_status_flags(&duped_fd).unwrap();
        assert!(!flags.contains(OFlag::O_WRONLY));
        assert!(!flags.contains(OFlag::O_RDWR));
    }

    #[test]
    fn test_is_writable_fd_file_read_only() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().read(true).open(temp.path()).unwrap();
        let result = is_writable_fd(&file).unwrap();

        assert!(!result);
    }

    #[test]
    fn test_is_writable_fd_file_write_only() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().write(true).open(temp.path()).unwrap();
        let result = is_writable_fd(&file).unwrap();

        assert!(result);
    }

    #[test]
    fn test_is_writable_fd_file_read_write() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .read(true)
            .write(true)
            .open(temp.path())
            .unwrap();
        let result = is_writable_fd(&file).unwrap();

        assert!(result);
    }

    #[test]
    fn test_is_writable_fd_owned_fd_read_only() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().read(true).open(temp.path()).unwrap();
        let owned_fd = unsafe { OwnedFd::from_raw_fd(file.as_raw_fd()) };
        std::mem::forget(file);

        let result = is_writable_fd(&owned_fd).unwrap();
        assert!(!result);
    }

    #[test]
    fn test_is_writable_fd_owned_fd_write_only() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().write(true).open(temp.path()).unwrap();
        let owned_fd = unsafe { OwnedFd::from_raw_fd(file.as_raw_fd()) };
        std::mem::forget(file);

        let result = is_writable_fd(&owned_fd).unwrap();
        assert!(result);
    }

    #[test]
    fn test_is_writable_fd_owned_fd_read_write() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .read(true)
            .write(true)
            .open(temp.path())
            .unwrap();
        let owned_fd = unsafe { OwnedFd::from_raw_fd(file.as_raw_fd()) };
        std::mem::forget(file);

        let result = is_writable_fd(&owned_fd).unwrap();
        assert!(result);
    }

    #[test]
    fn test_is_writable_fd_borrowed_fd_read_only() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().read(true).open(temp.path()).unwrap();
        let borrowed_fd = file.as_fd();

        let result = is_writable_fd(borrowed_fd).unwrap();
        assert!(!result);
    }

    #[test]
    fn test_is_writable_fd_borrowed_fd_write_only() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().write(true).open(temp.path()).unwrap();
        let borrowed_fd = file.as_fd();

        let result = is_writable_fd(borrowed_fd).unwrap();
        assert!(result);
    }

    #[test]
    fn test_is_writable_fd_borrowed_fd_read_write() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .read(true)
            .write(true)
            .open(temp.path())
            .unwrap();
        let borrowed_fd = file.as_fd();

        let result = is_writable_fd(borrowed_fd).unwrap();
        assert!(result);
    }

    #[test]
    fn test_is_writable_fd_dev_null_read() {
        let file = OpenOptions::new().read(true).open("/dev/null").unwrap();
        let result = is_writable_fd(&file).unwrap();

        assert!(!result);
    }

    #[test]
    fn test_is_writable_fd_dev_null_write() {
        let file = OpenOptions::new().write(true).open("/dev/null").unwrap();
        let result = is_writable_fd(&file).unwrap();

        assert!(result);
    }

    #[test]
    fn test_is_writable_fd_dev_null_read_write() {
        let file = OpenOptions::new()
            .read(true)
            .write(true)
            .open("/dev/null")
            .unwrap();
        let result = is_writable_fd(&file).unwrap();

        assert!(result);
    }

    #[test]
    fn test_is_writable_fd_pipe_read_end() {
        let (read_fd, _) = pipe().unwrap();

        let result = is_writable_fd(&read_fd).unwrap();
        assert!(!result);
    }

    #[test]
    fn test_is_writable_fd_pipe_write_end() {
        let (_, write_fd) = pipe().unwrap();

        let result = is_writable_fd(&write_fd).unwrap();
        assert!(result);
    }

    #[test]
    fn test_is_writable_fd_append_mode() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .write(true)
            .append(true)
            .open(temp.path())
            .unwrap();
        let result = is_writable_fd(&file).unwrap();

        assert!(result);
    }

    #[test]
    fn test_is_writable_fd_create_mode() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .write(true)
            .create(true)
            .open(temp.path())
            .unwrap();
        let result = is_writable_fd(&file).unwrap();

        assert!(result);
    }

    #[test]
    fn test_is_writable_fd_truncate_mode() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .write(true)
            .truncate(true)
            .open(temp.path())
            .unwrap();
        let result = is_writable_fd(&file).unwrap();

        assert!(result);
    }

    #[test]
    fn test_is_writable_fd_read_append_mode() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .read(true)
            .append(true)
            .open(temp.path())
            .unwrap();
        let result = is_writable_fd(&file).unwrap();

        assert!(result);
    }

    #[test]
    fn test_is_writable_fd_create_new_mode() {
        let temp = NamedTempFile::new().unwrap();
        std::fs::remove_file(temp.path()).unwrap();
        let file = OpenOptions::new()
            .write(true)
            .create_new(true)
            .open(temp.path())
            .unwrap();
        let result = is_writable_fd(&file).unwrap();

        assert!(result);
    }

    #[test]
    fn test_is_writable_fd_read_only_with_create() {
        let temp = NamedTempFile::new().unwrap();
        let file = open(
            temp.path(),
            OFlag::O_RDONLY | OFlag::O_CREAT | OFlag::O_TRUNC,
            Mode::empty(),
        )
        .map(File::from)
        .unwrap();
        let result = is_writable_fd(&file).unwrap();

        assert!(!result);
    }

    #[test]
    fn test_is_writable_fd_reference_to_file() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().read(true).open(temp.path()).unwrap();
        let file_ref = &file;

        let result = is_writable_fd(file_ref).unwrap();
        assert!(!result);
    }

    #[test]
    fn test_is_writable_fd_mutable_reference_to_file() {
        let temp = NamedTempFile::new().unwrap();
        let mut file = OpenOptions::new().write(true).open(temp.path()).unwrap();
        let file_ref = &mut file;

        let result = is_writable_fd(file_ref).unwrap();
        assert!(result);
    }

    #[test]
    fn test_is_writable_fd_box_file() {
        let temp = NamedTempFile::new().unwrap();
        let file = Box::new(OpenOptions::new().read(true).open(temp.path()).unwrap());

        let result = is_writable_fd(&file).unwrap();
        assert!(!result);
    }

    #[test]
    fn test_is_writable_fd_arc_file() {
        use std::sync::Arc;
        let temp = NamedTempFile::new().unwrap();
        let file = Arc::new(OpenOptions::new().read(true).open(temp.path()).unwrap());

        let result = is_writable_fd(&file).unwrap();
        assert!(!result);
    }

    #[test]
    fn test_is_writable_fd_rc_file() {
        use std::rc::Rc;
        let temp = NamedTempFile::new().unwrap();
        let file = Rc::new(OpenOptions::new().read(true).open(temp.path()).unwrap());

        let result = is_writable_fd(&file).unwrap();
        assert!(!result);
    }

    #[test]
    fn test_is_writable_fd_invalid_fd() {
        let result = is_writable_fd(AT_BADFD);

        assert!(result.is_err());
        assert_eq!(result.unwrap_err(), Errno::EBADF);
    }

    #[test]
    fn test_is_writable_fd_multiple_calls_consistency() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new()
            .read(true)
            .write(true)
            .open(temp.path())
            .unwrap();

        let result1 = is_writable_fd(&file).unwrap();
        let result2 = is_writable_fd(&file).unwrap();
        let result3 = is_writable_fd(&file).unwrap();

        assert_eq!(result1, result2);
        assert_eq!(result2, result3);
    }

    #[test]
    fn test_is_writable_fd_different_file_types() {
        let temp = NamedTempFile::new().unwrap();
        let file1 = OpenOptions::new().write(true).open(temp.path()).unwrap();
        let file2 = OpenOptions::new().write(true).open("/dev/null").unwrap();

        let result1 = is_writable_fd(&file1).unwrap();
        let result2 = is_writable_fd(&file2).unwrap();

        assert!(result1);
        assert!(result2);
    }

    #[test]
    fn test_is_writable_fd_dup_file_descriptor() {
        let temp = NamedTempFile::new().unwrap();
        let file = OpenOptions::new().read(true).open(temp.path()).unwrap();
        let duped_fd = dup(&file).unwrap();

        let result = is_writable_fd(&duped_fd).unwrap();
        assert!(!result);
    }

    #[test]
    fn test_base_offset_root_and_non_root() {
        // Mirrors the computation used in CanonicalPath::new_map.
        let off = |parent_len: usize| parent_len + usize::from(parent_len > 1);

        assert_eq!(off(1), 1, "root parent must not drop first byte");
        assert_eq!(off(5), 6, "non-root parent must skip one separator");
    }
}
