/* Tests a kernel with a constant array passed to a function with an infinite 
   loop.

   Copyright (c) 2012 Pekka Jääskeläinen / Tampere University of Technology
   
   Permission is hereby granted, free of charge, to any person obtaining a copy
   of this software and associated documentation files (the "Software"), to deal
   in the Software without restriction, including without limitation the rights
   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
   copies of the Software, and to permit persons to whom the Software is
   furnished to do so, subject to the following conditions:
   
   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.
   
   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
   THE SOFTWARE.
*/

#include "pocl_opencl.h"

// Enable OpenCL C++ exceptions
#define CL_HPP_ENABLE_EXCEPTIONS

#include <CL/opencl.hpp>

#include <cstdio>
#include <cstdlib>
#include <iostream>

#ifndef _MSC_VER
#  include <unistd.h>
#else
#  include "vccompat.hpp"
#endif

static char
kernelSourceCode[] =
"kernel void echo(int input, global int *output)\n"
"{\n"
"  output[input] = input + 1;\n"
"}\n";

int
main(void)
{
  std::vector<cl::Platform> platformList;
  bool ok = false;
  try {

    // Pick platform
    cl::Platform::get(&platformList);

    // Pick first platform
    cl_context_properties cprops[] = {
        CL_CONTEXT_PLATFORM, (cl_context_properties)(platformList[0])(), 0};
    cl::Context context(CL_DEVICE_TYPE_ALL, cprops);

    // Query the set of devices attched to the context
    std::vector<cl::Device> devices = context.getInfo<CL_CONTEXT_DEVICES>();

    // Create and program from source
    cl::Program::Sources sources({kernelSourceCode});
    cl::Program program(context, sources);

    // Build program
    program.build(devices);

    int output[2];
    // Create buffer for that uses the host ptr C
    cl::Buffer outBuffer =
        cl::Buffer(context, CL_MEM_WRITE_ONLY | CL_MEM_USE_HOST_PTR,
                   2 * sizeof(int), (void *)&output[0]);

    // Create command queue
    cl::CommandQueue queue(context, devices[0], 0);

    // Create kernel object
    cl::Kernel kernel(program, "echo");

    int arg = 0;
    // Set kernel args
    kernel.setArg(0, sizeof(int), &arg);
    kernel.setArg(1, outBuffer);

    // Queue the first launch cmd.
    queue.enqueueNDRangeKernel(kernel, cl::NullRange, cl::NDRange(1),
                               cl::NullRange);

    arg = 1;
    // Now change the argument. It should not override the first one.
    // On the other hand, the old 2nd arg should be remembered.
    kernel.setArg(0, sizeof(int), &arg);

    // Queue the 2nd launch cmd.
    queue.enqueueNDRangeKernel(kernel, cl::NullRange, cl::NDRange(1),
                               cl::NullRange);

    int *res = (int *)queue.enqueueMapBuffer(outBuffer, CL_TRUE, CL_MAP_READ, 0,
                                             2 * sizeof(int));

    ok = true;
    if (!(res[0] == 1 && res[1] == 2)) {
      std::cerr << res[0] << res[1] << std::endl;
      ok = false;
    }

    queue.enqueueUnmapMemObject(outBuffer, (void *)res);
    queue.finish();
  } catch (cl::Error &err) {
    std::cerr << "ERROR: " << err.what() << "(" << err.err() << ")"
              << std::endl;
    return EXIT_FAILURE;
  }

  platformList[0].unloadCompiler();

  if (ok) {
    std::cout << "OK" << std::endl;
    return EXIT_SUCCESS;
  } else {
    std::cout << "FAIL\n";
    return EXIT_FAILURE;
  }
}
