#!/usr/bin/env python3
# SPDX-License-Identifier: Apache-2.0
# Copyright 2013-2016 The Meson development team

'''Runs the basic test suite through a cross compiler.

This is now just a wrapper around run_project_tests.py with specific arguments
'''

import argparse
import subprocess
from mesonbuild import mesonlib
from mesonbuild.coredata import version as meson_version
from pathlib import Path
import json
import os


def runtests(cross_file, failfast, cross_only, test_list, env=None):
    tests = ['--only'] + test_list
    if not cross_only:
        tests.append('native')
    cmd = mesonlib.python_command + ['run_project_tests.py', '--backend', 'ninja']
    if failfast:
        cmd += ['--failfast']
    cmd += tests
    cmd += ['--cross-file', cross_file]
    if cross_only:
        cmd += ['--native-file', 'cross/none.txt']
    return subprocess.call(cmd, env=env)

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--failfast', action='store_true')
    parser.add_argument('--cross-only', action='store_true')
    parser.add_argument('cross_file')
    options = parser.parse_args()
    cf_path = Path(options.cross_file)
    try:
        data = json.loads(cf_path.read_text(encoding='utf-8'))
        real_cf = cf_path.resolve().parent / data['file']
        assert real_cf.exists()
        env = os.environ.copy()
        env.update(data['env'])
        return runtests(real_cf.as_posix(), options.failfast, options.cross_only, data['tests'], env=env)
    except Exception:
        return runtests(options.cross_file, options.failfast, options.cross_only, ['common'])

if __name__ == '__main__':
    print('Meson build system', meson_version, 'Cross Tests')
    raise SystemExit(main())
