#!/usr/bin/tclsh
#
# This scripts writes a input file for packmol, and runs packmol, 
# that automatically setup
# a solvation shell with 0.16 sodium and chloride concentrations such
# that the whole system is neutral, considering neutral histidines and
# charged
#
# Make it executable with: chmod +x solvate.tcl
# Run with: ./solvate.tcl structure.pdb -shell 15. -charge +5 -density 1.0 -o solvated.pdb
#
# Where: 
#        structure.pdb is the pdb file to be solvated (usually a protein)
# 
#        "15." is the size of the solvation shell. This
#        is an optional parameter. If not set, 15. will be used.
# 
#        +5 is the total charge of the system, to be neutralized.
#        This is also and optional parameter, if not used, the package
#        considers histidine residues as neutral, Arg and Lys as +1
#        and Glu and Asp as -1, and sets up the Sodium and Chloride
#        concentrations automatically.
#        Alternatively, use the -noions to not add any ions, just water.
#
#        1.0 is the desired density. Optional. If not set, the density
#        will be set to 1.0 g/ml.
#
#        solvated.pdb: is the (optional) name for the solvated system output
#        file. If this argument is not provided, it will be the default
#        "solvated.pdb" file.
#
#        pack.inp: is the (optional) name for the packmol input file that
#        will be generated. If not provided, packmol_input.inp will be used.
#
# L. Martinez, Aug 2, 2011. (leandromartinez98@gmail.com)
#

set args [ split $argv " " ]
set iarg 0
foreach arg $args {
  if { [ string trim $arg ] > " " } {
    incr iarg
    set input_argument($iarg) $arg
  }
}
if { $iarg == 0 } {
puts " " 
puts " Solvate.tcl script for Packmol "
puts " "
puts " Run with: "
puts " "
puts " solvate.tcl structure.pdb -shell 15. -charge +5 -density 1.0 -i pack.inp -o solvated.pdb"
puts "
 Where: 
        structure.pdb is the pdb file to be solvated (usually a protein)
 
        \"15.\" is the size of the solvation shell. This
        is an optional parameter. If not set, 15. will be used.
 
        +5 is the total charge of the system, to be neutralized.
        This is also and optional parameter, if not used, the package
        considers histidine residues as neutral, Arg and Lys as +1
        and Glu and Asp as -1. The Na+ and Cl- concentrations are set
        the closest possible to 0.16M, approximately the physiological
        concentration.
        Alternatively, use the -noions to not add any ions, just water.

        1.0 is the desired density. Optional. If not set, the density
        will be set to 1.0 g/ml.

        solvated.pdb: is the (optional) name for the solvated system output
        file. If this argument is not provided, it will be the default
        solvated.pdb file.

        pack.inp: is the (optional) name for the packmol input file that
        will be generated. If not provided, packmol_input.inp will be used.
"
exit
}

set pdb_file $input_argument(1)
set charge "none"
set shell "none"
set density "none"
set output "solvated.pdb"
set packmol_input "packmol_input.inp"
set noions "false"
for { set i 2 } { $i <= $iarg } { incr i } {
  if { $input_argument($i) == "-charge" } { set charge $input_argument([expr $i + 1]) }
  if { $input_argument($i) == "-shell" } { set shell $input_argument([expr $i + 1]) }
  if { $input_argument($i) == "-density" } { set density $input_argument([expr $i + 1]) }
  if { $input_argument($i) == "-o" } { set output $input_argument([expr $i + 1]) }
  if { $input_argument($i) == "-i" } { set packmol_input $input_argument([expr $i + 1]) }
  if { $input_argument($i) == "-noions" } { set noions "true" }
}

puts " ###########################################################################"
puts " solvate.tcl script: Solvates a (protein) structure with water and ions,"
puts "                     using Packmol. "
puts " ###########################################################################"

puts " Input pdb file to be solvated: $pdb_file "
puts -nonewline " User set structure charge: $charge "
if { $charge == "none" } { puts " - will compute from structure. " } else { puts " " } 
puts -nonewline " User set shell size: $shell "
if { $shell == "none" } { puts " - using default: 15.0 Angs. " } else { puts " " }
puts -nonewline " User set density: $density "
if { $density == "none" } { puts " - using default: 1.0 g/ml " } else { puts " " }

# If the user didn't set the sell size, use default one

if { $shell == "none" } { set shell 15. }

#
# Computing the size of the structure (maximum and minimum size on each direction)
#

set pdb_file_name $pdb_file
set pdb_file [ open $pdb_file r ]
set pdb_file [ read $pdb_file ]
set pdb_file [ split $pdb_file "\n" ]

set natoms 0
foreach line $pdb_file {
  if { [ string range $line 0 3 ] == "ATOM" |
       [ string range $line 0 5 ] == "HETATM" } {
    incr natoms
    set x [ string trim [ string range $line 30 37 ] ]
    set y [ string trim [ string range $line 38 45 ] ]
    set z [ string trim [ string range $line 46 54 ] ]
    if { [ info exists xmin ] == 1 } {
      if { $x < $xmin } { set xmin $x }
      if { $y < $ymin } { set ymin $y }
      if { $z < $zmin } { set zmin $z }
      if { $x > $xmax } { set xmax $x }
      if { $y > $ymax } { set ymax $y }
      if { $z > $zmax } { set zmax $z }
    } else {
      set xmin $x
      set ymin $y
      set zmin $z
      set xmax $x
      set ymax $y
      set zmax $z
    } 
  }
}

set x_length [ expr $xmax - $xmin + 2.0*$shell ] 
set y_length [ expr $ymax - $ymin + 2.0*$shell ] 
set z_length [ expr $zmax - $zmin + 2.0*$shell ] 

puts " -------------------------------------------------------"
puts "  Minimum and maximum coordinates of structure atoms "
puts "  X_min = $xmin     X_max = $xmax " 
puts "  Y_min = $ymin     Y_max = $ymax " 
puts "  Z_min = $zmin     Z_max = $zmax " 
puts " -------------------------------------------------------"
puts "  Box side length in each direction: "
puts "  x: $x_length "
puts "  y: $y_length "
puts "  z: $z_length "
puts " -------------------------------------------------------"

#
# If the user didn't set the total charge of the system, compute the charge
#

if { $charge == "none" } { 
  set charge 0
  set number 0
  set nhis 0; set narg 0; set nlys 0; set nglu 0; set nasp 0
  foreach line $pdb_file {
    set amino [string range $line 17 19]
    if { [ string trim [ string range $line 12 15 ] ] == "N" } { 
      if { $amino == "HIS" } { incr nhis }
      if { $amino == "HSD" } { incr nhis }
      if { $amino == "ARG" } { set charge [expr $charge + 1]; incr narg }
      if { $amino == "LYS" } { set charge [expr $charge + 1]; incr nlys }
      if { $amino == "GLU" } { set charge [expr $charge - 1]; incr nglu }
      if { $amino == "ASP" } { set charge [expr $charge - 1]; incr nasp }
    }
  }
  puts " -------------------------------------------------------"
  puts "  HIS = $nhis (associated charge = 0) "
  puts "  ARG = $narg (associated charge = +$narg) "
  puts "  LYS = $nlys (associated charge = +$nlys) "
  puts "  GLU = $nglu (associated charge = -$nglu) "
  puts "  ASP = $nasp (associated charge = -$nasp) "
  puts " -------------------------------------------------------"
  puts "  Total structure charge = $charge"
  puts " -------------------------------------------------------"
}

#
# Compute the molar mass of the structure
#

foreach data { \ 
"H      1.00800 " \
"HC     1.00800 " \
"HA     1.00800 " \
"HT     1.00800 " \
"HP     1.00800 " \
"HB     1.00800 " \
"HR1    1.00800 " \
"HR2    1.00800 " \
"HR3    1.00800 " \
"HS     1.00800 " \
"HE1    1.00800 " \
"HE2    1.00800 " \
"HA1    1.00800 " \
"HA2    1.00800 " \
"HA3    1.00800 " \
"HF1    1.00800 " \
"HF2    1.00800 " \
"C     12.01100 " \
"CA    12.01100 " \
"CT1   12.01100 " \
"CT2   12.01100 " \
"CT3   12.01100 " \
"CPH1  12.01100 " \
"CPH2  12.01100 " \
"CPT   12.01100 " \
"CY    12.01100 " \
"CP1   12.01100 " \
"CP2   12.01100 " \
"CP3   12.01100 " \
"CC    12.01100 " \
"CD    12.01100 " \
"CPA   12.01100 " \
"CPB   12.01100 " \
"CPM   12.01100 " \
"CM    12.01100 " \
"CS    12.01100 " \
"CE1   12.01100 " \
"CE2   12.01100 " \
"CST   12.01100 " \
"CT    12.01100 " \
"CT1x  12.01100 " \
"CT2x  12.01100 " \
"CT3x  12.01100 " \
"CN    12.01100 " \
"CAP   12.01100 " \
"COA   12.01100 " \
"C3    12.01100 " \
"N     14.00700 " \
"NR1   14.00700 " \
"NR2   14.00700 " \
"NR3   14.00700 " \
"NH1   14.00700 " \
"NH2   14.00700 " \
"NH3   14.00700 " \
"NC2   14.00700 " \
"NY    14.00700 " \
"NP    14.00700 " \
"NPH   14.00700 " \
"NC    14.00700 " \
"O     15.99900 " \
"OB    15.99900 " \
"OC    15.99900 " \
"OH1   15.99900 " \
"OS    15.99940 " \
"OT    15.99940 " \
"OM    15.99900 " \
"OST   15.99900 " \
"OCA   15.99900 " \
"S     32.06000 " \
"SM    32.06000 " \
"SS    32.06000 " \
"HE     4.00260 " \
"NE    20.17970 " \
"CF1   12.01100 " \
"CF2   12.01100 " \
"CF3   12.01100 " \
"FE    55.84700 " \
"CLAL  35.45300 " \
"FA    18.99800 " \
"F1    18.99800 " \
"F2    18.99800 " \
"F3    18.99800 " \
"DUM    0.00000 " \
"SOD  22.989770 " \
"MG   24.305000 " \
"POT  39.102000 " \
"CES 132.900000 " \
"CAL  40.080000 " \
"CLA  35.450000 " \
"ZN   65.370000 " \
"H      1.00800 " \
"HC     1.00800 " \
"HA     1.00800 " \
"HT     1.00800 " \
"HP     1.00800 " \
"HB     1.00800 " \
"HR1    1.00800 " \
"HR2    1.00800 " \
"HR3    1.00800 " \
"HS     1.00800 " \
"HE1    1.00800 " \
"HE2    1.00800 " \
"HA1    1.00800 " \
"HA2    1.00800 " \
"HA3    1.00800 " \
"HF1    1.00800 " \
"HF2    1.00800 " \
"C     12.01100 " \
"CA    12.01100 " \
"CT1   12.01100 " \
"CT2   12.01100 " \
"CT3   12.01100 " \
"CPH1  12.01100 " \
"CPH2  12.01100 " \
"CPT   12.01100 " \
"CY    12.01100 " \
"CP1   12.01100 " \
"CP2   12.01100 " \
"CP3   12.01100 " \
"CC    12.01100 " \
"CD    12.01100 " \
"CPA   12.01100 " \
"CPB   12.01100 " \
"CPM   12.01100 " \
"CM    12.01100 " \
"CS    12.01100 " \
"CE1   12.01100 " \
"CE2   12.01100 " \
"CST   12.01100 " \
"CT    12.01100 " \
"CT1x  12.01100 " \
"CT2x  12.01100 " \
"CT3x  12.01100 " \
"CN    12.01100 " \
"CAP   12.01100 " \
"COA   12.01100 " \
"C3    12.01100 " \
"N     14.00700 " \
"NR1   14.00700 " \
"NR2   14.00700 " \
"NR3   14.00700 " \
"NH1   14.00700 " \
"NH2   14.00700 " \
"NH3   14.00700 " \
"NC2   14.00700 " \
"NY    14.00700 " \
"NP    14.00700 " \
"NPH   14.00700 " \
"NC    14.00700 " \
"O     15.99900 " \
"OB    15.99900 " \
"OC    15.99900 " \
"OH1   15.99900 " \
"OS    15.99940 " \
"OT    15.99940 " \
"OM    15.99900 " \
"OST   15.99900 " \
"OCA   15.99900 " \
"S     32.06000 " \
"SM    32.06000 " \
"SS    32.06000 " \
"HE     4.00260 " \
"NE    20.17970 " \
"CF1   12.01100 " \
"CF2   12.01100 " \
"CF3   12.01100 " \
"FE    55.84700 " \
"CLAL  35.45300 " \
"FA    18.99800 " \
"F1    18.99800 " \
"F2    18.99800 " \
"F3    18.99800 " \
"DUM    0.00000 " \
"SOD  22.989770 " \
"MG   24.305000 " \
"POT  39.102000 " \
"CES 132.900000 " \
"CAL  40.080000 " \
"CLA  35.450000 " \
"ZN   65.370000 " \
"HT     1.00800 " \
"OT    15.99940 " \
"HAS    1.00800 " \
"HOS    1.00800 " \
"CTS   12.01100 " \
"CBS   12.01100 " \
"OHS   15.99940 " \
"OES   15.99940 " \
"OXT   15.99940 " \
"HL    1.008000 " \
"HCL   1.008000 " \
"HT    1.008000 " \
"HOL   1.008000 " \
"HAL1  1.008000 " \
"HAL2  1.008000 " \
"HAL3  1.008000 " \
"HEL1  1.008000 " \
"HEL2  1.008000 " \
"CL   12.011000 " \
"CTL1 12.011000 " \
"CTL2 12.011000 " \
"CTL3 12.011000 " \
"CTL5 12.011000 " \
"CEL1 12.011000 " \
"CEL2 12.011000 " \
"NTL  14.007000 " \
"NH3L 14.007000 " \
"OBL  15.999400 " \
"OCL  15.999400 " \
"OT   15.999400 " \
"OSL  15.999400 " \
"O2L  15.999400 " \
"OHL  15.999400 " \
"PL   30.974000 " \
"SL   32.060000 " \
"SOD  22.989770 " \
"MG   24.305000 " \
"POT  39.102000 " \
"CES 132.900000 " \
"CAL  40.080000 " \
"CLA  35.450000 " \
"ZN   65.370000 " \
"DUM   0.000000 " \
"HT    1.008000 " \
"HN1   1.008000 " \
"HN2   1.008000 " \
"HN3   1.008000 " \
"HN3B  1.008000 " \
"HN3C  1.008000 " \
"HNP   1.008000 " \
"HN4   1.008000 " \
"HN5   1.008000 " \
"HN6   1.008000 " \
"HN7   1.008000 " \
"HN8   1.008000 " \
"HN9   1.008000 " \
"HNE1  1.008000 " \
"HNE2  1.008000 " \
"CN1  12.011000 " \
"CN1A 12.011000 " \
"CN1T 12.011000 " \
"CN2  12.011000 " \
"CN3  12.011000 " \
"CN3A 12.011000 " \
"CN3B 12.011000 " \
"CN3C 12.011000 " \
"CN3D 12.011000 " \
"CN3T 12.011000 " \
"CN4  12.011000 " \
"CN5  12.011000 " \
"CN5G 12.011000 " \
"CN7  12.011000 " \
"CN7B 12.011000 " \
"CN7C 12.011000 " \
"CN7D 12.011000 " \
"CN8  12.011000 " \
"CN8B 12.011000 " \
"CN9  12.011000 " \
"CNE1 12.011000 " \
"CNE2 12.011000 " \
"CNA  12.011000 " \
"CNA2 12.011000 " \
"CN6  12.011000 " \
"CN7E 12.011000 " \
"NN1  14.007000 " \
"NN1C 14.007000 " \
"NN2  14.007000 " \
"NN2B 14.007000 " \
"NN2C 14.007000 " \
"NN2U 14.007000 " \
"NN2G 14.007000 " \
"NN3  14.007000 " \
"NN3A 14.007000 " \
"NN3I 14.007000 " \
"NN3G 14.007000 " \
"NN4  14.007000 " \
"NN5  14.007000 " \
"NN6  14.007000 " \
"OT   15.999400 " \
"ON1  15.999400 " \
"ON1C 15.999400 " \
"ON2  15.999400 " \
"ON3  15.999400 " \
"ON4  15.999400 " \
"ON5  15.999400 " \
"ON6  15.999400 " \
"ON6B 15.999400 " \
"ON2B 15.999400 " \
"FN1  18.998400 " \
"FNA  18.998400 " \
"P    30.974000 " \
"P2   30.974000 " \
"P3   30.974000 " \
"SOD  22.989770 " \
"MG   24.305000 " \
"POT  39.102000 " \
"CES 132.900000 " \
"CAL  40.080000 " \
"CLA  35.450000 " \
"ZN   65.370000 " \
"DUM   0.001    " \
"CPH1 12.011000 " \
"CPH2 12.011000 " \
"HR3   1.008000 " \
"HR1   1.008000 " \
"NR1  14.007000 " \
"NR2  14.007000 " \
"HL    1.008000 " \
"HCL   1.008000 " \
"HT    1.008000 " \
"HOL   1.008000 " \
"HAL1  1.008000 " \
"HAL2  1.008000 " \
"HAL3  1.008000 " \
"HEL1  1.008000 " \
"HEL2  1.008000 " \
"CL   12.011000 " \
"CTL1 12.011000 " \
"CTL2 12.011000 " \
"CTL3 12.011000 " \
"CTL5 12.011000 " \
"CEL1 12.011000 " \
"CEL2 12.011000 " \
"NTL  14.007000 " \
"NH3L 14.007000 " \
"OBL  15.999400 " \
"OCL  15.999400 " \
"OT   15.999400 " \
"OSL  15.999400 " \
"O2L  15.999400 " \
"OHL  15.999400 " \
"PL   30.974000 " \
"SL   32.060000 " \
"CPH1 12.011000 " \
"CPH2 12.011000 " \
"HR3   1.008000 " \
"HR1   1.008000 " \
"NR1  14.007000 " \
"NR2  14.007000 " \
"HT    1.008000 " \
"HN1   1.008000 " \
"HN2   1.008000 " \
"HN3   1.008000 " \
"HN3B  1.008000 " \
"HN3C  1.008000 " \
"HNP   1.008000 " \
"HN4   1.008000 " \
"HN5   1.008000 " \
"HN6   1.008000 " \
"HN7   1.008000 " \
"HN8   1.008000 " \
"HN9   1.008000 " \
"HNE1  1.008000 " \
"HNE2  1.008000 " \
"CN1  12.011000 " \
"CN1A 12.011000 " \
"CN1T 12.011000 " \
"CN2  12.011000 " \
"CN3  12.011000 " \
"CN3A 12.011000 " \
"CN3B 12.011000 " \
"CN3C 12.011000 " \
"CN3D 12.011000 " \
"CN3T 12.011000 " \
"CN4  12.011000 " \
"CN5  12.011000 " \
"CN5G 12.011000 " \
"CN7  12.011000 " \
"CN7B 12.011000 " \
"CN7C 12.011000 " \
"CN7D 12.011000 " \
"CN8  12.011000 " \
"CN8B 12.011000 " \
"CN9  12.011000 " \
"CNE1 12.011000 " \
"CNE2 12.011000 " \
"CNA  12.011000 " \
"CNA2 12.011000 " \
"CN6  12.011000 " \
"CN7E 12.011000 " \
"NN1  14.007000 " \
"NN1C 14.007000 " \
"NN2  14.007000 " \
"NN2B 14.007000 " \
"NN2C 14.007000 " \
"NN2U 14.007000 " \
"NN2G 14.007000 " \
"NN3  14.007000 " \
"NN3A 14.007000 " \
"NN3I 14.007000 " \
"NN3G 14.007000 " \
"NN4  14.007000 " \
"NN5  14.007000 " \
"NN6  14.007000 " \
"OT   15.999400 " \
"ON1  15.999400 " \
"ON1C 15.999400 " \
"ON2  15.999400 " \
"ON3  15.999400 " \
"ON4  15.999400 " \
"ON5  15.999400 " \
"ON6  15.999400 " \
"ON6B 15.999400 " \
"ON2B 15.999400 " \
"FN1  18.998400 " \
"FNA  18.998400 " \
"P    30.974000 " \
"P2   30.974000 " \
"P3   30.974000 " \
"SOD  22.989770 " \
"MG   24.305000 " \
"POT  39.102000 " \
"CES 132.900000 " \
"CAL  40.080000 " \
"CLA  35.450000 " \
"ZN   65.370000 " \
"DUM   0.001    " \
"CPH1 12.011000 " \
"CPH2 12.011000 " \
"HR3   1.008000 " \
"HR1   1.008000 " \
"NR1  14.007000 " \
"NR2  14.007000 " \
"H      1.00800 " \
"HC     1.00800 " \
"HA     1.00800 " \
"HT     1.00800 " \
"HP     1.00800 " \
"HB     1.00800 " \
"HR1    1.00800 " \
"HR2    1.00800 " \
"HR3    1.00800 " \
"HS     1.00800 " \
"HE1    1.00800 " \
"HE2    1.00800 " \
"HA1    1.00800 " \
"HA2    1.00800 " \
"HA3    1.00800 " \
"HF1    1.00800 " \
"HF2    1.00800 " \
"C     12.01100 " \
"CA    12.01100 " \
"CT1   12.01100 " \
"CT2   12.01100 " \
"CT3   12.01100 " \
"CPH1  12.01100 " \
"CPH2  12.01100 " \
"CPT   12.01100 " \
"CY    12.01100 " \
"CP1   12.01100 " \
"CP2   12.01100 " \
"CP3   12.01100 " \
"CC    12.01100 " \
"CD    12.01100 " \
"CPA   12.01100 " \
"CPB   12.01100 " \
"CPM   12.01100 " \
"CM    12.01100 " \
"CS    12.01100 " \
"CE1   12.01100 " \
"CE2   12.01100 " \
"CST   12.01100 " \
"CT    12.01100 " \
"CT1x  12.01100 " \
"CT2x  12.01100 " \
"CT3x  12.01100 " \
"CN    12.01100 " \
"CAP   12.01100 " \
"COA   12.01100 " \
"C3    12.01100 " \
"N     14.00700 " \
"NR1   14.00700 " \
"NR2   14.00700 " \
"NR3   14.00700 " \
"NH1   14.00700 " \
"NH2   14.00700 " \
"NH3   14.00700 " \
"NC2   14.00700 " \
"NY    14.00700 " \
"NP    14.00700 " \
"NPH   14.00700 " \
"NC    14.00700 " \
"O     15.99900 " \
"OB    15.99900 " \
"OC    15.99900 " \
"OH1   15.99900 " \
"OS    15.99940 " \
"OT    15.99940 " \
"OM    15.99900 " \
"OST   15.99900 " \
"OCA   15.99900 " \
"S     32.06000 " \
"SM    32.06000 " \
"SS    32.06000 " \
"HE     4.00260 " \
"NE    20.17970 " \
"CF1   12.01100 " \
"CF2   12.01100 " \
"CF3   12.01100 " \
"FE    55.84700 " \
"CLAL  35.45300 " \
"FA    18.99800 " \
"F1    18.99800 " \
"F2    18.99800 " \
"F3    18.99800 " \
"DUM    0.00000 " \
"HL    1.008000 " \
"HCL   1.008000 " \
"HT    1.008000 " \
"HOL   1.008000 " \
"HAL1  1.008000 " \
"HAL2  1.008000 " \
"HAL3  1.008000 " \
"HEL1  1.008000 " \
"HEL2  1.008000 " \
"CL   12.011000 " \
"CTL1 12.011000 " \
"CTL2 12.011000 " \
"CTL3 12.011000 " \
"CTL5 12.011000 " \
"CEL1 12.011000 " \
"CEL2 12.011000 " \
"NTL  14.007000 " \
"NH3L 14.007000 " \
"OBL  15.999400 " \
"OCL  15.999400 " \
"OT   15.999400 " \
"OSL  15.999400 " \
"O2L  15.999400 " \
"OHL  15.999400 " \
"PL   30.974000 " \
"SL   32.060000 " \
"SOD  22.989770 " \
"MG   24.305000 " \
"POT  39.102000 " \
"CES 132.900000 " \
"CAL  40.080000 " \
"CLA  35.450000 " \
"ZN   65.370000 " \
"HN1   1.008000 " \
"HN2   1.008000 " \
"HN3   1.008000 " \
"HN3B  1.008000 " \
"HN3C  1.008000 " \
"HNP   1.008000 " \
"HN4   1.008000 " \
"HN5   1.008000 " \
"HN6   1.008000 " \
"HN7   1.008000 " \
"HN8   1.008000 " \
"HN9   1.008000 " \
"HNE1  1.008000 " \
"HNE2  1.008000 " \
"CN1  12.011000 " \
"CN1A 12.011000 " \
"CN1T 12.011000 " \
"CN2  12.011000 " \
"CN3  12.011000 " \
"CN3A 12.011000 " \
"CN3B 12.011000 " \
"CN3C 12.011000 " \
"CN3D 12.011000 " \
"CN3T 12.011000 " \
"CN4  12.011000 " \
"CN5  12.011000 " \
"CN5G 12.011000 " \
"CN7  12.011000 " \
"CN7B 12.011000 " \
"CN7C 12.011000 " \
"CN7D 12.011000 " \
"CN8  12.011000 " \
"CN8B 12.011000 " \
"CN9  12.011000 " \
"CNE1 12.011000 " \
"CNE2 12.011000 " \
"CNA  12.011000 " \
"CNA2 12.011000 " \
"CN6  12.011000 " \
"CN7E 12.011000 " \
"NN1  14.007000 " \
"NN1C 14.007000 " \
"NN2  14.007000 " \
"NN2B 14.007000 " \
"NN2C 14.007000 " \
"NN2U 14.007000 " \
"NN2G 14.007000 " \
"NN3  14.007000 " \
"NN3A 14.007000 " \
"NN3I 14.007000 " \
"NN3G 14.007000 " \
"NN4  14.007000 " \
"NN5  14.007000 " \
"NN6  14.007000 " \
"ON1  15.999400 " \
"ON1C 15.999400 " \
"ON2  15.999400 " \
"ON3  15.999400 " \
"ON4  15.999400 " \
"ON5  15.999400 " \
"ON6  15.999400 " \
"ON6B 15.999400 " \
"ON2B 15.999400 " \
"FN1  18.998400 " \
"FNA  18.998400 " \
"P    30.974000 " \
"P2   30.974000 " \
"P3   30.974000 " \
"SOD  22.989770 " \
"MG   24.305000 " \
"POT  39.102000 " \
"CES 132.900000 " \
"CAL  40.080000 " \
"CLA  35.450000 " \
"ZN   65.370000 " \
"DUM   0.001    " \
"H      1.00800 " \
"HC     1.00800 " \
"HA     1.00800 " \
"HT     1.00800 " \
"HP     1.00800 " \
"HR1    1.00800 " \
"HR2    1.00800 " \
"HR3    1.00800 " \
"HS     1.00800 " \
"HE1    1.00800 " \
"HE2    1.00800 " \
"HA1    1.00800 " \
"HA2    1.00800 " \
"HA3    1.00800 " \
"HB1    1.00800 " \
"HB2    1.00800 " \
"C     12.01100 " \
"CA    12.01100 " \
"CT1   12.01100 " \
"CT2   12.01100 " \
"CT3   12.01100 " \
"CPH1  12.01100 " \
"CPH2  12.01100 " \
"CPT   12.01100 " \
"CY    12.01100 " \
"CP1   12.01100 " \
"CP2   12.01100 " \
"CP3   12.01100 " \
"CC    12.01100 " \
"CD    12.01100 " \
"CPA   12.01100 " \
"CPB   12.01100 " \
"CPM   12.01100 " \
"CM    12.01100 " \
"CS    12.01100 " \
"CE1   12.01100 " \
"CE2   12.01100 " \
"N     14.00700 " \
"NR1   14.00700 " \
"NR2   14.00700 " \
"NR3   14.00700 " \
"NH1   14.00700 " \
"NH2   14.00700 " \
"NH3   14.00700 " \
"NC2   14.00700 " \
"NY    14.00700 " \
"NP    14.00700 " \
"NPH   14.00700 " \
"O     15.99900 " \
"OB    15.99900 " \
"OC    15.99900 " \
"OH1   15.99900 " \
"OS    15.99940 " \
"OT    15.99940 " \
"OM    15.99900 " \
"S     32.06000 " \
"SM    32.06000 " \
"SS    32.06000 " \
"HE     4.00260 " \
"NE    20.17970 " \
"LP    0.000000 " \
"CAL   40.08000 " \
"ZN    65.37000 " \
"FE    55.84700 " \
"DUM    0.00000 " \
"H      1.00800 " \
"HC     1.00800 " \
"HA     1.00800 " \
"HT     1.00800 " \
"LP     0.0     " \
"CT    12.01100 " \
"C     12.01100 " \
"CH1E  13.01900 " \
"CH2E  14.02700 " \
"CH3E  15.03500 " \
"CR1E  13.01900 " \
"CM    12.01100 " \
"N     14.00670 " \
"NR    14.00670 " \
"NP    14.00670 " \
"NH1E  15.01470 " \
"NH2E  16.02270 " \
"NH3E  17.03070 " \
"NC2E  16.02270 " \
"NH1   14.00670 " \
"NH2   14.00670 " \
"NH3   14.00670 " \
"NC2   14.00670 " \
"O     15.99940 " \
"OC    15.99940 " \
"OH1E  17.00740 " \
"OH2E  18.01540 " \
"OH1   15.99940 " \
"OH2   15.99940 " \
"OM    15.99940 " \
"OT    15.99940 " \
"OS    15.99940 " \
"S     32.06000 " \
"SH1E  33.06800 " \
"FE    55.84700 " \
"H      1.00800 " \
"H2     1.00800 " \
"HO     1.00800 " \
"HT     1.00800 " \
"LP     0.0     " \
"C     12.01100 " \
"CH    13.01900 " \
"C2    14.02700 " \
"CA    12.01100 " \
"CB    12.01100 " \
"CE    13.01900 " \
"CF    13.01900 " \
"C3    15.03500 " \
"CS    12.01100 " \
"N2    14.00670 " \
"NA    14.00670 " \
"NB    14.00670 " \
"NC    14.00670 " \
"NS    14.00670 " \
"NH2E  16.02270 " \
"NH3   14.00670 " \
"O     15.99940 " \
"O2    15.99940 " \
"OS    15.99940 " \
"OH    15.99940 " \
"OH2   15.99940 " \
"OT    15.99940 " \
"OSS   15.99940 " \
"OST   15.99940 " \
"SD    22.98980 " \
"P     30.97400 " \
"BR    79.90400 " \
"MG    24.30500 " 
} {
 set data [ split $data " " ]
 set i 0
 foreach value $data {
   if { $value > " " } { incr i; set val($i) $value }
   if { $i == 2 } {
     set mass($val(1)) $val(2)
   }
 }
}

#
# Charm atom classes
#

foreach data { \
"S    S     " \
"O1   OC    " \
"O2   OC    " \
"O3   OC    " \
"O4   OC    " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT3   " \
"HB1  HA    " \
"HB2  HA    " \
"HB3  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CT2   " \
"HD1  HA    " \
"HD2  HA    " \
"NE   NC2   " \
"HE   HC    " \
"CZ   C     " \
"NH1  NC2   " \
"HH11 HC    " \
"HH12 HC    " \
"NH2  NC2   " \
"HH21 HC    " \
"HH22 HC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CC    " \
"OD1  O     " \
"ND2  NH2   " \
"HD21 H     " \
"HD22 H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CC    " \
"OD1  OC    " \
"OD2  OC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"SG   S     " \
"HG1  HS    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CC    " \
"OE1  O     " \
"NE2  NH2   " \
"HE21 H     " \
"HE22 H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CC    " \
"OE1  OC    " \
"OE2  OC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT2   " \
"HA1  HB    " \
"HA2  HB    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"ND1  NR1   " \
"HD1  H     " \
"CG   CPH1  " \
"CE1  CPH2  " \
"HE1  HR1   " \
"NE2  NR2   " \
"CD2  CPH1  " \
"HD2  HR3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"ND1  NR2   " \
"CG   CPH1  " \
"CE1  CPH2  " \
"HE1  HR1   " \
"NE2  NR1   " \
"HE2  H     " \
"CD2  CPH1  " \
"HD2  HR3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CD2  CPH1  " \
"HD2  HR1   " \
"CG   CPH1  " \
"NE2  NR3   " \
"HE2  H     " \
"ND1  NR3   " \
"HD1  H     " \
"CE1  CPH2  " \
"HE1  HR2   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT1   " \
"HB   HA    " \
"CG2  CT3   " \
"HG21 HA    " \
"HG22 HA    " \
"HG23 HA    " \
"CG1  CT2   " \
"HG11 HA    " \
"HG12 HA    " \
"CD   CT3   " \
"HD1  HA    " \
"HD2  HA    " \
"HD3  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT1   " \
"HG   HA    " \
"CD1  CT3   " \
"HD11 HA    " \
"HD12 HA    " \
"HD13 HA    " \
"CD2  CT3   " \
"HD21 HA    " \
"HD22 HA    " \
"HD23 HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CT2   " \
"HD1  HA    " \
"HD2  HA    " \
"CE   CT2   " \
"HE1  HA    " \
"HE2  HA    " \
"NZ   NH3   " \
"HZ1  HC    " \
"HZ2  HC    " \
"HZ3  HC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"SD   S     " \
"CE   CT3   " \
"HE1  HA    " \
"HE2  HA    " \
"HE3  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CA    " \
"CD1  CA    " \
"HD1  HP    " \
"CE1  CA    " \
"HE1  HP    " \
"CZ   CA    " \
"HZ   HP    " \
"CD2  CA    " \
"HD2  HP    " \
"CE2  CA    " \
"HE2  HP    " \
"C    C     " \
"O    O     " \
"N    N     " \
"CD   CP3   " \
"HD1  HA    " \
"HD2  HA    " \
"CA   CP1   " \
"HA   HB    " \
"CB   CP2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CP2   " \
"HG1  HA    " \
"HG2  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"OG   OH1   " \
"HG1  H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT1   " \
"HB   HA    " \
"OG1  OH1   " \
"HG1  H     " \
"CG2  CT3   " \
"HG21 HA    " \
"HG22 HA    " \
"HG23 HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CY    " \
"CD1  CA    " \
"HD1  HP    " \
"NE1  NY    " \
"HE1  H     " \
"CE2  CPT   " \
"CD2  CPT   " \
"CE3  CA    " \
"HE3  HP    " \
"CZ3  CA    " \
"HZ3  HP    " \
"CZ2  CA    " \
"HZ2  HP    " \
"CH2  CA    " \
"HH2  HP    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CA    " \
"CD1  CA    " \
"HD1  HP    " \
"CE1  CA    " \
"HE1  HP    " \
"CZ   CA    " \
"OH   OH1   " \
"HH   H     " \
"CD2  CA    " \
"HD2  HP    " \
"CE2  CA    " \
"HE2  HP    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT1   " \
"HB   HA    " \
"CG1  CT3   " \
"HG11 HA    " \
"HG12 HA    " \
"HG13 HA    " \
"CG2  CT3   " \
"HG21 HA    " \
"HG22 HA    " \
"HG23 HA    " \
"C    C     " \
"O    O     " \
"CL   CT3   " \
"HL1  HA    " \
"HL2  HA    " \
"HL3  HA    " \
"CLP  C     " \
"OL   O     " \
"NL   NH1   " \
"HL   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT3   " \
"HB1  HA    " \
"HB2  HA    " \
"HB3  HA    " \
"CRP  C     " \
"OR   O     " \
"NR   NH1   " \
"HR   H     " \
"CR   CT3   " \
"HR1  HA    " \
"HR2  HA    " \
"HR3  HA    " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"SOD  SOD   " \
"MG   MG    " \
"POT   POT  " \
"CES  CES   " \
"CAL  CAL   " \
"CLA  CLA   " \
"ZN   ZN    " \
"N    NH3   " \
"HT1  HC    " \
"HT2  HC    " \
"HT3  HC    " \
"CA   CT1   " \
"HA   HB    " \
"N    NH3   " \
"HT1  HC    " \
"HT2  HC    " \
"HT3  HC    " \
"CA   CT2   " \
"HA1  HB    " \
"HA2  HB    " \
"N    NP    " \
"HN1  HC    " \
"HN2  HC    " \
"CD   CP3   " \
"HD1  HA    " \
"HD2  HA    " \
"CA   CP1   " \
"HA   HB    " \
"CAY  CT3   " \
"HY1  HA    " \
"HY2  HA    " \
"HY3  HA    " \
"CY   C     " \
"OY   O     " \
"CAY  CT3   " \
"HY1  HA    " \
"HY2  HA    " \
"HY3  HA    " \
"CY   C     " \
"OY   O     " \
"CAY  CT3   " \
"HY1  HA    " \
"HY2  HA    " \
"HY3  HA    " \
"CY   C     " \
"OY   O     " \
"CAY  CT3   " \
"HY1  HA    " \
"HY2  HA    " \
"HY3  HA    " \
"CY   C     " \
"OY   O     " \
"C    CC    " \
"OT1  OC    " \
"OT2  OC    " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"C    CD    " \
"OT1  OB    " \
"OT2  OS    " \
"CT   CT3   " \
"HT1  HA    " \
"HT2  HA    " \
"HT3  HA    " \
"C    CC    " \
"O    O     " \
"NT   NH2   " \
"HT1  H     " \
"HT2  H     " \
"C    C     " \
"O    O     " \
"NT   NH1   " \
"HNT  H     " \
"CAT  CT3   " \
"HT1  HA    " \
"HT2  HA    " \
"HT3  HA    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CD    " \
"OD1  OB    " \
"OD2  OH1   " \
"HD2  H     " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CD    " \
"OE1  OB    " \
"OE2  OH1   " \
"HE2  H     " \
"CE   CT2   " \
"HE1  HA    " \
"HE2  HA    " \
"NZ   NH2   " \
"HZ1  HC    " \
"HZ2  HC    " \
"1CB  CT2   " \
"1SG  SM    " \
"2SG  SM    " \
"2CB  CT2   " \
"CE1  CPH2  " \
"HE1  HR1   " \
"ND1  NR2   " \
"CG   CPH1  " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"NE2  NR1   " \
"HE2  H     " \
"CD2  CPH1  " \
"HD2  HR3   " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT3   " \
"HB1  HA    " \
"HB2  HA    " \
"HB3  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CT2   " \
"HD1  HA    " \
"HD2  HA    " \
"NE   NC2   " \
"HE   HC    " \
"CZ   C     " \
"NH1  NC2   " \
"HH11 HC    " \
"HH12 HC    " \
"NH2  NC2   " \
"HH21 HC    " \
"HH22 HC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CC    " \
"OD1  O     " \
"ND2  NH2   " \
"HD21 H     " \
"HD22 H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CC    " \
"OD1  OC    " \
"OD2  OC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"SG   S     " \
"HG1  HS    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CC    " \
"OE1  O     " \
"NE2  NH2   " \
"HE21 H     " \
"HE22 H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CC    " \
"OE1  OC    " \
"OE2  OC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT2   " \
"HA1  HB    " \
"HA2  HB    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"ND1  NR1   " \
"HD1  H     " \
"CG   CPH1  " \
"CE1  CPH2  " \
"HE1  HR1   " \
"NE2  NR2   " \
"CD2  CPH1  " \
"HD2  HR3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"ND1  NR2   " \
"CG   CPH1  " \
"CE1  CPH2  " \
"HE1  HR1   " \
"NE2  NR1   " \
"HE2  H     " \
"CD2  CPH1  " \
"HD2  HR3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CD2  CPH1  " \
"HD2  HR1   " \
"CG   CPH1  " \
"NE2  NR3   " \
"HE2  H     " \
"ND1  NR3   " \
"HD1  H     " \
"CE1  CPH2  " \
"HE1  HR2   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT1   " \
"HB   HA    " \
"CG2  CT3   " \
"HG21 HA    " \
"HG22 HA    " \
"HG23 HA    " \
"CG1  CT2   " \
"HG11 HA    " \
"HG12 HA    " \
"CD   CT3   " \
"HD1  HA    " \
"HD2  HA    " \
"HD3  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT1   " \
"HG   HA    " \
"CD1  CT3   " \
"HD11 HA    " \
"HD12 HA    " \
"HD13 HA    " \
"CD2  CT3   " \
"HD21 HA    " \
"HD22 HA    " \
"HD23 HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CT2   " \
"HD1  HA    " \
"HD2  HA    " \
"CE   CT2   " \
"HE1  HA    " \
"HE2  HA    " \
"NZ   NH3   " \
"HZ1  HC    " \
"HZ2  HC    " \
"HZ3  HC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"SD   S     " \
"CE   CT3   " \
"HE1  HA    " \
"HE2  HA    " \
"HE3  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CA    " \
"CD1  CA    " \
"HD1  HP    " \
"CE1  CA    " \
"HE1  HP    " \
"CZ   CA    " \
"HZ   HP    " \
"CD2  CA    " \
"HD2  HP    " \
"CE2  CA    " \
"HE2  HP    " \
"C    C     " \
"O    O     " \
"N    N     " \
"CD   CP3   " \
"HD1  HA    " \
"HD2  HA    " \
"CA   CP1   " \
"HA   HB    " \
"CB   CP2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CP2   " \
"HG1  HA    " \
"HG2  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"OG   OH1   " \
"HG1  H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT1   " \
"HB   HA    " \
"OG1  OH1   " \
"HG1  H     " \
"CG2  CT3   " \
"HG21 HA    " \
"HG22 HA    " \
"HG23 HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CY    " \
"CD1  CA    " \
"HD1  HP    " \
"NE1  NY    " \
"HE1  H     " \
"CE2  CPT   " \
"CD2  CPT   " \
"CE3  CA    " \
"HE3  HP    " \
"CZ3  CA    " \
"HZ3  HP    " \
"CZ2  CA    " \
"HZ2  HP    " \
"CH2  CA    " \
"HH2  HP    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CA    " \
"CD1  CA    " \
"HD1  HP    " \
"CE1  CA    " \
"HE1  HP    " \
"CZ   CA    " \
"OH   OH1   " \
"HH   H     " \
"CD2  CA    " \
"HD2  HP    " \
"CE2  CA    " \
"HE2  HP    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT1   " \
"HB   HA    " \
"CG1  CT3   " \
"HG11 HA    " \
"HG12 HA    " \
"HG13 HA    " \
"CG2  CT3   " \
"HG21 HA    " \
"HG22 HA    " \
"HG23 HA    " \
"C    C     " \
"O    O     " \
"CL   CT3   " \
"HL1  HA    " \
"HL2  HA    " \
"HL3  HA    " \
"CLP  C     " \
"OL   O     " \
"NL   NH1   " \
"HL   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT3   " \
"HB1  HA    " \
"HB2  HA    " \
"HB3  HA    " \
"CRP  C     " \
"OR   O     " \
"NR   NH1   " \
"HR   H     " \
"CR   CT3   " \
"HR1  HA    " \
"HR2  HA    " \
"HR3  HA    " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"SOD  SOD   " \
"MG   MG    " \
"POT   POT  " \
"CES  CES   " \
"CAL  CAL   " \
"CLA  CLA   " \
"ZN   ZN    " \
"N    NH3   " \
"HT1  HC    " \
"HT2  HC    " \
"HT3  HC    " \
"CA   CT1   " \
"HA   HB    " \
"N    NH3   " \
"HT1  HC    " \
"HT2  HC    " \
"HT3  HC    " \
"CA   CT2   " \
"HA1  HB    " \
"HA2  HB    " \
"N    NP    " \
"HN1  HC    " \
"HN2  HC    " \
"CD   CP3   " \
"HD1  HA    " \
"HD2  HA    " \
"CA   CP1   " \
"HA   HB    " \
"CAY  CT3   " \
"HY1  HA    " \
"HY2  HA    " \
"HY3  HA    " \
"CY   C     " \
"OY   O     " \
"CAY  CT3   " \
"HY1  HA    " \
"HY2  HA    " \
"HY3  HA    " \
"CY   C     " \
"OY   O     " \
"C    CC    " \
"OT1  OC    " \
"OT2  OC    " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"C    CD    " \
"OT1  OB    " \
"OT2  OS    " \
"CT   CT3   " \
"HT1  HA    " \
"HT2  HA    " \
"HT3  HA    " \
"C    CC    " \
"O    O     " \
"NT   NH2   " \
"HT1  H     " \
"HT2  H     " \
"C    C     " \
"O    O     " \
"NT   NH1   " \
"HNT  H     " \
"CAT  CT3   " \
"HT1  HA    " \
"HT2  HA    " \
"HT3  HA    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CD    " \
"OD1  OB    " \
"OD2  OH1   " \
"HD2  H     " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CD    " \
"OE1  OB    " \
"OE2  OH1   " \
"HE2  H     " \
"CE   CT2   " \
"HE1  HA    " \
"HE2  HA    " \
"NZ   NH2   " \
"HZ1  HC    " \
"HZ2  HC    " \
"1CB  CT2   " \
"1SG  SM    " \
"2SG  SM    " \
"2CB  CT2   " \
"CE1  CPH2  " \
"HE1  HR1   " \
"ND1  NR2   " \
"CG   CPH1  " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"NE2  NR1   " \
"HE2  H     " \
"CD2  CPH1  " \
"HD2  HR3   " \
"C1   CTS   " \
"H1   HAS   " \
"O1   OHS   " \
"HO1  HOS   " \
"C5   CTS   " \
"H5   HAS   " \
"O5   OES   " \
"C2   CTS   " \
"H2   HAS   " \
"O2   OHS   " \
"HO2  HOS   " \
"C3   CTS   " \
"H3   HAS   " \
"O3   OHS   " \
"HO3  HOS   " \
"C4   CTS   " \
"H4   HAS   " \
"O4   OHS   " \
"HO4  HOS   " \
"C6   CTS   " \
"H61  HAS   " \
"H62  HAS   " \
"O6   OHS   " \
"HO6  HOS   " \
"C1   CBS   " \
"N    NTL   " \
"C13  CTL5  " \
"H13A HL    " \
"H13B HL    " \
"H13C HL    " \
"C14  CTL5  " \
"H14A HL    " \
"H14B HL    " \
"H14C HL    " \
"C15  CTL5  " \
"H15A HL    " \
"H15B HL    " \
"H15C HL    " \
"C12  CTL2  " \
"H12A HL    " \
"H12B HL    " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"P    PL    " \
"O13  O2L   " \
"O14  O2L   " \
"O11  OSL   " \
"O12  OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2   CTL2  " \
"HS   HAL2  " \
"HR   HAL2  " \
"C3   CTL2  " \
"HX   HAL2  " \
"HY   HAL2  " \
"O31  OSL   " \
"C31  CL    " \
"O32  OBL   " \
"C32  CTL2  " \
"H2X  HAL2  " \
"H2Y  HAL2  " \
"C33  CTL2  " \
"H3X  HAL2  " \
"H3Y  HAL2  " \
"C34  CTL2  " \
"H4X  HAL2  " \
"H4Y  HAL2  " \
"C35  CTL2  " \
"H5X  HAL2  " \
"H5Y  HAL2  " \
"C36  CTL2  " \
"H6X  HAL2  " \
"H6Y  HAL2  " \
"C37  CTL2  " \
"H7X  HAL2  " \
"H7Y  HAL2  " \
"C38  CTL2  " \
"H8X  HAL2  " \
"H8Y  HAL2  " \
"C39  CTL2  " \
"H9X  HAL2  " \
"H9Y  HAL2  " \
"C310 CTL2  " \
"H10X HAL2  " \
"H10Y HAL2  " \
"C311 CTL2  " \
"H11X HAL2  " \
"H11Y HAL2  " \
"C312 CTL3  " \
"H12X HAL3  " \
"H12Y HAL3  " \
"H12Z HAL3  " \
"N     NH3  " \
"HN1   HCL  " \
"HN2   HCL  " \
"HN3   HCL  " \
"C12   CTL  " \
"H12A  HAL  " \
"H12B  HAL  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"P    PL    " \
"O13  O2L   " \
"O14  O2L   " \
"O11  OSL   " \
"O12  OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2  CTL1   " \
"HS  HAL1   " \
"OH2  OHL   " \
"HO2  HOL   " \
"C3  CTL2   " \
"HX  HAL2   " \
"HY  HAL2   " \
"OH3  OHL   " \
"HO3  HOL   " \
"N     NH3  " \
"HN1   HCL  " \
"HN2   HCL  " \
"HN3   HCL  " \
"C12   CTL  " \
"H12A  HAL  " \
"H12B  HAL  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"P    PL    " \
"O13  O2L   " \
"O14  O2L   " \
"O11  OSL   " \
"O12  OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2   CTL1  " \
"HS   HAL1  " \
"O21  OSL   " \
"C21  CL    " \
"O22  OBL   " \
"C22  CTL2  " \
"H2R  HAL2  " \
"H2S  HAL2  " \
"C3   CTL2  " \
"HX   HAL2  " \
"HY   HAL2  " \
"O31  OSL   " \
"C31  CL    " \
"O32  OBL   " \
"C32  CTL2  " \
"H2X  HAL2  " \
"H2Y  HAL2  " \
"C23  CTL2  " \
"H3R  HAL2  " \
"H3S  HAL2  " \
"C24  CTL2  " \
"H4R  HAL2  " \
"H4S  HAL2  " \
"C25  CTL2  " \
"H5R  HAL2  " \
"H5S  HAL2  " \
"C26  CTL2  " \
"H6R  HAL2  " \
"H6S  HAL2  " \
"C27  CTL2  " \
"H7R  HAL2  " \
"H7S  HAL2  " \
"C28  CTL2  " \
"H8R  HAL2  " \
"H8S  HAL2  " \
"C29  CTL2  " \
"H9R  HAL2  " \
"H9S  HAL2  " \
"C210 CTL2  " \
"H10R HAL2  " \
"H10S HAL2  " \
"C211 CTL2  " \
"H11R HAL2  " \
"H11S HAL2  " \
"C212 CTL3  " \
"H12R HAL3  " \
"H12S HAL3  " \
"H12T HAL3  " \
"C33  CTL2  " \
"H3X  HAL2  " \
"H3Y  HAL2  " \
"C34  CTL2  " \
"H4X  HAL2  " \
"H4Y  HAL2  " \
"C35  CTL2  " \
"H5X  HAL2  " \
"H5Y  HAL2  " \
"C36  CTL2  " \
"H6X  HAL2  " \
"H6Y  HAL2  " \
"C37  CTL2  " \
"H7X  HAL2  " \
"H7Y  HAL2  " \
"C38  CTL2  " \
"H8X  HAL2  " \
"H8Y  HAL2  " \
"C39  CTL2  " \
"H9X  HAL2  " \
"H9Y  HAL2  " \
"C310 CTL2  " \
"H10X HAL2  " \
"H10Y HAL2  " \
"C311 CTL2  " \
"H11X HAL2  " \
"H11Y HAL2  " \
"C312 CTL3  " \
"H12X HAL3  " \
"H12Y HAL3  " \
"H12Z HAL3  " \
"N    NTL   " \
"C13  CTL5  " \
"H13A HL    " \
"H13B HL    " \
"H13C HL    " \
"C14  CTL5  " \
"H14A HL    " \
"H14B HL    " \
"H14C HL    " \
"C15  CTL5  " \
"H15A HL    " \
"H15B HL    " \
"H15C HL    " \
"C12  CTL2  " \
"H12A HL    " \
"H12B HL    " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"P    PL    " \
"O13  O2L   " \
"O14  O2L   " \
"O11  OSL   " \
"O12  OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2   CTL1  " \
"HS   HAL1  " \
"O21  OSL   " \
"C21  CL    " \
"O22  OBL   " \
"C22  CTL2  " \
"H2R  HAL2  " \
"H2S  HAL2  " \
"C3   CTL2  " \
"HX   HAL2  " \
"HY   HAL2  " \
"O31  OSL   " \
"C31  CL    " \
"O32  OBL   " \
"C32  CTL2  " \
"H2X  HAL2  " \
"H2Y  HAL2  " \
"C23  CTL2  " \
"H3R  HAL2  " \
"H3S  HAL2  " \
"C24  CTL2  " \
"H4R  HAL2  " \
"H4S  HAL2  " \
"C25  CTL2  " \
"H5R  HAL2  " \
"H5S  HAL2  " \
"C26  CTL2  " \
"H6R  HAL2  " \
"H6S  HAL2  " \
"C27  CTL2  " \
"H7R  HAL2  " \
"H7S  HAL2  " \
"C28  CTL2  " \
"H8R  HAL2  " \
"H8S  HAL2  " \
"C29  CTL2  " \
"H9R  HAL2  " \
"H9S  HAL2  " \
"C210 CTL2  " \
"H10R HAL2  " \
"H10S HAL2  " \
"C211 CTL2  " \
"H11R HAL2  " \
"H11S HAL2  " \
"C212 CTL2  " \
"H12R HAL2  " \
"H12S HAL2  " \
"C213 CTL2  " \
"H13R HAL2  " \
"H13S HAL2  " \
"C214 CTL3  " \
"H14R HAL3  " \
"H14S HAL3  " \
"H14T HAL3  " \
"C33  CTL2  " \
"H3X  HAL2  " \
"H3Y  HAL2  " \
"C34  CTL2  " \
"H4X  HAL2  " \
"H4Y  HAL2  " \
"C35  CTL2  " \
"H5X  HAL2  " \
"H5Y  HAL2  " \
"C36  CTL2  " \
"H6X  HAL2  " \
"H6Y  HAL2  " \
"C37  CTL2  " \
"H7X  HAL2  " \
"H7Y  HAL2  " \
"C38  CTL2  " \
"H8X  HAL2  " \
"H8Y  HAL2  " \
"C39  CTL2  " \
"H9X  HAL2  " \
"H9Y  HAL2  " \
"C310 CTL2  " \
"H10X HAL2  " \
"H10Y HAL2  " \
"C311 CTL2  " \
"H11X HAL2  " \
"H11Y HAL2  " \
"C312 CTL2  " \
"H12X HAL2  " \
"H12Y HAL2  " \
"C313 CTL2  " \
"H13X HAL2  " \
"H13Y HAL2  " \
"C314 CTL3  " \
"H14X HAL3  " \
"H14Y HAL3  " \
"H14Z HAL3  " \
"S    SL    " \
"OS1  OSL   " \
"OS2  O2L   " \
"OS3  O2L   " \
"OS4  O2L   " \
"C1   CTL2  " \
"H11  HAL2  " \
"H12  HAL2  " \
"C2   CTL2  " \
"H21  HAL2  " \
"H22  HAL2  " \
"C3   CTL2  " \
"H31  HAL2  " \
"H32  HAL2  " \
"C4   CTL2  " \
"H41  HAL2  " \
"H42  HAL2  " \
"C5   CTL2  " \
"H51  HAL2  " \
"H52  HAL2  " \
"C6   CTL2  " \
"H61  HAL2  " \
"H62  HAL2  " \
"C7   CTL2  " \
"H71  HAL2  " \
"H72  HAL2  " \
"C8   CTL2  " \
"H81  HAL2  " \
"H82  HAL2  " \
"C9   CTL2  " \
"H91  HAL2  " \
"H92  HAL2  " \
"C10  CTL2  " \
"H101 HAL2  " \
"H102 HAL2  " \
"C11  CTL2  " \
"H111 HAL2  " \
"H112 HAL2  " \
"C12  CTL3  " \
"H121 HAL3  " \
"H122 HAL3  " \
"H123 HAL3  " \
"N    NTL   " \
"C11  CTL2  " \
"C12  CTL5  " \
"C13  CTL5  " \
"C14  CTL5  " \
"H11  HL    " \
"H12  HL    " \
"H21  HL    " \
"H22  HL    " \
"H23  HL    " \
"H31  HL    " \
"H32  HL    " \
"H33  HL    " \
"H41  HL    " \
"H42  HL    " \
"H43  HL    " \
"C15  CTL2  " \
"H51  HAL2  " \
"H52  HAL2  " \
"P1   PL    " \
"O3   O2L   " \
"O4   O2L   " \
"O1   OSL   " \
"O2   OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2   CTL1  " \
"HS   HAL1  " \
"O21  OSL   " \
"C21  CL    " \
"O22  OBL   " \
"C22  CTL2  " \
"H2R  HAL2  " \
"H2S  HAL2  " \
"C3   CTL2  " \
"HX   HAL2  " \
"HY   HAL2  " \
"O31  OSL   " \
"C31  CL    " \
"O32  OBL   " \
"C32  CTL2  " \
"H2X  HAL2  " \
"H2Y  HAL2  " \
"C23  CTL2  " \
"H3R  HAL2  " \
"H3S  HAL2  " \
"C24  CTL2  " \
"H4R  HAL2  " \
"H4S  HAL2  " \
"C25  CTL2  " \
"H5R  HAL2  " \
"H5S  HAL2  " \
"C26  CTL2  " \
"H6R  HAL2  " \
"H6S  HAL2  " \
"C27  CTL2  " \
"H7R  HAL2  " \
"H7S  HAL2  " \
"C28  CTL2  " \
"H8R  HAL2  " \
"H8S  HAL2  " \
"C29  CEL1  " \
"H91  HEL1  " \
"C210 CEL1  " \
"H101 HEL1  " \
"C211 CTL2  " \
"H11R HAL2  " \
"H11S HAL2  " \
"C212 CTL2  " \
"H12R HAL2  " \
"H12S HAL2  " \
"C213 CTL2  " \
"H13R HAL2  " \
"H13S HAL2  " \
"C214 CTL2  " \
"H14R HAL2  " \
"H14S HAL2  " \
"C215 CTL2  " \
"H15R HAL2  " \
"H15S HAL2  " \
"C216 CTL2  " \
"H16R HAL2  " \
"H16S HAL2  " \
"C217 CTL2  " \
"H17R HAL2  " \
"H17S HAL2  " \
"C218 CTL3  " \
"H18R HAL3  " \
"H18S HAL3  " \
"H18T HAL3  " \
"C33  CTL2  " \
"H3X  HAL2  " \
"H3Y  HAL2  " \
"C34  CTL2  " \
"H4X  HAL2  " \
"H4Y  HAL2  " \
"C35  CTL2  " \
"H5X  HAL2  " \
"H5Y  HAL2  " \
"C36  CTL2  " \
"H6X  HAL2  " \
"H6Y  HAL2  " \
"C37  CTL2  " \
"H7X  HAL2  " \
"H7Y  HAL2  " \
"C38  CTL2  " \
"H8X  HAL2  " \
"H8Y  HAL2  " \
"C39  CTL2  " \
"H9X  HAL2  " \
"H9Y  HAL2  " \
"C310 CTL2  " \
"H10X HAL2  " \
"H10Y HAL2  " \
"C311 CTL2  " \
"H11X HAL2  " \
"H11Y HAL2  " \
"C312 CTL2  " \
"H12X HAL2  " \
"H12Y HAL2  " \
"C313 CTL2  " \
"H13X HAL2  " \
"H13Y HAL2  " \
"C314 CTL2  " \
"H14X HAL2  " \
"H14Y HAL2  " \
"C315 CTL2  " \
"H15X HAL2  " \
"H15Y HAL2  " \
"C316 CTL3  " \
"H16X HAL3  " \
"H16Y HAL3  " \
"H16Z HAL3  " \
"N     NH3  " \
"HN1   HCL  " \
"HN2   HCL  " \
"HN3   HCL  " \
"C12   CTL  " \
"H12A  HAL  " \
"H12B  HAL  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"P    PL    " \
"O13  O2L   " \
"O14  O2L   " \
"O11  OSL   " \
"O12  OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2   CTL1  " \
"HS   HAL1  " \
"O21  OSL   " \
"C21  CL    " \
"O22  OBL   " \
"C22  CTL2  " \
"H2R  HAL2  " \
"H2S  HAL2  " \
"C3   CTL2  " \
"HX   HAL2  " \
"HY   HAL2  " \
"O31  OSL   " \
"C31  CL    " \
"O32  OBL   " \
"C32  CTL2  " \
"H2X  HAL2  " \
"H2Y  HAL2  " \
"C23  CTL2  " \
"H3R  HAL2  " \
"H3S  HAL2  " \
"C24  CTL2  " \
"H4R  HAL2  " \
"H4S  HAL2  " \
"C25  CTL2  " \
"H5R  HAL2  " \
"H5S  HAL2  " \
"C26  CTL2  " \
"H6R  HAL2  " \
"H6S  HAL2  " \
"C27  CTL2  " \
"H7R  HAL2  " \
"H7S  HAL2  " \
"C28  CTL2  " \
"H8R  HAL2  " \
"H8S  HAL2  " \
"C29  CEL1  " \
"H91  HEL1  " \
"C210 CEL1  " \
"H101 HEL1  " \
"C211 CTL2  " \
"H11R HAL2  " \
"H11S HAL2  " \
"C212 CTL2  " \
"H12R HAL2  " \
"H12S HAL2  " \
"C213 CTL2  " \
"H13R HAL2  " \
"H13S HAL2  " \
"C214 CTL2  " \
"H14R HAL2  " \
"H14S HAL2  " \
"C215 CTL2  " \
"H15R HAL2  " \
"H15S HAL2  " \
"C216 CTL2  " \
"H16R HAL2  " \
"H16S HAL2  " \
"C217 CTL2  " \
"H17R HAL2  " \
"H17S HAL2  " \
"C218 CTL3  " \
"H18R HAL3  " \
"H18S HAL3  " \
"H18T HAL3  " \
"C33  CTL2  " \
"H3X  HAL2  " \
"H3Y  HAL2  " \
"C34  CTL2  " \
"H4X  HAL2  " \
"H4Y  HAL2  " \
"C35  CTL2  " \
"H5X  HAL2  " \
"H5Y  HAL2  " \
"C36  CTL2  " \
"H6X  HAL2  " \
"H6Y  HAL2  " \
"C37  CTL2  " \
"H7X  HAL2  " \
"H7Y  HAL2  " \
"C38  CTL2  " \
"H8X  HAL2  " \
"H8Y  HAL2  " \
"C39  CTL2  " \
"H9X  HAL2  " \
"H9Y  HAL2  " \
"C310 CTL2  " \
"H10X HAL2  " \
"H10Y HAL2  " \
"C311 CTL2  " \
"H11X HAL2  " \
"H11Y HAL2  " \
"C312 CTL2  " \
"H12X HAL2  " \
"H12Y HAL2  " \
"C313 CTL2  " \
"H13X HAL2  " \
"H13Y HAL2  " \
"C314 CTL2  " \
"H14X HAL2  " \
"H14Y HAL2  " \
"C315 CTL2  " \
"H15X HAL2  " \
"H15Y HAL2  " \
"C316 CTL3  " \
"H16X HAL3  " \
"H16Y HAL3  " \
"H16Z HAL3  " \
"C1   CL    " \
"O1   OCL   " \
"O2   OCL   " \
"C2   CTL2  " \
"H2A  HAL2  " \
"H2B  HAL2  " \
"C3   CTL2  " \
"H3A  HAL2  " \
"H3B  HAL2  " \
"C4   CTL2  " \
"H4A  HAL2  " \
"H4B  HAL2  " \
"C5   CTL2  " \
"H5A  HAL2  " \
"H5B  HAL2  " \
"C6   CTL2  " \
"H6A  HAL2  " \
"H6B  HAL2  " \
"C7   CTL2  " \
"H7A  HAL2  " \
"H7B  HAL2  " \
"C8   CTL2  " \
"H8A  HAL2  " \
"H8B  HAL2  " \
"C9   CTL2  " \
"H9A  HAL2  " \
"H9B  HAL2  " \
"C10  CTL2  " \
"H10A HAL2  " \
"H10B HAL2  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"C12  CTL2  " \
"H12A HAL2  " \
"H12B HAL2  " \
"C13  CTL2  " \
"H13A HAL2  " \
"H13B HAL2  " \
"C14  CTL2  " \
"H14A HAL2  " \
"H14B HAL2  " \
"C15  CTL2  " \
"H15A HAL2  " \
"H15B HAL2  " \
"C16  CTL3  " \
"H16A HAL3  " \
"H16B HAL3  " \
"H16C HAL3  " \
"C1   CL    " \
"O1   OCL   " \
"O2   OCL   " \
"C2   CTL2  " \
"H2A  HAL2  " \
"H2B  HAL2  " \
"C3   CTL2  " \
"H3A  HAL2  " \
"H3B  HAL2  " \
"C4   CTL2  " \
"H4A  HAL2  " \
"H4B  HAL2  " \
"C5   CTL2  " \
"H5A  HAL2  " \
"H5B  HAL2  " \
"C6   CTL2  " \
"H6A  HAL2  " \
"H6B  HAL2  " \
"C7   CTL2  " \
"H7A  HAL2  " \
"H7B  HAL2  " \
"C8   CTL2  " \
"H8A  HAL2  " \
"H8B  HAL2  " \
"C9   CEL1  " \
"H9A  HEL1  " \
"C10  CEL1  " \
"H10A HEL1  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"C12  CTL2  " \
"H12A HAL2  " \
"H12B HAL2  " \
"C13  CTL2  " \
"H13A HAL2  " \
"H13B HAL2  " \
"C14  CTL2  " \
"H14A HAL2  " \
"H14B HAL2  " \
"C15  CTL2  " \
"H15A HAL2  " \
"H15B HAL2  " \
"C16  CTL2  " \
"H16A HAL2  " \
"H16B HAL2  " \
"C17  CTL2  " \
"H17A HAL2  " \
"H17B HAL2  " \
"C18  CTL3  " \
"H18A HAL3  " \
"H18B HAL3  " \
"H18C HAL3  " \
"C1   CL    " \
"O1   OCL   " \
"O2   OCL   " \
"C2   CTL2  " \
"H2A  HAL2  " \
"H2B  HAL2  " \
"C3   CTL2  " \
"H3A  HAL2  " \
"H3B  HAL2  " \
"C4   CEL1  " \
"H4   HEL1  " \
"C5   CEL1  " \
"H5   HEL1  " \
"C6   CTL2  " \
"H6A  HAL2  " \
"H6B  HAL2  " \
"C7   CEL1  " \
"H7   HEL1  " \
"C8   CEL1  " \
"H8   HEL1  " \
"C9   CTL2  " \
"H9A  HAL2  " \
"H9B  HAL2  " \
"C10  CEL1  " \
"H10  HEL1  " \
"C11  CEL1  " \
"H11  HEL1  " \
"C12  CTL2  " \
"H12A HAL2  " \
"H12B HAL2  " \
"C13  CEL1  " \
"H13  HEL1  " \
"C14  CEL1  " \
"H14  HEL1  " \
"C15  CTL2  " \
"H15A HAL2  " \
"H15B HAL2  " \
"C16  CEL1  " \
"H16  HEL1  " \
"C17  CEL1  " \
"H17  HEL1  " \
"C18  CTL2  " \
"H18A HAL2  " \
"H18B HAL2  " \
"C19  CEL1  " \
"H19  HEL1  " \
"C20  CEL1  " \
"H20  HEL1  " \
"C21  CTL2  " \
"H21A HAL2  " \
"H21B HAL2  " \
"C22  CTL3  " \
"H22A HAL3  " \
"H22B HAL3  " \
"H22C HAL3  " \
"C1   CL    " \
"O1   OCL   " \
"O2   OCL   " \
"C2   CTL2  " \
"H2A  HAL2  " \
"H2B  HAL2  " \
"C3   CTL2  " \
"H3A  HAL2  " \
"H3B  HAL2  " \
"C4   CTL2  " \
"H4A  HAL2  " \
"H4B  HAL2  " \
"C5   CTL2  " \
"H5A  HAL2  " \
"H5B  HAL2  " \
"C6   CTL2  " \
"H6A  HAL2  " \
"H6B  HAL2  " \
"C7   CTL2  " \
"H7A  HAL2  " \
"H7B  HAL2  " \
"C8   CTL2  " \
"H8A  HAL2  " \
"H8B  HAL2  " \
"C9   CTL2  " \
"H9A  HAL2  " \
"H9B  HAL2  " \
"C10  CTL2  " \
"H10A HAL2  " \
"H10B HAL2  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"C12  CTL2  " \
"H12A HAL2  " \
"H12B HAL2  " \
"C13  CTL2  " \
"H13A HAL2  " \
"H13B HAL2  " \
"C14  CTL2  " \
"H14A HAL2  " \
"H14B HAL2  " \
"C15  CTL2  " \
"H15A HAL2  " \
"H15B HAL2  " \
"C16  CTL2  " \
"H16A HAL2  " \
"H16B HAL2  " \
"C17  CTL2  " \
"H17A HAL2  " \
"H17B HAL2  " \
"C18  CTL3  " \
"H18A HAL3  " \
"H18B HAL3  " \
"H18C HAL3  " \
"N    NTL   " \
"C5   CTL2  " \
"C6   CTL5  " \
"C7   CTL5  " \
"C8   CTL5  " \
"H5A  HL    " \
"H5B  HL    " \
"H6A  HL    " \
"H6B  HL    " \
"H6C  HL    " \
"H7A  HL    " \
"H7B  HL    " \
"H7C  HL    " \
"H8A  HL    " \
"H8B  HL    " \
"H8C  HL    " \
"C4   CTL2  " \
"H4A  HAL2  " \
"H4B  HAL2  " \
"P    PL    " \
"OP3  O2L   " \
"OP4  O2L   " \
"OP1  OSL   " \
"OP2  OSL   " \
"C3   CTL2  " \
"H3A  HAL2  " \
"H3B  HAL2  " \
"C2   CTL1  " \
"H2A  HAL1  " \
"O2   OHL   " \
"H2   HOL   " \
"C1   CTL2  " \
"H1A  HAL2  " \
"H1B  HAL2  " \
"O1   OHL   " \
"H1   HOL   " \
"1C1  CTL2  " \
"1H1A HAL2  " \
"1H1B HAL2  " \
"1O1  OSL   " \
"2C1  CL    " \
"2O1  OBL   " \
"2C2  CTL2  " \
"2H2A HAL2  " \
"2H2B HAL2  " \
"1C2  CTL1  " \
"1H2A HAL1  " \
"1O2  OSL   " \
"2C1  CL    " \
"2O1  OBL   " \
"2C2  CTL2  " \
"2H2A HAL2  " \
"2H2B HAL2  " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"SOD  SOD   " \
"MG   MG    " \
"POT   POT  " \
"CES  CES   " \
"CAL  CAL   " \
"CLA  CLA   " \
"ZN   ZN    " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6B  " \
"C1'  CN7B  " \
"H1'  HN7   " \
"N9   NN2B  " \
"C4   CN5   " \
"N2   NN1   " \
"H21  HN1   " \
"H22  HN1   " \
"N3   NN3G  " \
"C2   CN2   " \
"N1   NN2G  " \
"H1   HN2   " \
"C6   CN1   " \
"O6   ON1   " \
"C5   CN5G  " \
"N7   NN4   " \
"C8   CN4   " \
"H8   HN3   " \
"C2'  CN7B  " \
"H2'' HN7   " \
"O2'  ON5   " \
"H2'  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON2   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6B  " \
"C1'  CN7B  " \
"H1'  HN7   " \
"N9   NN2   " \
"C5   CN5   " \
"N7   NN4   " \
"C8   CN4   " \
"H8   HN3   " \
"N1   NN3A  " \
"C2   CN4   " \
"H2   HN3   " \
"N3   NN3A  " \
"C4   CN5   " \
"C6   CN2   " \
"N6   NN1   " \
"H61  HN1   " \
"H62  HN1   " \
"C2'  CN7B  " \
"H2'' HN7   " \
"O2'  ON5   " \
"H2'  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON2   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6B  " \
"C1'  CN7B  " \
"H1'  HN7   " \
"N1   NN2   " \
"C6   CN3   " \
"H6   HN3   " \
"C5   CN3   " \
"H5   HN3   " \
"C2   CN1   " \
"O2   ON1C  " \
"N3   NN3   " \
"C4   CN2   " \
"N4   NN1   " \
"H41  HN1   " \
"H42  HN1   " \
"C2'  CN7B  " \
"H2'' HN7   " \
"O2'  ON5   " \
"H2'  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON2   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6B  " \
"C1'  CN7B  " \
"H1'  HN7   " \
"N1   NN2B  " \
"C6   CN3   " \
"H6   HN3   " \
"C2   CN1T  " \
"O2   ON1   " \
"N3   NN2U  " \
"H3   HN2   " \
"C4   CN1   " \
"O4   ON1   " \
"C5   CN3T  " \
"C5M  CN9   " \
"H51  HN9   " \
"H52  HN9   " \
"H53  HN9   " \
"C2'  CN7B  " \
"H2'' HN7   " \
"O2'  ON5   " \
"H2'  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON2   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6B  " \
"C1'  CN7B  " \
"H1'  HN7   " \
"N1   NN2B  " \
"C6   CN3   " \
"H6   HN3   " \
"C2   CN1T  " \
"O2   ON1   " \
"N3   NN2U  " \
"H3   HN2   " \
"C4   CN1   " \
"O4   ON1   " \
"C5   CN3   " \
"H5   HN3   " \
"C2'  CN7B  " \
"H2'' HN7   " \
"O2'  ON5   " \
"H2'  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON2   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6   " \
"C1'  CN7B  " \
"H1'  HN7   " \
"C2'  CN8   " \
"H2'  HN8   " \
"H2'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6   " \
"C1'  CN7B  " \
"H1'  HN7   " \
"C2'  CN8   " \
"H2'  HN8   " \
"H2'' HN8   " \
"H5T  HN5   " \
"O5'  ON5   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C5'  CN9   " \
"H5'  HN9   " \
"H5'' HN9   " \
"H53' HN9   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"O5T  ON4   " \
"H5T  HN4   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"O5T  ON2   " \
"C5T  CN9   " \
"H5T1 HN9   " \
"H5T2 HN9   " \
"H5T3 HN9   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON5   " \
"H3T  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"P3   P     " \
"O1P3 ON3   " \
"O2P3 ON3   " \
"O3'  ON2   " \
"O3T  ON4   " \
"H3T  HN4   " \
"C3'  CN7   " \
"H3'  HN7   " \
"P3   P     " \
"O1P3 ON3   " \
"O2P3 ON3   " \
"O3'  ON2   " \
"O3T  ON2   " \
"C3T  CN9   " \
"H3T1 HN9   " \
"H3T2 HN9   " \
"H3T3 HN9   " \
"C3'  CN7   " \
"H3'  HN7   " \
"P3   P     " \
"O3'  ON2   " \
"O1P3 ON3   " \
"O2P3 ON3   " \
"O3P3 ON3   " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"SOD  SOD   " \
"MG   MG    " \
"POT   POT  " \
"CES  CES   " \
"CAL  CAL   " \
"CLA  CLA   " \
"ZN   ZN    " \
"N    NTL   " \
"C13  CTL5  " \
"H13A HL    " \
"H13B HL    " \
"H13C HL    " \
"C14  CTL5  " \
"H14A HL    " \
"H14B HL    " \
"H14C HL    " \
"C15  CTL5  " \
"H15A HL    " \
"H15B HL    " \
"H15C HL    " \
"C12  CTL2  " \
"H12A HL    " \
"H12B HL    " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"P    PL    " \
"O13  O2L   " \
"O14  O2L   " \
"O11  OSL   " \
"O12  OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2   CTL2  " \
"HS   HAL2  " \
"HR   HAL2  " \
"C3   CTL2  " \
"HX   HAL2  " \
"HY   HAL2  " \
"O31  OSL   " \
"C31  CL    " \
"O32  OBL   " \
"C32  CTL2  " \
"H2X  HAL2  " \
"H2Y  HAL2  " \
"C33  CTL2  " \
"H3X  HAL2  " \
"H3Y  HAL2  " \
"C34  CTL2  " \
"H4X  HAL2  " \
"H4Y  HAL2  " \
"C35  CTL2  " \
"H5X  HAL2  " \
"H5Y  HAL2  " \
"C36  CTL2  " \
"H6X  HAL2  " \
"H6Y  HAL2  " \
"C37  CTL2  " \
"H7X  HAL2  " \
"H7Y  HAL2  " \
"C38  CTL2  " \
"H8X  HAL2  " \
"H8Y  HAL2  " \
"C39  CTL2  " \
"H9X  HAL2  " \
"H9Y  HAL2  " \
"C310 CTL2  " \
"H10X HAL2  " \
"H10Y HAL2  " \
"C311 CTL2  " \
"H11X HAL2  " \
"H11Y HAL2  " \
"C312 CTL3  " \
"H12X HAL3  " \
"H12Y HAL3  " \
"H12Z HAL3  " \
"N     NH3  " \
"HN1   HCL  " \
"HN2   HCL  " \
"HN3   HCL  " \
"C12   CTL  " \
"H12A  HAL  " \
"H12B  HAL  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"P    PL    " \
"O13  O2L   " \
"O14  O2L   " \
"O11  OSL   " \
"O12  OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2  CTL1   " \
"HS  HAL1   " \
"OH2  OHL   " \
"HO2  HOL   " \
"C3  CTL2   " \
"HX  HAL2   " \
"HY  HAL2   " \
"OH3  OHL   " \
"HO3  HOL   " \
"N     NH3  " \
"HN1   HCL  " \
"HN2   HCL  " \
"HN3   HCL  " \
"C12   CTL  " \
"H12A  HAL  " \
"H12B  HAL  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"P    PL    " \
"O13  O2L   " \
"O14  O2L   " \
"O11  OSL   " \
"O12  OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2   CTL1  " \
"HS   HAL1  " \
"O21  OSL   " \
"C21  CL    " \
"O22  OBL   " \
"C22  CTL2  " \
"H2R  HAL2  " \
"H2S  HAL2  " \
"C3   CTL2  " \
"HX   HAL2  " \
"HY   HAL2  " \
"O31  OSL   " \
"C31  CL    " \
"O32  OBL   " \
"C32  CTL2  " \
"H2X  HAL2  " \
"H2Y  HAL2  " \
"C23  CTL2  " \
"H3R  HAL2  " \
"H3S  HAL2  " \
"C24  CTL2  " \
"H4R  HAL2  " \
"H4S  HAL2  " \
"C25  CTL2  " \
"H5R  HAL2  " \
"H5S  HAL2  " \
"C26  CTL2  " \
"H6R  HAL2  " \
"H6S  HAL2  " \
"C27  CTL2  " \
"H7R  HAL2  " \
"H7S  HAL2  " \
"C28  CTL2  " \
"H8R  HAL2  " \
"H8S  HAL2  " \
"C29  CTL2  " \
"H9R  HAL2  " \
"H9S  HAL2  " \
"C210 CTL2  " \
"H10R HAL2  " \
"H10S HAL2  " \
"C211 CTL2  " \
"H11R HAL2  " \
"H11S HAL2  " \
"C212 CTL3  " \
"H12R HAL3  " \
"H12S HAL3  " \
"H12T HAL3  " \
"C33  CTL2  " \
"H3X  HAL2  " \
"H3Y  HAL2  " \
"C34  CTL2  " \
"H4X  HAL2  " \
"H4Y  HAL2  " \
"C35  CTL2  " \
"H5X  HAL2  " \
"H5Y  HAL2  " \
"C36  CTL2  " \
"H6X  HAL2  " \
"H6Y  HAL2  " \
"C37  CTL2  " \
"H7X  HAL2  " \
"H7Y  HAL2  " \
"C38  CTL2  " \
"H8X  HAL2  " \
"H8Y  HAL2  " \
"C39  CTL2  " \
"H9X  HAL2  " \
"H9Y  HAL2  " \
"C310 CTL2  " \
"H10X HAL2  " \
"H10Y HAL2  " \
"C311 CTL2  " \
"H11X HAL2  " \
"H11Y HAL2  " \
"C312 CTL3  " \
"H12X HAL3  " \
"H12Y HAL3  " \
"H12Z HAL3  " \
"N    NTL   " \
"C13  CTL5  " \
"H13A HL    " \
"H13B HL    " \
"H13C HL    " \
"C14  CTL5  " \
"H14A HL    " \
"H14B HL    " \
"H14C HL    " \
"C15  CTL5  " \
"H15A HL    " \
"H15B HL    " \
"H15C HL    " \
"C12  CTL2  " \
"H12A HL    " \
"H12B HL    " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"P    PL    " \
"O13  O2L   " \
"O14  O2L   " \
"O11  OSL   " \
"O12  OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2   CTL1  " \
"HS   HAL1  " \
"O21  OSL   " \
"C21  CL    " \
"O22  OBL   " \
"C22  CTL2  " \
"H2R  HAL2  " \
"H2S  HAL2  " \
"C3   CTL2  " \
"HX   HAL2  " \
"HY   HAL2  " \
"O31  OSL   " \
"C31  CL    " \
"O32  OBL   " \
"C32  CTL2  " \
"H2X  HAL2  " \
"H2Y  HAL2  " \
"C23  CTL2  " \
"H3R  HAL2  " \
"H3S  HAL2  " \
"C24  CTL2  " \
"H4R  HAL2  " \
"H4S  HAL2  " \
"C25  CTL2  " \
"H5R  HAL2  " \
"H5S  HAL2  " \
"C26  CTL2  " \
"H6R  HAL2  " \
"H6S  HAL2  " \
"C27  CTL2  " \
"H7R  HAL2  " \
"H7S  HAL2  " \
"C28  CTL2  " \
"H8R  HAL2  " \
"H8S  HAL2  " \
"C29  CTL2  " \
"H9R  HAL2  " \
"H9S  HAL2  " \
"C210 CTL2  " \
"H10R HAL2  " \
"H10S HAL2  " \
"C211 CTL2  " \
"H11R HAL2  " \
"H11S HAL2  " \
"C212 CTL2  " \
"H12R HAL2  " \
"H12S HAL2  " \
"C213 CTL2  " \
"H13R HAL2  " \
"H13S HAL2  " \
"C214 CTL3  " \
"H14R HAL3  " \
"H14S HAL3  " \
"H14T HAL3  " \
"C33  CTL2  " \
"H3X  HAL2  " \
"H3Y  HAL2  " \
"C34  CTL2  " \
"H4X  HAL2  " \
"H4Y  HAL2  " \
"C35  CTL2  " \
"H5X  HAL2  " \
"H5Y  HAL2  " \
"C36  CTL2  " \
"H6X  HAL2  " \
"H6Y  HAL2  " \
"C37  CTL2  " \
"H7X  HAL2  " \
"H7Y  HAL2  " \
"C38  CTL2  " \
"H8X  HAL2  " \
"H8Y  HAL2  " \
"C39  CTL2  " \
"H9X  HAL2  " \
"H9Y  HAL2  " \
"C310 CTL2  " \
"H10X HAL2  " \
"H10Y HAL2  " \
"C311 CTL2  " \
"H11X HAL2  " \
"H11Y HAL2  " \
"C312 CTL2  " \
"H12X HAL2  " \
"H12Y HAL2  " \
"C313 CTL2  " \
"H13X HAL2  " \
"H13Y HAL2  " \
"C314 CTL3  " \
"H14X HAL3  " \
"H14Y HAL3  " \
"H14Z HAL3  " \
"S    SL    " \
"OS1  OSL   " \
"OS2  O2L   " \
"OS3  O2L   " \
"OS4  O2L   " \
"C1   CTL2  " \
"H11  HAL2  " \
"H12  HAL2  " \
"C2   CTL2  " \
"H21  HAL2  " \
"H22  HAL2  " \
"C3   CTL2  " \
"H31  HAL2  " \
"H32  HAL2  " \
"C4   CTL2  " \
"H41  HAL2  " \
"H42  HAL2  " \
"C5   CTL2  " \
"H51  HAL2  " \
"H52  HAL2  " \
"C6   CTL2  " \
"H61  HAL2  " \
"H62  HAL2  " \
"C7   CTL2  " \
"H71  HAL2  " \
"H72  HAL2  " \
"C8   CTL2  " \
"H81  HAL2  " \
"H82  HAL2  " \
"C9   CTL2  " \
"H91  HAL2  " \
"H92  HAL2  " \
"C10  CTL2  " \
"H101 HAL2  " \
"H102 HAL2  " \
"C11  CTL2  " \
"H111 HAL2  " \
"H112 HAL2  " \
"C12  CTL3  " \
"H121 HAL3  " \
"H122 HAL3  " \
"H123 HAL3  " \
"N    NTL   " \
"C11  CTL2  " \
"C12  CTL5  " \
"C13  CTL5  " \
"C14  CTL5  " \
"H11  HL    " \
"H12  HL    " \
"H21  HL    " \
"H22  HL    " \
"H23  HL    " \
"H31  HL    " \
"H32  HL    " \
"H33  HL    " \
"H41  HL    " \
"H42  HL    " \
"H43  HL    " \
"C15  CTL2  " \
"H51  HAL2  " \
"H52  HAL2  " \
"P1   PL    " \
"O3   O2L   " \
"O4   O2L   " \
"O1   OSL   " \
"O2   OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2   CTL1  " \
"HS   HAL1  " \
"O21  OSL   " \
"C21  CL    " \
"O22  OBL   " \
"C22  CTL2  " \
"H2R  HAL2  " \
"H2S  HAL2  " \
"C3   CTL2  " \
"HX   HAL2  " \
"HY   HAL2  " \
"O31  OSL   " \
"C31  CL    " \
"O32  OBL   " \
"C32  CTL2  " \
"H2X  HAL2  " \
"H2Y  HAL2  " \
"C23  CTL2  " \
"H3R  HAL2  " \
"H3S  HAL2  " \
"C24  CTL2  " \
"H4R  HAL2  " \
"H4S  HAL2  " \
"C25  CTL2  " \
"H5R  HAL2  " \
"H5S  HAL2  " \
"C26  CTL2  " \
"H6R  HAL2  " \
"H6S  HAL2  " \
"C27  CTL2  " \
"H7R  HAL2  " \
"H7S  HAL2  " \
"C28  CTL2  " \
"H8R  HAL2  " \
"H8S  HAL2  " \
"C29  CEL1  " \
"H91  HEL1  " \
"C210 CEL1  " \
"H101 HEL1  " \
"C211 CTL2  " \
"H11R HAL2  " \
"H11S HAL2  " \
"C212 CTL2  " \
"H12R HAL2  " \
"H12S HAL2  " \
"C213 CTL2  " \
"H13R HAL2  " \
"H13S HAL2  " \
"C214 CTL2  " \
"H14R HAL2  " \
"H14S HAL2  " \
"C215 CTL2  " \
"H15R HAL2  " \
"H15S HAL2  " \
"C216 CTL2  " \
"H16R HAL2  " \
"H16S HAL2  " \
"C217 CTL2  " \
"H17R HAL2  " \
"H17S HAL2  " \
"C218 CTL3  " \
"H18R HAL3  " \
"H18S HAL3  " \
"H18T HAL3  " \
"C33  CTL2  " \
"H3X  HAL2  " \
"H3Y  HAL2  " \
"C34  CTL2  " \
"H4X  HAL2  " \
"H4Y  HAL2  " \
"C35  CTL2  " \
"H5X  HAL2  " \
"H5Y  HAL2  " \
"C36  CTL2  " \
"H6X  HAL2  " \
"H6Y  HAL2  " \
"C37  CTL2  " \
"H7X  HAL2  " \
"H7Y  HAL2  " \
"C38  CTL2  " \
"H8X  HAL2  " \
"H8Y  HAL2  " \
"C39  CTL2  " \
"H9X  HAL2  " \
"H9Y  HAL2  " \
"C310 CTL2  " \
"H10X HAL2  " \
"H10Y HAL2  " \
"C311 CTL2  " \
"H11X HAL2  " \
"H11Y HAL2  " \
"C312 CTL2  " \
"H12X HAL2  " \
"H12Y HAL2  " \
"C313 CTL2  " \
"H13X HAL2  " \
"H13Y HAL2  " \
"C314 CTL2  " \
"H14X HAL2  " \
"H14Y HAL2  " \
"C315 CTL2  " \
"H15X HAL2  " \
"H15Y HAL2  " \
"C316 CTL3  " \
"H16X HAL3  " \
"H16Y HAL3  " \
"H16Z HAL3  " \
"N     NH3  " \
"HN1   HCL  " \
"HN2   HCL  " \
"HN3   HCL  " \
"C12   CTL  " \
"H12A  HAL  " \
"H12B  HAL  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"P    PL    " \
"O13  O2L   " \
"O14  O2L   " \
"O11  OSL   " \
"O12  OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2   CTL1  " \
"HS   HAL1  " \
"O21  OSL   " \
"C21  CL    " \
"O22  OBL   " \
"C22  CTL2  " \
"H2R  HAL2  " \
"H2S  HAL2  " \
"C3   CTL2  " \
"HX   HAL2  " \
"HY   HAL2  " \
"O31  OSL   " \
"C31  CL    " \
"O32  OBL   " \
"C32  CTL2  " \
"H2X  HAL2  " \
"H2Y  HAL2  " \
"C23  CTL2  " \
"H3R  HAL2  " \
"H3S  HAL2  " \
"C24  CTL2  " \
"H4R  HAL2  " \
"H4S  HAL2  " \
"C25  CTL2  " \
"H5R  HAL2  " \
"H5S  HAL2  " \
"C26  CTL2  " \
"H6R  HAL2  " \
"H6S  HAL2  " \
"C27  CTL2  " \
"H7R  HAL2  " \
"H7S  HAL2  " \
"C28  CTL2  " \
"H8R  HAL2  " \
"H8S  HAL2  " \
"C29  CEL1  " \
"H91  HEL1  " \
"C210 CEL1  " \
"H101 HEL1  " \
"C211 CTL2  " \
"H11R HAL2  " \
"H11S HAL2  " \
"C212 CTL2  " \
"H12R HAL2  " \
"H12S HAL2  " \
"C213 CTL2  " \
"H13R HAL2  " \
"H13S HAL2  " \
"C214 CTL2  " \
"H14R HAL2  " \
"H14S HAL2  " \
"C215 CTL2  " \
"H15R HAL2  " \
"H15S HAL2  " \
"C216 CTL2  " \
"H16R HAL2  " \
"H16S HAL2  " \
"C217 CTL2  " \
"H17R HAL2  " \
"H17S HAL2  " \
"C218 CTL3  " \
"H18R HAL3  " \
"H18S HAL3  " \
"H18T HAL3  " \
"C33  CTL2  " \
"H3X  HAL2  " \
"H3Y  HAL2  " \
"C34  CTL2  " \
"H4X  HAL2  " \
"H4Y  HAL2  " \
"C35  CTL2  " \
"H5X  HAL2  " \
"H5Y  HAL2  " \
"C36  CTL2  " \
"H6X  HAL2  " \
"H6Y  HAL2  " \
"C37  CTL2  " \
"H7X  HAL2  " \
"H7Y  HAL2  " \
"C38  CTL2  " \
"H8X  HAL2  " \
"H8Y  HAL2  " \
"C39  CTL2  " \
"H9X  HAL2  " \
"H9Y  HAL2  " \
"C310 CTL2  " \
"H10X HAL2  " \
"H10Y HAL2  " \
"C311 CTL2  " \
"H11X HAL2  " \
"H11Y HAL2  " \
"C312 CTL2  " \
"H12X HAL2  " \
"H12Y HAL2  " \
"C313 CTL2  " \
"H13X HAL2  " \
"H13Y HAL2  " \
"C314 CTL2  " \
"H14X HAL2  " \
"H14Y HAL2  " \
"C315 CTL2  " \
"H15X HAL2  " \
"H15Y HAL2  " \
"C316 CTL3  " \
"H16X HAL3  " \
"H16Y HAL3  " \
"H16Z HAL3  " \
"C1   CL    " \
"O1   OCL   " \
"O2   OCL   " \
"C2   CTL2  " \
"H2A  HAL2  " \
"H2B  HAL2  " \
"C3   CTL2  " \
"H3A  HAL2  " \
"H3B  HAL2  " \
"C4   CTL2  " \
"H4A  HAL2  " \
"H4B  HAL2  " \
"C5   CTL2  " \
"H5A  HAL2  " \
"H5B  HAL2  " \
"C6   CTL2  " \
"H6A  HAL2  " \
"H6B  HAL2  " \
"C7   CTL2  " \
"H7A  HAL2  " \
"H7B  HAL2  " \
"C8   CTL2  " \
"H8A  HAL2  " \
"H8B  HAL2  " \
"C9   CTL2  " \
"H9A  HAL2  " \
"H9B  HAL2  " \
"C10  CTL2  " \
"H10A HAL2  " \
"H10B HAL2  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"C12  CTL2  " \
"H12A HAL2  " \
"H12B HAL2  " \
"C13  CTL2  " \
"H13A HAL2  " \
"H13B HAL2  " \
"C14  CTL2  " \
"H14A HAL2  " \
"H14B HAL2  " \
"C15  CTL2  " \
"H15A HAL2  " \
"H15B HAL2  " \
"C16  CTL3  " \
"H16A HAL3  " \
"H16B HAL3  " \
"H16C HAL3  " \
"C1   CL    " \
"O1   OCL   " \
"O2   OCL   " \
"C2   CTL2  " \
"H2A  HAL2  " \
"H2B  HAL2  " \
"C3   CTL2  " \
"H3A  HAL2  " \
"H3B  HAL2  " \
"C4   CTL2  " \
"H4A  HAL2  " \
"H4B  HAL2  " \
"C5   CTL2  " \
"H5A  HAL2  " \
"H5B  HAL2  " \
"C6   CTL2  " \
"H6A  HAL2  " \
"H6B  HAL2  " \
"C7   CTL2  " \
"H7A  HAL2  " \
"H7B  HAL2  " \
"C8   CTL2  " \
"H8A  HAL2  " \
"H8B  HAL2  " \
"C9   CEL1  " \
"H9A  HEL1  " \
"C10  CEL1  " \
"H10A HEL1  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"C12  CTL2  " \
"H12A HAL2  " \
"H12B HAL2  " \
"C13  CTL2  " \
"H13A HAL2  " \
"H13B HAL2  " \
"C14  CTL2  " \
"H14A HAL2  " \
"H14B HAL2  " \
"C15  CTL2  " \
"H15A HAL2  " \
"H15B HAL2  " \
"C16  CTL2  " \
"H16A HAL2  " \
"H16B HAL2  " \
"C17  CTL2  " \
"H17A HAL2  " \
"H17B HAL2  " \
"C18  CTL3  " \
"H18A HAL3  " \
"H18B HAL3  " \
"H18C HAL3  " \
"C1   CL    " \
"O1   OCL   " \
"O2   OCL   " \
"C2   CTL2  " \
"H2A  HAL2  " \
"H2B  HAL2  " \
"C3   CTL2  " \
"H3A  HAL2  " \
"H3B  HAL2  " \
"C4   CEL1  " \
"H4   HEL1  " \
"C5   CEL1  " \
"H5   HEL1  " \
"C6   CTL2  " \
"H6A  HAL2  " \
"H6B  HAL2  " \
"C7   CEL1  " \
"H7   HEL1  " \
"C8   CEL1  " \
"H8   HEL1  " \
"C9   CTL2  " \
"H9A  HAL2  " \
"H9B  HAL2  " \
"C10  CEL1  " \
"H10  HEL1  " \
"C11  CEL1  " \
"H11  HEL1  " \
"C12  CTL2  " \
"H12A HAL2  " \
"H12B HAL2  " \
"C13  CEL1  " \
"H13  HEL1  " \
"C14  CEL1  " \
"H14  HEL1  " \
"C15  CTL2  " \
"H15A HAL2  " \
"H15B HAL2  " \
"C16  CEL1  " \
"H16  HEL1  " \
"C17  CEL1  " \
"H17  HEL1  " \
"C18  CTL2  " \
"H18A HAL2  " \
"H18B HAL2  " \
"C19  CEL1  " \
"H19  HEL1  " \
"C20  CEL1  " \
"H20  HEL1  " \
"C21  CTL2  " \
"H21A HAL2  " \
"H21B HAL2  " \
"C22  CTL3  " \
"H22A HAL3  " \
"H22B HAL3  " \
"H22C HAL3  " \
"C1   CL    " \
"O1   OCL   " \
"O2   OCL   " \
"C2   CTL2  " \
"H2A  HAL2  " \
"H2B  HAL2  " \
"C3   CTL2  " \
"H3A  HAL2  " \
"H3B  HAL2  " \
"C4   CTL2  " \
"H4A  HAL2  " \
"H4B  HAL2  " \
"C5   CTL2  " \
"H5A  HAL2  " \
"H5B  HAL2  " \
"C6   CTL2  " \
"H6A  HAL2  " \
"H6B  HAL2  " \
"C7   CTL2  " \
"H7A  HAL2  " \
"H7B  HAL2  " \
"C8   CTL2  " \
"H8A  HAL2  " \
"H8B  HAL2  " \
"C9   CTL2  " \
"H9A  HAL2  " \
"H9B  HAL2  " \
"C10  CTL2  " \
"H10A HAL2  " \
"H10B HAL2  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"C12  CTL2  " \
"H12A HAL2  " \
"H12B HAL2  " \
"C13  CTL2  " \
"H13A HAL2  " \
"H13B HAL2  " \
"C14  CTL2  " \
"H14A HAL2  " \
"H14B HAL2  " \
"C15  CTL2  " \
"H15A HAL2  " \
"H15B HAL2  " \
"C16  CTL2  " \
"H16A HAL2  " \
"H16B HAL2  " \
"C17  CTL2  " \
"H17A HAL2  " \
"H17B HAL2  " \
"C18  CTL3  " \
"H18A HAL3  " \
"H18B HAL3  " \
"H18C HAL3  " \
"N    NTL   " \
"C5   CTL2  " \
"C6   CTL5  " \
"C7   CTL5  " \
"C8   CTL5  " \
"H5A  HL    " \
"H5B  HL    " \
"H6A  HL    " \
"H6B  HL    " \
"H6C  HL    " \
"H7A  HL    " \
"H7B  HL    " \
"H7C  HL    " \
"H8A  HL    " \
"H8B  HL    " \
"H8C  HL    " \
"C4   CTL2  " \
"H4A  HAL2  " \
"H4B  HAL2  " \
"P    PL    " \
"OP3  O2L   " \
"OP4  O2L   " \
"OP1  OSL   " \
"OP2  OSL   " \
"C3   CTL2  " \
"H3A  HAL2  " \
"H3B  HAL2  " \
"C2   CTL1  " \
"H2A  HAL1  " \
"O2   OHL   " \
"H2   HOL   " \
"C1   CTL2  " \
"H1A  HAL2  " \
"H1B  HAL2  " \
"O1   OHL   " \
"H1   HOL   " \
"1C1  CTL2  " \
"1H1A HAL2  " \
"1H1B HAL2  " \
"1O1  OSL   " \
"2C1  CL    " \
"2O1  OBL   " \
"2C2  CTL2  " \
"2H2A HAL2  " \
"2H2B HAL2  " \
"1C2  CTL1  " \
"1H2A HAL1  " \
"1O2  OSL   " \
"2C1  CL    " \
"2O1  OBL   " \
"2C2  CTL2  " \
"2H2A HAL2  " \
"2H2B HAL2  " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6B  " \
"C1'  CN7B  " \
"H1'  HN7   " \
"N9   NN2B  " \
"C4   CN5   " \
"N2   NN1   " \
"H21  HN1   " \
"H22  HN1   " \
"N3   NN3G  " \
"C2   CN2   " \
"N1   NN2G  " \
"H1   HN2   " \
"C6   CN1   " \
"O6   ON1   " \
"C5   CN5G  " \
"N7   NN4   " \
"C8   CN4   " \
"H8   HN3   " \
"C2'  CN7B  " \
"H2'' HN7   " \
"O2'  ON5   " \
"H2'  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON2   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6B  " \
"C1'  CN7B  " \
"H1'  HN7   " \
"N9   NN2   " \
"C5   CN5   " \
"N7   NN4   " \
"C8   CN4   " \
"H8   HN3   " \
"N1   NN3A  " \
"C2   CN4   " \
"H2   HN3   " \
"N3   NN3A  " \
"C4   CN5   " \
"C6   CN2   " \
"N6   NN1   " \
"H61  HN1   " \
"H62  HN1   " \
"C2'  CN7B  " \
"H2'' HN7   " \
"O2'  ON5   " \
"H2'  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON2   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6B  " \
"C1'  CN7B  " \
"H1'  HN7   " \
"N1   NN2   " \
"C6   CN3   " \
"H6   HN3   " \
"C5   CN3   " \
"H5   HN3   " \
"C2   CN1   " \
"O2   ON1C  " \
"N3   NN3   " \
"C4   CN2   " \
"N4   NN1   " \
"H41  HN1   " \
"H42  HN1   " \
"C2'  CN7B  " \
"H2'' HN7   " \
"O2'  ON5   " \
"H2'  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON2   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6B  " \
"C1'  CN7B  " \
"H1'  HN7   " \
"N1   NN2B  " \
"C6   CN3   " \
"H6   HN3   " \
"C2   CN1T  " \
"O2   ON1   " \
"N3   NN2U  " \
"H3   HN2   " \
"C4   CN1   " \
"O4   ON1   " \
"C5   CN3T  " \
"C5M  CN9   " \
"H51  HN9   " \
"H52  HN9   " \
"H53  HN9   " \
"C2'  CN7B  " \
"H2'' HN7   " \
"O2'  ON5   " \
"H2'  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON2   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6B  " \
"C1'  CN7B  " \
"H1'  HN7   " \
"N1   NN2B  " \
"C6   CN3   " \
"H6   HN3   " \
"C2   CN1T  " \
"O2   ON1   " \
"N3   NN2U  " \
"H3   HN2   " \
"C4   CN1   " \
"O4   ON1   " \
"C5   CN3   " \
"H5   HN3   " \
"C2'  CN7B  " \
"H2'' HN7   " \
"O2'  ON5   " \
"H2'  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON2   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6   " \
"C1'  CN7B  " \
"H1'  HN7   " \
"C2'  CN8   " \
"H2'  HN8   " \
"H2'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6   " \
"C1'  CN7B  " \
"H1'  HN7   " \
"C2'  CN8   " \
"H2'  HN8   " \
"H2'' HN8   " \
"H5T  HN5   " \
"O5'  ON5   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C5'  CN9   " \
"H5'  HN9   " \
"H5'' HN9   " \
"H53' HN9   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"O5T  ON4   " \
"H5T  HN4   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"O5T  ON2   " \
"C5T  CN9   " \
"H5T1 HN9   " \
"H5T2 HN9   " \
"H5T3 HN9   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON5   " \
"H3T  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"P3   P     " \
"O1P3 ON3   " \
"O2P3 ON3   " \
"O3'  ON2   " \
"O3T  ON4   " \
"H3T  HN4   " \
"C3'  CN7   " \
"H3'  HN7   " \
"P3   P     " \
"O1P3 ON3   " \
"O2P3 ON3   " \
"O3'  ON2   " \
"O3T  ON2   " \
"C3T  CN9   " \
"H3T1 HN9   " \
"H3T2 HN9   " \
"H3T3 HN9   " \
"C3'  CN7   " \
"H3'  HN7   " \
"P3   P     " \
"O3'  ON2   " \
"O1P3 ON3   " \
"O2P3 ON3   " \
"O3P3 ON3   " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"SOD  SOD   " \
"MG   MG    " \
"POT   POT  " \
"CES  CES   " \
"CAL  CAL   " \
"CLA  CLA   " \
"ZN   ZN    " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT3   " \
"HB1  HA    " \
"HB2  HA    " \
"HB3  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CT2   " \
"HD1  HA    " \
"HD2  HA    " \
"NE   NC2   " \
"HE   HC    " \
"CZ   C     " \
"NH1  NC2   " \
"HH11 HC    " \
"HH12 HC    " \
"NH2  NC2   " \
"HH21 HC    " \
"HH22 HC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CC    " \
"OD1  O     " \
"ND2  NH2   " \
"HD21 H     " \
"HD22 H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CC    " \
"OD1  OC    " \
"OD2  OC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"SG   S     " \
"HG1  HS    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CC    " \
"OE1  O     " \
"NE2  NH2   " \
"HE21 H     " \
"HE22 H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CC    " \
"OE1  OC    " \
"OE2  OC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT2   " \
"HA1  HB    " \
"HA2  HB    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"ND1  NR1   " \
"HD1  H     " \
"CG   CPH1  " \
"CE1  CPH2  " \
"HE1  HR1   " \
"NE2  NR2   " \
"CD2  CPH1  " \
"HD2  HR3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"ND1  NR2   " \
"CG   CPH1  " \
"CE1  CPH2  " \
"HE1  HR1   " \
"NE2  NR1   " \
"HE2  H     " \
"CD2  CPH1  " \
"HD2  HR3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CD2  CPH1  " \
"HD2  HR1   " \
"CG   CPH1  " \
"NE2  NR3   " \
"HE2  H     " \
"ND1  NR3   " \
"HD1  H     " \
"CE1  CPH2  " \
"HE1  HR2   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT1   " \
"HB   HA    " \
"CG2  CT3   " \
"HG21 HA    " \
"HG22 HA    " \
"HG23 HA    " \
"CG1  CT2   " \
"HG11 HA    " \
"HG12 HA    " \
"CD   CT3   " \
"HD1  HA    " \
"HD2  HA    " \
"HD3  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT1   " \
"HG   HA    " \
"CD1  CT3   " \
"HD11 HA    " \
"HD12 HA    " \
"HD13 HA    " \
"CD2  CT3   " \
"HD21 HA    " \
"HD22 HA    " \
"HD23 HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CT2   " \
"HD1  HA    " \
"HD2  HA    " \
"CE   CT2   " \
"HE1  HA    " \
"HE2  HA    " \
"NZ   NH3   " \
"HZ1  HC    " \
"HZ2  HC    " \
"HZ3  HC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"SD   S     " \
"CE   CT3   " \
"HE1  HA    " \
"HE2  HA    " \
"HE3  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CA    " \
"CD1  CA    " \
"HD1  HP    " \
"CE1  CA    " \
"HE1  HP    " \
"CZ   CA    " \
"HZ   HP    " \
"CD2  CA    " \
"HD2  HP    " \
"CE2  CA    " \
"HE2  HP    " \
"C    C     " \
"O    O     " \
"N    N     " \
"CD   CP3   " \
"HD1  HA    " \
"HD2  HA    " \
"CA   CP1   " \
"HA   HB    " \
"CB   CP2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CP2   " \
"HG1  HA    " \
"HG2  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"OG   OH1   " \
"HG1  H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT1   " \
"HB   HA    " \
"OG1  OH1   " \
"HG1  H     " \
"CG2  CT3   " \
"HG21 HA    " \
"HG22 HA    " \
"HG23 HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CY    " \
"CD1  CA    " \
"HD1  HP    " \
"NE1  NY    " \
"HE1  H     " \
"CE2  CPT   " \
"CD2  CPT   " \
"CE3  CA    " \
"HE3  HP    " \
"CZ3  CA    " \
"HZ3  HP    " \
"CZ2  CA    " \
"HZ2  HP    " \
"CH2  CA    " \
"HH2  HP    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CA    " \
"CD1  CA    " \
"HD1  HP    " \
"CE1  CA    " \
"HE1  HP    " \
"CZ   CA    " \
"OH   OH1   " \
"HH   H     " \
"CD2  CA    " \
"HD2  HP    " \
"CE2  CA    " \
"HE2  HP    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT1   " \
"HB   HA    " \
"CG1  CT3   " \
"HG11 HA    " \
"HG12 HA    " \
"HG13 HA    " \
"CG2  CT3   " \
"HG21 HA    " \
"HG22 HA    " \
"HG23 HA    " \
"C    C     " \
"O    O     " \
"CL   CT3   " \
"HL1  HA    " \
"HL2  HA    " \
"HL3  HA    " \
"CLP  C     " \
"OL   O     " \
"NL   NH1   " \
"HL   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT3   " \
"HB1  HA    " \
"HB2  HA    " \
"HB3  HA    " \
"CRP  C     " \
"OR   O     " \
"NR   NH1   " \
"HR   H     " \
"CR   CT3   " \
"HR1  HA    " \
"HR2  HA    " \
"HR3  HA    " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"N    NH3   " \
"HT1  HC    " \
"HT2  HC    " \
"HT3  HC    " \
"CA   CT1   " \
"HA   HB    " \
"N    NH3   " \
"HT1  HC    " \
"HT2  HC    " \
"HT3  HC    " \
"CA   CT2   " \
"HA1  HB    " \
"HA2  HB    " \
"N    NP    " \
"HN1  HC    " \
"HN2  HC    " \
"CD   CP3   " \
"HD1  HA    " \
"HD2  HA    " \
"CA   CP1   " \
"HA   HB    " \
"CAY  CT3   " \
"HY1  HA    " \
"HY2  HA    " \
"HY3  HA    " \
"CY   C     " \
"OY   O     " \
"CAY  CT3   " \
"HY1  HA    " \
"HY2  HA    " \
"HY3  HA    " \
"CY   C     " \
"OY   O     " \
"C    CC    " \
"OT1  OC    " \
"OT2  OC    " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"C    CD    " \
"OT1  OB    " \
"OT2  OS    " \
"CT   CT3   " \
"HT1  HA    " \
"HT2  HA    " \
"HT3  HA    " \
"C    CC    " \
"O    O     " \
"NT   NH2   " \
"HT1  H     " \
"HT2  H     " \
"C    C     " \
"O    O     " \
"NT   NH1   " \
"HNT  H     " \
"CAT  CT3   " \
"HT1  HA    " \
"HT2  HA    " \
"HT3  HA    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CD    " \
"OD1  OB    " \
"OD2  OH1   " \
"HD2  H     " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CD    " \
"OE1  OB    " \
"OE2  OH1   " \
"HE2  H     " \
"CE   CT2   " \
"HE1  HA    " \
"HE2  HA    " \
"NZ   NH2   " \
"HZ1  HC    " \
"HZ2  HC    " \
"1CB  CT2   " \
"1SG  SM    " \
"2SG  SM    " \
"2CB  CT2   " \
"CE1  CPH2  " \
"HE1  HR1   " \
"ND1  NR2   " \
"CG   CPH1  " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"NE2  NR1   " \
"HE2  H     " \
"CD2  CPH1  " \
"HD2  HR3   " \
"N    NTL   " \
"C13  CTL5  " \
"H13A HL    " \
"H13B HL    " \
"H13C HL    " \
"C14  CTL5  " \
"H14A HL    " \
"H14B HL    " \
"H14C HL    " \
"C15  CTL5  " \
"H15A HL    " \
"H15B HL    " \
"H15C HL    " \
"C12  CTL2  " \
"H12A HL    " \
"H12B HL    " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"P    PL    " \
"O13  O2L   " \
"O14  O2L   " \
"O11  OSL   " \
"O12  OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2   CTL2  " \
"HS   HAL2  " \
"HR   HAL2  " \
"C3   CTL2  " \
"HX   HAL2  " \
"HY   HAL2  " \
"O31  OSL   " \
"C31  CL    " \
"O32  OBL   " \
"C32  CTL2  " \
"H2X  HAL2  " \
"H2Y  HAL2  " \
"C33  CTL2  " \
"H3X  HAL2  " \
"H3Y  HAL2  " \
"C34  CTL2  " \
"H4X  HAL2  " \
"H4Y  HAL2  " \
"C35  CTL2  " \
"H5X  HAL2  " \
"H5Y  HAL2  " \
"C36  CTL2  " \
"H6X  HAL2  " \
"H6Y  HAL2  " \
"C37  CTL2  " \
"H7X  HAL2  " \
"H7Y  HAL2  " \
"C38  CTL2  " \
"H8X  HAL2  " \
"H8Y  HAL2  " \
"C39  CTL2  " \
"H9X  HAL2  " \
"H9Y  HAL2  " \
"C310 CTL2  " \
"H10X HAL2  " \
"H10Y HAL2  " \
"C311 CTL2  " \
"H11X HAL2  " \
"H11Y HAL2  " \
"C312 CTL3  " \
"H12X HAL3  " \
"H12Y HAL3  " \
"H12Z HAL3  " \
"N     NH3  " \
"HN1   HCL  " \
"HN2   HCL  " \
"HN3   HCL  " \
"C12   CTL  " \
"H12A  HAL  " \
"H12B  HAL  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"P    PL    " \
"O13  O2L   " \
"O14  O2L   " \
"O11  OSL   " \
"O12  OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2  CTL1   " \
"HS  HAL1   " \
"OH2  OHL   " \
"HO2  HOL   " \
"C3  CTL2   " \
"HX  HAL2   " \
"HY  HAL2   " \
"OH3  OHL   " \
"HO3  HOL   " \
"N     NH3  " \
"HN1   HCL  " \
"HN2   HCL  " \
"HN3   HCL  " \
"C12   CTL  " \
"H12A  HAL  " \
"H12B  HAL  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"P    PL    " \
"O13  O2L   " \
"O14  O2L   " \
"O11  OSL   " \
"O12  OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2   CTL1  " \
"HS   HAL1  " \
"O21  OSL   " \
"C21  CL    " \
"O22  OBL   " \
"C22  CTL2  " \
"H2R  HAL2  " \
"H2S  HAL2  " \
"C3   CTL2  " \
"HX   HAL2  " \
"HY   HAL2  " \
"O31  OSL   " \
"C31  CL    " \
"O32  OBL   " \
"C32  CTL2  " \
"H2X  HAL2  " \
"H2Y  HAL2  " \
"C23  CTL2  " \
"H3R  HAL2  " \
"H3S  HAL2  " \
"C24  CTL2  " \
"H4R  HAL2  " \
"H4S  HAL2  " \
"C25  CTL2  " \
"H5R  HAL2  " \
"H5S  HAL2  " \
"C26  CTL2  " \
"H6R  HAL2  " \
"H6S  HAL2  " \
"C27  CTL2  " \
"H7R  HAL2  " \
"H7S  HAL2  " \
"C28  CTL2  " \
"H8R  HAL2  " \
"H8S  HAL2  " \
"C29  CTL2  " \
"H9R  HAL2  " \
"H9S  HAL2  " \
"C210 CTL2  " \
"H10R HAL2  " \
"H10S HAL2  " \
"C211 CTL2  " \
"H11R HAL2  " \
"H11S HAL2  " \
"C212 CTL3  " \
"H12R HAL3  " \
"H12S HAL3  " \
"H12T HAL3  " \
"C33  CTL2  " \
"H3X  HAL2  " \
"H3Y  HAL2  " \
"C34  CTL2  " \
"H4X  HAL2  " \
"H4Y  HAL2  " \
"C35  CTL2  " \
"H5X  HAL2  " \
"H5Y  HAL2  " \
"C36  CTL2  " \
"H6X  HAL2  " \
"H6Y  HAL2  " \
"C37  CTL2  " \
"H7X  HAL2  " \
"H7Y  HAL2  " \
"C38  CTL2  " \
"H8X  HAL2  " \
"H8Y  HAL2  " \
"C39  CTL2  " \
"H9X  HAL2  " \
"H9Y  HAL2  " \
"C310 CTL2  " \
"H10X HAL2  " \
"H10Y HAL2  " \
"C311 CTL2  " \
"H11X HAL2  " \
"H11Y HAL2  " \
"C312 CTL3  " \
"H12X HAL3  " \
"H12Y HAL3  " \
"H12Z HAL3  " \
"N    NTL   " \
"C13  CTL5  " \
"H13A HL    " \
"H13B HL    " \
"H13C HL    " \
"C14  CTL5  " \
"H14A HL    " \
"H14B HL    " \
"H14C HL    " \
"C15  CTL5  " \
"H15A HL    " \
"H15B HL    " \
"H15C HL    " \
"C12  CTL2  " \
"H12A HL    " \
"H12B HL    " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"P    PL    " \
"O13  O2L   " \
"O14  O2L   " \
"O11  OSL   " \
"O12  OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2   CTL1  " \
"HS   HAL1  " \
"O21  OSL   " \
"C21  CL    " \
"O22  OBL   " \
"C22  CTL2  " \
"H2R  HAL2  " \
"H2S  HAL2  " \
"C3   CTL2  " \
"HX   HAL2  " \
"HY   HAL2  " \
"O31  OSL   " \
"C31  CL    " \
"O32  OBL   " \
"C32  CTL2  " \
"H2X  HAL2  " \
"H2Y  HAL2  " \
"C23  CTL2  " \
"H3R  HAL2  " \
"H3S  HAL2  " \
"C24  CTL2  " \
"H4R  HAL2  " \
"H4S  HAL2  " \
"C25  CTL2  " \
"H5R  HAL2  " \
"H5S  HAL2  " \
"C26  CTL2  " \
"H6R  HAL2  " \
"H6S  HAL2  " \
"C27  CTL2  " \
"H7R  HAL2  " \
"H7S  HAL2  " \
"C28  CTL2  " \
"H8R  HAL2  " \
"H8S  HAL2  " \
"C29  CTL2  " \
"H9R  HAL2  " \
"H9S  HAL2  " \
"C210 CTL2  " \
"H10R HAL2  " \
"H10S HAL2  " \
"C211 CTL2  " \
"H11R HAL2  " \
"H11S HAL2  " \
"C212 CTL2  " \
"H12R HAL2  " \
"H12S HAL2  " \
"C213 CTL2  " \
"H13R HAL2  " \
"H13S HAL2  " \
"C214 CTL3  " \
"H14R HAL3  " \
"H14S HAL3  " \
"H14T HAL3  " \
"C33  CTL2  " \
"H3X  HAL2  " \
"H3Y  HAL2  " \
"C34  CTL2  " \
"H4X  HAL2  " \
"H4Y  HAL2  " \
"C35  CTL2  " \
"H5X  HAL2  " \
"H5Y  HAL2  " \
"C36  CTL2  " \
"H6X  HAL2  " \
"H6Y  HAL2  " \
"C37  CTL2  " \
"H7X  HAL2  " \
"H7Y  HAL2  " \
"C38  CTL2  " \
"H8X  HAL2  " \
"H8Y  HAL2  " \
"C39  CTL2  " \
"H9X  HAL2  " \
"H9Y  HAL2  " \
"C310 CTL2  " \
"H10X HAL2  " \
"H10Y HAL2  " \
"C311 CTL2  " \
"H11X HAL2  " \
"H11Y HAL2  " \
"C312 CTL2  " \
"H12X HAL2  " \
"H12Y HAL2  " \
"C313 CTL2  " \
"H13X HAL2  " \
"H13Y HAL2  " \
"C314 CTL3  " \
"H14X HAL3  " \
"H14Y HAL3  " \
"H14Z HAL3  " \
"S    SL    " \
"OS1  OSL   " \
"OS2  O2L   " \
"OS3  O2L   " \
"OS4  O2L   " \
"C1   CTL2  " \
"H11  HAL2  " \
"H12  HAL2  " \
"C2   CTL2  " \
"H21  HAL2  " \
"H22  HAL2  " \
"C3   CTL2  " \
"H31  HAL2  " \
"H32  HAL2  " \
"C4   CTL2  " \
"H41  HAL2  " \
"H42  HAL2  " \
"C5   CTL2  " \
"H51  HAL2  " \
"H52  HAL2  " \
"C6   CTL2  " \
"H61  HAL2  " \
"H62  HAL2  " \
"C7   CTL2  " \
"H71  HAL2  " \
"H72  HAL2  " \
"C8   CTL2  " \
"H81  HAL2  " \
"H82  HAL2  " \
"C9   CTL2  " \
"H91  HAL2  " \
"H92  HAL2  " \
"C10  CTL2  " \
"H101 HAL2  " \
"H102 HAL2  " \
"C11  CTL2  " \
"H111 HAL2  " \
"H112 HAL2  " \
"C12  CTL3  " \
"H121 HAL3  " \
"H122 HAL3  " \
"H123 HAL3  " \
"N    NTL   " \
"C11  CTL2  " \
"C12  CTL5  " \
"C13  CTL5  " \
"C14  CTL5  " \
"H11  HL    " \
"H12  HL    " \
"H21  HL    " \
"H22  HL    " \
"H23  HL    " \
"H31  HL    " \
"H32  HL    " \
"H33  HL    " \
"H41  HL    " \
"H42  HL    " \
"H43  HL    " \
"C15  CTL2  " \
"H51  HAL2  " \
"H52  HAL2  " \
"P1   PL    " \
"O3   O2L   " \
"O4   O2L   " \
"O1   OSL   " \
"O2   OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2   CTL1  " \
"HS   HAL1  " \
"O21  OSL   " \
"C21  CL    " \
"O22  OBL   " \
"C22  CTL2  " \
"H2R  HAL2  " \
"H2S  HAL2  " \
"C3   CTL2  " \
"HX   HAL2  " \
"HY   HAL2  " \
"O31  OSL   " \
"C31  CL    " \
"O32  OBL   " \
"C32  CTL2  " \
"H2X  HAL2  " \
"H2Y  HAL2  " \
"C23  CTL2  " \
"H3R  HAL2  " \
"H3S  HAL2  " \
"C24  CTL2  " \
"H4R  HAL2  " \
"H4S  HAL2  " \
"C25  CTL2  " \
"H5R  HAL2  " \
"H5S  HAL2  " \
"C26  CTL2  " \
"H6R  HAL2  " \
"H6S  HAL2  " \
"C27  CTL2  " \
"H7R  HAL2  " \
"H7S  HAL2  " \
"C28  CTL2  " \
"H8R  HAL2  " \
"H8S  HAL2  " \
"C29  CEL1  " \
"H91  HEL1  " \
"C210 CEL1  " \
"H101 HEL1  " \
"C211 CTL2  " \
"H11R HAL2  " \
"H11S HAL2  " \
"C212 CTL2  " \
"H12R HAL2  " \
"H12S HAL2  " \
"C213 CTL2  " \
"H13R HAL2  " \
"H13S HAL2  " \
"C214 CTL2  " \
"H14R HAL2  " \
"H14S HAL2  " \
"C215 CTL2  " \
"H15R HAL2  " \
"H15S HAL2  " \
"C216 CTL2  " \
"H16R HAL2  " \
"H16S HAL2  " \
"C217 CTL2  " \
"H17R HAL2  " \
"H17S HAL2  " \
"C218 CTL3  " \
"H18R HAL3  " \
"H18S HAL3  " \
"H18T HAL3  " \
"C33  CTL2  " \
"H3X  HAL2  " \
"H3Y  HAL2  " \
"C34  CTL2  " \
"H4X  HAL2  " \
"H4Y  HAL2  " \
"C35  CTL2  " \
"H5X  HAL2  " \
"H5Y  HAL2  " \
"C36  CTL2  " \
"H6X  HAL2  " \
"H6Y  HAL2  " \
"C37  CTL2  " \
"H7X  HAL2  " \
"H7Y  HAL2  " \
"C38  CTL2  " \
"H8X  HAL2  " \
"H8Y  HAL2  " \
"C39  CTL2  " \
"H9X  HAL2  " \
"H9Y  HAL2  " \
"C310 CTL2  " \
"H10X HAL2  " \
"H10Y HAL2  " \
"C311 CTL2  " \
"H11X HAL2  " \
"H11Y HAL2  " \
"C312 CTL2  " \
"H12X HAL2  " \
"H12Y HAL2  " \
"C313 CTL2  " \
"H13X HAL2  " \
"H13Y HAL2  " \
"C314 CTL2  " \
"H14X HAL2  " \
"H14Y HAL2  " \
"C315 CTL2  " \
"H15X HAL2  " \
"H15Y HAL2  " \
"C316 CTL3  " \
"H16X HAL3  " \
"H16Y HAL3  " \
"H16Z HAL3  " \
"N     NH3  " \
"HN1   HCL  " \
"HN2   HCL  " \
"HN3   HCL  " \
"C12   CTL  " \
"H12A  HAL  " \
"H12B  HAL  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"P    PL    " \
"O13  O2L   " \
"O14  O2L   " \
"O11  OSL   " \
"O12  OSL   " \
"C1   CTL2  " \
"HA   HAL2  " \
"HB   HAL2  " \
"C2   CTL1  " \
"HS   HAL1  " \
"O21  OSL   " \
"C21  CL    " \
"O22  OBL   " \
"C22  CTL2  " \
"H2R  HAL2  " \
"H2S  HAL2  " \
"C3   CTL2  " \
"HX   HAL2  " \
"HY   HAL2  " \
"O31  OSL   " \
"C31  CL    " \
"O32  OBL   " \
"C32  CTL2  " \
"H2X  HAL2  " \
"H2Y  HAL2  " \
"C23  CTL2  " \
"H3R  HAL2  " \
"H3S  HAL2  " \
"C24  CTL2  " \
"H4R  HAL2  " \
"H4S  HAL2  " \
"C25  CTL2  " \
"H5R  HAL2  " \
"H5S  HAL2  " \
"C26  CTL2  " \
"H6R  HAL2  " \
"H6S  HAL2  " \
"C27  CTL2  " \
"H7R  HAL2  " \
"H7S  HAL2  " \
"C28  CTL2  " \
"H8R  HAL2  " \
"H8S  HAL2  " \
"C29  CEL1  " \
"H91  HEL1  " \
"C210 CEL1  " \
"H101 HEL1  " \
"C211 CTL2  " \
"H11R HAL2  " \
"H11S HAL2  " \
"C212 CTL2  " \
"H12R HAL2  " \
"H12S HAL2  " \
"C213 CTL2  " \
"H13R HAL2  " \
"H13S HAL2  " \
"C214 CTL2  " \
"H14R HAL2  " \
"H14S HAL2  " \
"C215 CTL2  " \
"H15R HAL2  " \
"H15S HAL2  " \
"C216 CTL2  " \
"H16R HAL2  " \
"H16S HAL2  " \
"C217 CTL2  " \
"H17R HAL2  " \
"H17S HAL2  " \
"C218 CTL3  " \
"H18R HAL3  " \
"H18S HAL3  " \
"H18T HAL3  " \
"C33  CTL2  " \
"H3X  HAL2  " \
"H3Y  HAL2  " \
"C34  CTL2  " \
"H4X  HAL2  " \
"H4Y  HAL2  " \
"C35  CTL2  " \
"H5X  HAL2  " \
"H5Y  HAL2  " \
"C36  CTL2  " \
"H6X  HAL2  " \
"H6Y  HAL2  " \
"C37  CTL2  " \
"H7X  HAL2  " \
"H7Y  HAL2  " \
"C38  CTL2  " \
"H8X  HAL2  " \
"H8Y  HAL2  " \
"C39  CTL2  " \
"H9X  HAL2  " \
"H9Y  HAL2  " \
"C310 CTL2  " \
"H10X HAL2  " \
"H10Y HAL2  " \
"C311 CTL2  " \
"H11X HAL2  " \
"H11Y HAL2  " \
"C312 CTL2  " \
"H12X HAL2  " \
"H12Y HAL2  " \
"C313 CTL2  " \
"H13X HAL2  " \
"H13Y HAL2  " \
"C314 CTL2  " \
"H14X HAL2  " \
"H14Y HAL2  " \
"C315 CTL2  " \
"H15X HAL2  " \
"H15Y HAL2  " \
"C316 CTL3  " \
"H16X HAL3  " \
"H16Y HAL3  " \
"H16Z HAL3  " \
"C1   CL    " \
"O1   OCL   " \
"O2   OCL   " \
"C2   CTL2  " \
"H2A  HAL2  " \
"H2B  HAL2  " \
"C3   CTL2  " \
"H3A  HAL2  " \
"H3B  HAL2  " \
"C4   CTL2  " \
"H4A  HAL2  " \
"H4B  HAL2  " \
"C5   CTL2  " \
"H5A  HAL2  " \
"H5B  HAL2  " \
"C6   CTL2  " \
"H6A  HAL2  " \
"H6B  HAL2  " \
"C7   CTL2  " \
"H7A  HAL2  " \
"H7B  HAL2  " \
"C8   CTL2  " \
"H8A  HAL2  " \
"H8B  HAL2  " \
"C9   CTL2  " \
"H9A  HAL2  " \
"H9B  HAL2  " \
"C10  CTL2  " \
"H10A HAL2  " \
"H10B HAL2  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"C12  CTL2  " \
"H12A HAL2  " \
"H12B HAL2  " \
"C13  CTL2  " \
"H13A HAL2  " \
"H13B HAL2  " \
"C14  CTL2  " \
"H14A HAL2  " \
"H14B HAL2  " \
"C15  CTL2  " \
"H15A HAL2  " \
"H15B HAL2  " \
"C16  CTL3  " \
"H16A HAL3  " \
"H16B HAL3  " \
"H16C HAL3  " \
"C1   CL    " \
"O1   OCL   " \
"O2   OCL   " \
"C2   CTL2  " \
"H2A  HAL2  " \
"H2B  HAL2  " \
"C3   CTL2  " \
"H3A  HAL2  " \
"H3B  HAL2  " \
"C4   CTL2  " \
"H4A  HAL2  " \
"H4B  HAL2  " \
"C5   CTL2  " \
"H5A  HAL2  " \
"H5B  HAL2  " \
"C6   CTL2  " \
"H6A  HAL2  " \
"H6B  HAL2  " \
"C7   CTL2  " \
"H7A  HAL2  " \
"H7B  HAL2  " \
"C8   CTL2  " \
"H8A  HAL2  " \
"H8B  HAL2  " \
"C9   CEL1  " \
"H9A  HEL1  " \
"C10  CEL1  " \
"H10A HEL1  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"C12  CTL2  " \
"H12A HAL2  " \
"H12B HAL2  " \
"C13  CTL2  " \
"H13A HAL2  " \
"H13B HAL2  " \
"C14  CTL2  " \
"H14A HAL2  " \
"H14B HAL2  " \
"C15  CTL2  " \
"H15A HAL2  " \
"H15B HAL2  " \
"C16  CTL2  " \
"H16A HAL2  " \
"H16B HAL2  " \
"C17  CTL2  " \
"H17A HAL2  " \
"H17B HAL2  " \
"C18  CTL3  " \
"H18A HAL3  " \
"H18B HAL3  " \
"H18C HAL3  " \
"C1   CL    " \
"O1   OCL   " \
"O2   OCL   " \
"C2   CTL2  " \
"H2A  HAL2  " \
"H2B  HAL2  " \
"C3   CTL2  " \
"H3A  HAL2  " \
"H3B  HAL2  " \
"C4   CEL1  " \
"H4   HEL1  " \
"C5   CEL1  " \
"H5   HEL1  " \
"C6   CTL2  " \
"H6A  HAL2  " \
"H6B  HAL2  " \
"C7   CEL1  " \
"H7   HEL1  " \
"C8   CEL1  " \
"H8   HEL1  " \
"C9   CTL2  " \
"H9A  HAL2  " \
"H9B  HAL2  " \
"C10  CEL1  " \
"H10  HEL1  " \
"C11  CEL1  " \
"H11  HEL1  " \
"C12  CTL2  " \
"H12A HAL2  " \
"H12B HAL2  " \
"C13  CEL1  " \
"H13  HEL1  " \
"C14  CEL1  " \
"H14  HEL1  " \
"C15  CTL2  " \
"H15A HAL2  " \
"H15B HAL2  " \
"C16  CEL1  " \
"H16  HEL1  " \
"C17  CEL1  " \
"H17  HEL1  " \
"C18  CTL2  " \
"H18A HAL2  " \
"H18B HAL2  " \
"C19  CEL1  " \
"H19  HEL1  " \
"C20  CEL1  " \
"H20  HEL1  " \
"C21  CTL2  " \
"H21A HAL2  " \
"H21B HAL2  " \
"C22  CTL3  " \
"H22A HAL3  " \
"H22B HAL3  " \
"H22C HAL3  " \
"C1   CL    " \
"O1   OCL   " \
"O2   OCL   " \
"C2   CTL2  " \
"H2A  HAL2  " \
"H2B  HAL2  " \
"C3   CTL2  " \
"H3A  HAL2  " \
"H3B  HAL2  " \
"C4   CTL2  " \
"H4A  HAL2  " \
"H4B  HAL2  " \
"C5   CTL2  " \
"H5A  HAL2  " \
"H5B  HAL2  " \
"C6   CTL2  " \
"H6A  HAL2  " \
"H6B  HAL2  " \
"C7   CTL2  " \
"H7A  HAL2  " \
"H7B  HAL2  " \
"C8   CTL2  " \
"H8A  HAL2  " \
"H8B  HAL2  " \
"C9   CTL2  " \
"H9A  HAL2  " \
"H9B  HAL2  " \
"C10  CTL2  " \
"H10A HAL2  " \
"H10B HAL2  " \
"C11  CTL2  " \
"H11A HAL2  " \
"H11B HAL2  " \
"C12  CTL2  " \
"H12A HAL2  " \
"H12B HAL2  " \
"C13  CTL2  " \
"H13A HAL2  " \
"H13B HAL2  " \
"C14  CTL2  " \
"H14A HAL2  " \
"H14B HAL2  " \
"C15  CTL2  " \
"H15A HAL2  " \
"H15B HAL2  " \
"C16  CTL2  " \
"H16A HAL2  " \
"H16B HAL2  " \
"C17  CTL2  " \
"H17A HAL2  " \
"H17B HAL2  " \
"C18  CTL3  " \
"H18A HAL3  " \
"H18B HAL3  " \
"H18C HAL3  " \
"N    NTL   " \
"C5   CTL2  " \
"C6   CTL5  " \
"C7   CTL5  " \
"C8   CTL5  " \
"H5A  HL    " \
"H5B  HL    " \
"H6A  HL    " \
"H6B  HL    " \
"H6C  HL    " \
"H7A  HL    " \
"H7B  HL    " \
"H7C  HL    " \
"H8A  HL    " \
"H8B  HL    " \
"H8C  HL    " \
"C4   CTL2  " \
"H4A  HAL2  " \
"H4B  HAL2  " \
"P    PL    " \
"OP3  O2L   " \
"OP4  O2L   " \
"OP1  OSL   " \
"OP2  OSL   " \
"C3   CTL2  " \
"H3A  HAL2  " \
"H3B  HAL2  " \
"C2   CTL1  " \
"H2A  HAL1  " \
"O2   OHL   " \
"H2   HOL   " \
"C1   CTL2  " \
"H1A  HAL2  " \
"H1B  HAL2  " \
"O1   OHL   " \
"H1   HOL   " \
"1C1  CTL2  " \
"1H1A HAL2  " \
"1H1B HAL2  " \
"1O1  OSL   " \
"2C1  CL    " \
"2O1  OBL   " \
"2C2  CTL2  " \
"2H2A HAL2  " \
"2H2B HAL2  " \
"1C2  CTL1  " \
"1H2A HAL1  " \
"1O2  OSL   " \
"2C1  CL    " \
"2O1  OBL   " \
"2C2  CTL2  " \
"2H2A HAL2  " \
"2H2B HAL2  " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT3   " \
"HB1  HA    " \
"HB2  HA    " \
"HB3  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CT2   " \
"HD1  HA    " \
"HD2  HA    " \
"NE   NC2   " \
"HE   HC    " \
"CZ   C     " \
"NH1  NC2   " \
"HH11 HC    " \
"HH12 HC    " \
"NH2  NC2   " \
"HH21 HC    " \
"HH22 HC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CC    " \
"OD1  O     " \
"ND2  NH2   " \
"HD21 H     " \
"HD22 H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CC    " \
"OD1  OC    " \
"OD2  OC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"SG   S     " \
"HG1  HS    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CC    " \
"OE1  O     " \
"NE2  NH2   " \
"HE21 H     " \
"HE22 H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CC    " \
"OE1  OC    " \
"OE2  OC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT2   " \
"HA1  HB    " \
"HA2  HB    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"ND1  NR1   " \
"HD1  H     " \
"CG   CPH1  " \
"CE1  CPH2  " \
"HE1  HR1   " \
"NE2  NR2   " \
"CD2  CPH1  " \
"HD2  HR3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"ND1  NR2   " \
"CG   CPH1  " \
"CE1  CPH2  " \
"HE1  HR1   " \
"NE2  NR1   " \
"HE2  H     " \
"CD2  CPH1  " \
"HD2  HR3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CD2  CPH1  " \
"HD2  HR1   " \
"CG   CPH1  " \
"NE2  NR3   " \
"HE2  H     " \
"ND1  NR3   " \
"HD1  H     " \
"CE1  CPH2  " \
"HE1  HR2   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT1   " \
"HB   HA    " \
"CG2  CT3   " \
"HG21 HA    " \
"HG22 HA    " \
"HG23 HA    " \
"CG1  CT2   " \
"HG11 HA    " \
"HG12 HA    " \
"CD   CT3   " \
"HD1  HA    " \
"HD2  HA    " \
"HD3  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT1   " \
"HG   HA    " \
"CD1  CT3   " \
"HD11 HA    " \
"HD12 HA    " \
"HD13 HA    " \
"CD2  CT3   " \
"HD21 HA    " \
"HD22 HA    " \
"HD23 HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CT2   " \
"HD1  HA    " \
"HD2  HA    " \
"CE   CT2   " \
"HE1  HA    " \
"HE2  HA    " \
"NZ   NH3   " \
"HZ1  HC    " \
"HZ2  HC    " \
"HZ3  HC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"SD   S     " \
"CE   CT3   " \
"HE1  HA    " \
"HE2  HA    " \
"HE3  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CA    " \
"CD1  CA    " \
"HD1  HP    " \
"CE1  CA    " \
"HE1  HP    " \
"CZ   CA    " \
"HZ   HP    " \
"CD2  CA    " \
"HD2  HP    " \
"CE2  CA    " \
"HE2  HP    " \
"C    C     " \
"O    O     " \
"N    N     " \
"CD   CP3   " \
"HD1  HA    " \
"HD2  HA    " \
"CA   CP1   " \
"HA   HB    " \
"CB   CP2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CP2   " \
"HG1  HA    " \
"HG2  HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"OG   OH1   " \
"HG1  H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT1   " \
"HB   HA    " \
"OG1  OH1   " \
"HG1  H     " \
"CG2  CT3   " \
"HG21 HA    " \
"HG22 HA    " \
"HG23 HA    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CY    " \
"CD1  CA    " \
"HD1  HP    " \
"NE1  NY    " \
"HE1  H     " \
"CE2  CPT   " \
"CD2  CPT   " \
"CE3  CA    " \
"HE3  HP    " \
"CZ3  CA    " \
"HZ3  HP    " \
"CZ2  CA    " \
"HZ2  HP    " \
"CH2  CA    " \
"HH2  HP    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CA    " \
"CD1  CA    " \
"HD1  HP    " \
"CE1  CA    " \
"HE1  HP    " \
"CZ   CA    " \
"OH   OH1   " \
"HH   H     " \
"CD2  CA    " \
"HD2  HP    " \
"CE2  CA    " \
"HE2  HP    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT1   " \
"HB   HA    " \
"CG1  CT3   " \
"HG11 HA    " \
"HG12 HA    " \
"HG13 HA    " \
"CG2  CT3   " \
"HG21 HA    " \
"HG22 HA    " \
"HG23 HA    " \
"C    C     " \
"O    O     " \
"CL   CT3   " \
"HL1  HA    " \
"HL2  HA    " \
"HL3  HA    " \
"CLP  C     " \
"OL   O     " \
"NL   NH1   " \
"HL   H     " \
"CA   CT1   " \
"HA   HB    " \
"CB   CT3   " \
"HB1  HA    " \
"HB2  HA    " \
"HB3  HA    " \
"CRP  C     " \
"OR   O     " \
"NR   NH1   " \
"HR   H     " \
"CR   CT3   " \
"HR1  HA    " \
"HR2  HA    " \
"HR3  HA    " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"N    NH3   " \
"HT1  HC    " \
"HT2  HC    " \
"HT3  HC    " \
"CA   CT1   " \
"HA   HB    " \
"N    NH3   " \
"HT1  HC    " \
"HT2  HC    " \
"HT3  HC    " \
"CA   CT2   " \
"HA1  HB    " \
"HA2  HB    " \
"N    NP    " \
"HN1  HC    " \
"HN2  HC    " \
"CD   CP3   " \
"HD1  HA    " \
"HD2  HA    " \
"CA   CP1   " \
"HA   HB    " \
"CAY  CT3   " \
"HY1  HA    " \
"HY2  HA    " \
"HY3  HA    " \
"CY   C     " \
"OY   O     " \
"CAY  CT3   " \
"HY1  HA    " \
"HY2  HA    " \
"HY3  HA    " \
"CY   C     " \
"OY   O     " \
"C    CC    " \
"OT1  OC    " \
"OT2  OC    " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB    " \
"C    CD    " \
"OT1  OB    " \
"OT2  OS    " \
"CT   CT3   " \
"HT1  HA    " \
"HT2  HA    " \
"HT3  HA    " \
"C    CC    " \
"O    O     " \
"NT   NH2   " \
"HT1  H     " \
"HT2  H     " \
"C    C     " \
"O    O     " \
"NT   NH1   " \
"HNT  H     " \
"CAT  CT3   " \
"HT1  HA    " \
"HT2  HA    " \
"HT3  HA    " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"CG   CD    " \
"OD1  OB    " \
"OD2  OH1   " \
"HD2  H     " \
"CG   CT2   " \
"HG1  HA    " \
"HG2  HA    " \
"CD   CD    " \
"OE1  OB    " \
"OE2  OH1   " \
"HE2  H     " \
"CE   CT2   " \
"HE1  HA    " \
"HE2  HA    " \
"NZ   NH2   " \
"HZ1  HC    " \
"HZ2  HC    " \
"1CB  CT2   " \
"1SG  SM    " \
"2SG  SM    " \
"2CB  CT2   " \
"CE1  CPH2  " \
"HE1  HR1   " \
"ND1  NR2   " \
"CG   CPH1  " \
"CB   CT2   " \
"HB1  HA    " \
"HB2  HA    " \
"NE2  NR1   " \
"HE2  H     " \
"CD2  CPH1  " \
"HD2  HR3   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6B  " \
"C1'  CN7B  " \
"H1'  HN7   " \
"N9   NN2B  " \
"C4   CN5   " \
"N2   NN1   " \
"H21  HN1   " \
"H22  HN1   " \
"N3   NN3G  " \
"C2   CN2   " \
"N1   NN2G  " \
"H1   HN2   " \
"C6   CN1   " \
"O6   ON1   " \
"C5   CN5G  " \
"N7   NN4   " \
"C8   CN4   " \
"H8   HN3   " \
"C2'  CN7B  " \
"H2'' HN7   " \
"O2'  ON5   " \
"H2'  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON2   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6B  " \
"C1'  CN7B  " \
"H1'  HN7   " \
"N9   NN2   " \
"C5   CN5   " \
"N7   NN4   " \
"C8   CN4   " \
"H8   HN3   " \
"N1   NN3A  " \
"C2   CN4   " \
"H2   HN3   " \
"N3   NN3A  " \
"C4   CN5   " \
"C6   CN2   " \
"N6   NN1   " \
"H61  HN1   " \
"H62  HN1   " \
"C2'  CN7B  " \
"H2'' HN7   " \
"O2'  ON5   " \
"H2'  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON2   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6B  " \
"C1'  CN7B  " \
"H1'  HN7   " \
"N1   NN2   " \
"C6   CN3   " \
"H6   HN3   " \
"C5   CN3   " \
"H5   HN3   " \
"C2   CN1   " \
"O2   ON1C  " \
"N3   NN3   " \
"C4   CN2   " \
"N4   NN1   " \
"H41  HN1   " \
"H42  HN1   " \
"C2'  CN7B  " \
"H2'' HN7   " \
"O2'  ON5   " \
"H2'  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON2   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6B  " \
"C1'  CN7B  " \
"H1'  HN7   " \
"N1   NN2B  " \
"C6   CN3   " \
"H6   HN3   " \
"C2   CN1T  " \
"O2   ON1   " \
"N3   NN2U  " \
"H3   HN2   " \
"C4   CN1   " \
"O4   ON1   " \
"C5   CN3T  " \
"C5M  CN9   " \
"H51  HN9   " \
"H52  HN9   " \
"H53  HN9   " \
"C2'  CN7B  " \
"H2'' HN7   " \
"O2'  ON5   " \
"H2'  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON2   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6B  " \
"C1'  CN7B  " \
"H1'  HN7   " \
"N1   NN2B  " \
"C6   CN3   " \
"H6   HN3   " \
"C2   CN1T  " \
"O2   ON1   " \
"N3   NN2U  " \
"H3   HN2   " \
"C4   CN1   " \
"O4   ON1   " \
"C5   CN3   " \
"H5   HN3   " \
"C2'  CN7B  " \
"H2'' HN7   " \
"O2'  ON5   " \
"H2'  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON2   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6   " \
"C1'  CN7B  " \
"H1'  HN7   " \
"C2'  CN8   " \
"H2'  HN8   " \
"H2'' HN8   " \
"C4'  CN7   " \
"H4'  HN7   " \
"O4'  ON6   " \
"C1'  CN7B  " \
"H1'  HN7   " \
"C2'  CN8   " \
"H2'  HN8   " \
"H2'' HN8   " \
"H5T  HN5   " \
"O5'  ON5   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"C5'  CN9   " \
"H5'  HN9   " \
"H5'' HN9   " \
"H53' HN9   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"O5T  ON4   " \
"H5T  HN4   " \
"C5'  CN8B  " \
"H5'  HN8   " \
"H5'' HN8   " \
"P    P     " \
"O1P  ON3   " \
"O2P  ON3   " \
"O5'  ON2   " \
"O5T  ON2   " \
"C5T  CN9   " \
"H5T1 HN9   " \
"H5T2 HN9   " \
"H5T3 HN9   " \
"C3'  CN7   " \
"H3'  HN7   " \
"O3'  ON5   " \
"H3T  HN5   " \
"C3'  CN7   " \
"H3'  HN7   " \
"P3   P     " \
"O1P3 ON3   " \
"O2P3 ON3   " \
"O3'  ON2   " \
"O3T  ON4   " \
"H3T  HN4   " \
"C3'  CN7   " \
"H3'  HN7   " \
"P3   P     " \
"O1P3 ON3   " \
"O2P3 ON3   " \
"O3'  ON2   " \
"O3T  ON2   " \
"C3T  CN9   " \
"H3T1 HN9   " \
"H3T2 HN9   " \
"H3T3 HN9   " \
"C3'  CN7   " \
"H3'  HN7   " \
"P3   P     " \
"O3'  ON2   " \
"O1P3 ON3   " \
"O2P3 ON3   " \
"O3P3 ON3   " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT1   " \
"HB   HA1   " \
"CG2  CT3   " \
"HG21 HA3   " \
"HG22 HA3   " \
"HG23 HA3   " \
"CG1  CT2   " \
"HG11 HA2   " \
"HG12 HA2   " \
"CD   CT3   " \
"HD1  HA3   " \
"HD2  HA3   " \
"HD3  HA3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT3   " \
"HB1  HA3   " \
"HB2  HA3   " \
"HB3  HA3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"CG   CT1   " \
"HG   HA1   " \
"CD1  CT3   " \
"HD11 HA3   " \
"HD12 HA3   " \
"HD13 HA3   " \
"CD2  CT3   " \
"HD21 HA3   " \
"HD22 HA3   " \
"HD23 HA3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"OG   OH1   " \
"HG1  H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"CG   CT2   " \
"HG1  HA2   " \
"HG2  HA2   " \
"CD   CT2   " \
"HD1  HA2   " \
"HD2  HA2   " \
"CE   CT2   " \
"HE1  HA2   " \
"HE2  HA2   " \
"NZ   NH3   " \
"HZ1  HC    " \
"HZ2  HC    " \
"HZ3  HC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"SG   S     " \
"HG1  HS    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"CG   CT2   " \
"HG1  HA2   " \
"HG2  HA2   " \
"CD   CC    " \
"OE1  O     " \
"NE2  NH2   " \
"HE21 H     " \
"HE22 H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"CG   CC    " \
"OD1  O     " \
"ND2  NH2   " \
"HD21 H     " \
"HD22 H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"CG   CT2   " \
"HG1  HA2   " \
"HG2  HA2   " \
"CD   CC    " \
"OE1  OC    " \
"OE2  OC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"CG   CC    " \
"OD1  OC    " \
"OD2  OC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT1   " \
"HB   HA1   " \
"OG1  OH1   " \
"HG1  H     " \
"CG2  CT3   " \
"HG21 HA3   " \
"HG22 HA3   " \
"HG23 HA3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"CG   CY    " \
"CD1  CA    " \
"HD1  HP    " \
"NE1  NY    " \
"HE1  H     " \
"CE2  CPT   " \
"CD2  CPT   " \
"CE3  CA    " \
"HE3  HP    " \
"CZ3  CA    " \
"HZ3  HP    " \
"CZ2  CA    " \
"HZ2  HP    " \
"CH2  CA    " \
"HH2  HP    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"CG   CT2   " \
"HG1  HA2   " \
"HG2  HA2   " \
"SD   S     " \
"CE   CT3   " \
"HE1  HA3   " \
"HE2  HA3   " \
"HE3  HA3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"CG   CA    " \
"CD1  CA    " \
"HD1  HP    " \
"CE1  CA    " \
"HE1  HP    " \
"CZ   CA    " \
"HZ   HP    " \
"CD2  CA    " \
"HD2  HP    " \
"CE2  CA    " \
"HE2  HP    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"CG   CA    " \
"CD1  CA    " \
"HD1  HP    " \
"CE1  CA    " \
"HE1  HP    " \
"CZ   CA    " \
"OH   OH1   " \
"HH   H     " \
"CD2  CA    " \
"HD2  HP    " \
"CE2  CA    " \
"HE2  HP    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"ND1  NR1   " \
"HD1  H     " \
"CG   CPH1  " \
"CE1  CPH2  " \
"HE1  HR1   " \
"NE2  NR2   " \
"CD2  CPH1  " \
"HD2  HR3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT1   " \
"HB   HA1   " \
"CG1  CT3   " \
"HG11 HA3   " \
"HG12 HA3   " \
"HG13 HA3   " \
"CG2  CT3   " \
"HG21 HA3   " \
"HG22 HA3   " \
"HG23 HA3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"CD2  CPH1  " \
"HD2  HR1   " \
"CG   CPH1  " \
"NE2  NR3   " \
"HE2  H     " \
"ND1  NR3   " \
"HD1  H     " \
"CE1  CPH2  " \
"HE1  HR2   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"ND1  NR2   " \
"CG   CPH1  " \
"CE1  CPH2  " \
"HE1  HR1   " \
"NE2  NR1   " \
"HE2  H     " \
"CD2  CPH1  " \
"HD2  HR3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA1  HB2   " \
"HA2  HB2   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"HN   H     " \
"CA   CT1   " \
"HA   HB1   " \
"CB   CT2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"CG   CT2   " \
"HG1  HA2   " \
"HG2  HA2   " \
"CD   CT2   " \
"HD1  HA2   " \
"HD2  HA2   " \
"NE   NC2   " \
"HE   HC    " \
"CZ   C     " \
"NH1  NC2   " \
"HH11 HC    " \
"HH12 HC    " \
"NH2  NC2   " \
"HH21 HC    " \
"HH22 HC    " \
"C    C     " \
"O    O     " \
"N    N     " \
"CD   CP3   " \
"HD1  HA2   " \
"HD2  HA2   " \
"CA   CP1   " \
"HA   HB1   " \
"CB   CP2   " \
"HB1  HA2   " \
"HB2  HA2   " \
"CG   CP2   " \
"HG1  HA2   " \
"HG2  HA2   " \
"C    C     " \
"O    O     " \
"N    NH3   " \
"HT1  HC    " \
"HT2  HC    " \
"HT3  HC    " \
"CA   CT1   " \
"HA   HB1   " \
"C    CC    " \
"OT1  OC    " \
"OT2  OC    " \
"CAY  CT3   " \
"HY1  HA3   " \
"HY2  HA3   " \
"HY3  HA3   " \
"CY   C     " \
"OY   O     " \
"C    C     " \
"O    O     " \
"NT   NH1   " \
"HNT  H     " \
"CAT  CT3   " \
"HT1  HA3   " \
"HT2  HA3   " \
"HT3  HA3   " \
"1CB  CT2   " \
"1SG  SM    " \
"2SG  SM    " \
"2CB  CT2   " \
"OH2  OT    " \
"OM   LP    " \
"H1   HT    " \
"H2   HT    " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"h11  ha3   " \
"h12  ha3   " \
"h13  ha3   " \
"c1   ct3   " \
"h21  ha3   " \
"h22  ha3   " \
"h23  ha3   " \
"c2   ct3   " \
"h11  ha3   " \
"h12  ha3   " \
"h13  ha3   " \
"c1   ct3   " \
"c2   ct2   " \
"h21  ha2   " \
"H22  ha2   " \
"h31  ha3   " \
"h32  ha3   " \
"h33  ha3   " \
"c3   ct3   " \
"h11  ha3   " \
"h12  ha3   " \
"h13  ha3   " \
"c1   ct3   " \
"h21  ha2   " \
"h22  ha2   " \
"c2   ct2   " \
"h31  ha2   " \
"h32  ha2   " \
"c3   ct2   " \
"h41  ha3   " \
"h42  ha3   " \
"h43  ha3   " \
"c4   ct3   " \
"CB   CT3   " \
"OG   OH1   " \
"HG1  H     " \
"HB1  HA3   " \
"HB2  HA3   " \
"HB3  HA3   " \
"CL   CT3   " \
"HL1  HA3   " \
"HL2  HA3   " \
"HL3  HA3   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CR   CT3   " \
"HR1  HA3   " \
"HR2  HA3   " \
"HR3  HA3   " \
"C1   CT3   " \
"C    CD    " \
"OM   OS    " \
"C2   CT3   " \
"O    OB    " \
"H11  HA3   " \
"H12  HA3   " \
"H13  HA3   " \
"H21  HA3   " \
"H22  HA3   " \
"H23  HA3   " \
"HA   HA1   " \
"C    CC    " \
"N    NH2   " \
"Hc   H     " \
"Ht   H     " \
"O    O     " \
"C2   CT3   " \
"C1   CD    " \
"H21  HA3   " \
"H22  HA3   " \
"H23  HA3   " \
"O2   OB    " \
"O1   OH1   " \
"HO1  H     " \
"HL   HA1   " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CR   CT3   " \
"HR1  HA3   " \
"HR2  HA3   " \
"HR3  HA3   " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH3E  " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"CG   CH2E  " \
"CD   CH2E  " \
"NE   NH1   " \
"HE   H     " \
"CZ   C     " \
"NH1  NC2   " \
"HH11 HC    " \
"HH12 HC    " \
"NH2  NC2   " \
"HH21 HC    " \
"HH22 HC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"CG   C     " \
"OD1  O     " \
"ND2  NH2   " \
"HD21 H     " \
"HD22 H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"CG   C     " \
"OD1  OC    " \
"OD2  OC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"SG   SH1E  " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"CG   CH2E  " \
"CD   C     " \
"OE1  O     " \
"NE2  NH2   " \
"HE21 H     " \
"HE22 H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"CG   CH2E  " \
"CD   C     " \
"OE1  OC    " \
"OE2  OC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH2E  " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"CG   C     " \
"ND1  NH1   " \
"HD1  H     " \
"CD2  CR1E  " \
"NE2  NR    " \
"CE1  CR1E  " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"CG   C     " \
"ND1  NR    " \
"CE1  CR1E  " \
"CD2  CR1E  " \
"NE2  NH1   " \
"HE2  H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"CG   C     " \
"CD2  CR1E  " \
"ND1  NH1   " \
"HD1  H     " \
"CE1  CR1E  " \
"NE2  NH1   " \
"HE2  H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH1E  " \
"CG2  CH3E  " \
"CG1  CH2E  " \
"CD   CH3E  " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"CG   CH1E  " \
"CD1  CH3E  " \
"CD2  CH3E  " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"CG   CH2E  " \
"CD   CH2E  " \
"CE   CH2E  " \
"NZ   NH3   " \
"HZ1  HC    " \
"HZ2  HC    " \
"HZ3  HC    " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"CG   CH2E  " \
"SD   S     " \
"CE   CH3E  " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CT    " \
"CG1  CH3E  " \
"CG2  CH3E  " \
"SG   SH1E  " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"CG   C     " \
"CD1  CR1E  " \
"CD2  CR1E  " \
"CE1  CR1E  " \
"CE2  CR1E  " \
"CZ   CR1E  " \
"C    C     " \
"O    O     " \
"N    N     " \
"CD   CH2E  " \
"CA   CH1E  " \
"CB   CH2E  " \
"CG   CH2E  " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"OG   OH1   " \
"HG   H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH1E  " \
"OG1  OH1   " \
"HG1  H     " \
"CG2  CH3E  " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"CG   C     " \
"CD2  C     " \
"CE2  C     " \
"CE3  CR1E  " \
"CD1  CR1E  " \
"NE1  NH1   " \
"HE1  H     " \
"CZ2  CR1E  " \
"CZ3  CR1E  " \
"CH2  CR1E  " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH2E  " \
"CG   C     " \
"CD1  CR1E  " \
"CE1  CR1E  " \
"CD2  CR1E  " \
"CE2  CR1E  " \
"CZ   C     " \
"OH   OH1   " \
"HH   H     " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH1E  " \
"CB   CH1E  " \
"CG1  CH3E  " \
"CG2  CH3E  " \
"C    C     " \
"O    O     " \
"FE   FE    " \
"NA   NP    " \
"NB   NP    " \
"NC   NP    " \
"ND   NP    " \
"C1A  C     " \
"CHA  CR1E  " \
"C4D  C     " \
"C1B  C     " \
"CHB  CR1E  " \
"C4A  C     " \
"C1C  C     " \
"CHC  CR1E  " \
"C4B  C     " \
"C1D  C     " \
"CHD  CR1E  " \
"C4C  C     " \
"C2A  C     " \
"CAA  CH2E  " \
"C3A  C     " \
"CMA  CH3E  " \
"CBA  CH2E  " \
"CGA  C     " \
"O1A  OC    " \
"O2A  OC    " \
"C2B  C     " \
"CMB  CH3E  " \
"C3B  C     " \
"CAB  CR1E  " \
"CBB  CH2E  " \
"C2C  C     " \
"CMC  CH3E  " \
"C3C  C     " \
"CAC  CR1E  " \
"CBC  CH2E  " \
"C2D  C     " \
"CMD  CH3E  " \
"C3D  C     " \
"CAD  CH2E  " \
"CBD  CH2E  " \
"CGD  C     " \
"O1D  OC    " \
"O2D  OC    " \
"CH3  CH3E  " \
"C    C     " \
"O    O     " \
"HA   HA    " \
"C    C     " \
"O    O     " \
"N    NH2   " \
"H1   H     " \
"H2   H     " \
"CL   CH3E  " \
"C    C     " \
"O    O     " \
"N    NH1   " \
"H    H     " \
"CA   CH3E  " \
"HT1  HC    " \
"HT2  HC    " \
"N    NH3   " \
"HT3  HC    " \
"CA   CH1E  " \
"HT1  HC    " \
"HT2  HC    " \
"N    NH3   " \
"HT3  HC    " \
"CA   CH2E  " \
"HT1  HC    " \
"HT2  HC    " \
"N    NH3   " \
"CD   CH2E  " \
"CA   CH1E  " \
"C    C     " \
"OT1  OC    " \
"OT2  OC    " \
"O1    OM   " \
"O2    OM   " \
"C    CM    " \
"O    OM    " \
"C    CM    " \
"O    OM    " \
"C2   CH3E  " \
"C1   CH2E  " \
"O    OH1   " \
"H    H     " \
"OX2  OH2   " \
"HX1  H     " \
"HX2  H     " \
"LX1  LP    " \
"LX2  LP    " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"S    S     " \
"O1   OC    " \
"O2   OC    " \
"O3   OC    " \
"O4   OC    " \
"C    CH3E  " \
"O    OH1   " \
"H    H     " \
"1CB  CH2E  " \
"1SG  S     " \
"2SG  S     " \
"2CB  CH2E  " \
"CG   C     " \
"ND1  NR    " \
"CE1  CR1E  " \
"CD2  CR1E  " \
"NE2  NH1   " \
"HE2  H     " \
"P    P     " \
"O1P  O2    " \
"O2P  O2    " \
"O5'  OS    " \
"C5'  C2    " \
"C4'  CH    " \
"O4'  OSS   " \
"C1'  CH    " \
"N9   NS    " \
"C4   CB    " \
"N3   NC    " \
"C2   CA    " \
"N2   N2    " \
"H21  H2    " \
"H22  H2    " \
"N1   NA    " \
"H1   H     " \
"C6   C     " \
"O6   O     " \
"C5   CB    " \
"N7   NB    " \
"C8   CE    " \
"C2'  CH    " \
"O2'  OH    " \
"H2'  HO    " \
"C3'  CH    " \
"O3'  OS    " \
"P    P     " \
"O1P  O2    " \
"O2P  O2    " \
"O5'  OS    " \
"C5'  C2    " \
"C4'  CH    " \
"O4'  OSS   " \
"C1'  CH    " \
"N9   NS    " \
"C4   CB    " \
"N3   NC    " \
"C2   CE    " \
"N1   NC    " \
"C6   CA    " \
"N6   N2    " \
"H61  H2    " \
"H62  H2    " \
"C5   CB    " \
"N7   NB    " \
"C8   CE    " \
"C2'  CH    " \
"O2'  OH    " \
"H2'  HO    " \
"C3'  CH    " \
"O3'  OS    " \
"P    P     " \
"O1P  O2    " \
"O2P  O2    " \
"O5'  OS    " \
"C5'  C2    " \
"C4'  CH    " \
"O4'  OSS   " \
"C1'  CH    " \
"N1   NS    " \
"C6   CF    " \
"C2   C     " \
"O2   O     " \
"N3   NC    " \
"C4   CA    " \
"N4   N2    " \
"H41  H2    " \
"H42  H2    " \
"C5   CF    " \
"C2'  CH    " \
"O2'  OH    " \
"H2'  HO    " \
"C3'  CH    " \
"O3'  OS    " \
"P    P     " \
"O1P  O2    " \
"O2P  O2    " \
"O5'  OS    " \
"C5'  C2    " \
"C4'  CH    " \
"O4'  OSS   " \
"C1'  CH    " \
"N1   NS    " \
"C6   CF    " \
"C2   C     " \
"O2   O     " \
"N3   NA    " \
"H3   H     " \
"C4   C     " \
"O4   O     " \
"C5   CS    " \
"C5A  C3    " \
"C2'  CH    " \
"O2'  OH    " \
"H2'  HO    " \
"C3'  CH    " \
"O3'  OS    " \
"P    P     " \
"O1P  O2    " \
"O2P  O2    " \
"O5'  OS    " \
"C5'  C2    " \
"C4'  CH    " \
"O4'  OSS   " \
"C1'  CH    " \
"N1   NS    " \
"C6   CF    " \
"C2   C     " \
"O2   O     " \
"N3   NA    " \
"H3   H     " \
"C4   C     " \
"O4   O     " \
"C5   CF    " \
"C2'  CH    " \
"O2'  OH    " \
"H2'  HO    " \
"C3'  CH    " \
"O3'  OS    " \
"O4'  OST   " \
"C2'  C2    " \
"C1   C3    " \
"O1   OS    " \
"P    P     " \
"O1P  O2    " \
"O2P  O2    " \
"O2   OS    " \
"C2   C3    " \
"C1   C3    " \
"C2   C2    " \
"O1   OS    " \
"P    P     " \
"O1P  O2    " \
"O2P  O2    " \
"O2   OS    " \
"C3   C2    " \
"C4   C3    " \
"C5'  C3    " \
"C4'  CH    " \
"O4'  OSS   " \
"C1'  CH    " \
"N1   NH2E  " \
"C2'  CH    " \
"O2'  OH    " \
"C3'  CH    " \
"O3'  OS    " \
"C5'  C3    " \
"C4'  CH    " \
"O4'  OST   " \
"C1'  CH    " \
"N1   NH2E  " \
"C2'  C2    " \
"C3'  CH    " \
"O3'  OS    " \
"C1    C3   " \
"C2    C2   " \
"C3    C2   " \
"C4    C2   " \
"C5    C2   " \
"C6    C2   " \
"C7    C2   " \
"C8    C3   " \
"N1   NA    " \
"C6   CF    " \
"H1   H     " \
"N9   NA    " \
"C4   CB    " \
"H9   H     " \
"H5T  HO    " \
"O5'  OH    " \
"C5'  C2    " \
"H5T  HO    " \
"O5T  OH    " \
"C3'  CH    " \
"O3'  OH    " \
"H3T  HO    " \
"1O3' OS    " \
"2P   P     " \
"2O1P O2    " \
"2O2P O2    " \
"2O5' OS    " \
"OH2  OH2   " \
"H1   H     " \
"H2   H     " \
"NA  SD     " \
"NA  SD     " \
"N1   NH3   " \
"C2   C2    " \
"C3   C2    " \
"C4   C2    " \
"N5   NH3   " \
"C6   C2    " \
"C7   C2    " \
"C8   C2    " \
"C9   C2    " \
"NA   NH3   " \
"CB   C2    " \
"CC   C2    " \
"CD   C2    " \
"NE   NH3   " \
"H11  H     " \
"H12  H     " \
"H13  H     " \
"H51  H     " \
"H52  H     " \
"HA1  H     " \
"HA2  H     " \
"HE1  H     " \
"HE2  H     " \
"HE3  H     " \
"MG MG  2.  " \
"ST2     .  " \
"OX2  OH2   " \
"HX1  H     " \
"HX2  H     " \
"LX1  LP    " \
"LX2  LP    " \
"OH2  OT    " \
"H1   HT    " \
"H2   HT    " \
"C5   C2    " \
"C6   C2    " \
"C5   CB    " \
"C7   C3    " \
"N7   NB    " \
"C8   CE    " \
"N2   N2    " \
"C2B  C3    " \
"C2A  C3    " \
"N2   N2    " \
"H21  H2    " \
"C2A  C3    " \
"C2'  CH    " \
"O2'  OH    " \
"C2A  C3    " \
"C5   CS    " \
"C6   CF    " \
"N1   NA    " \
"H11  H     " \
"C5   CS    " \
"C5A  C3    " \
"N1   NA    " \
"C1   C3    " \
"C6   CH    " \
"C3   C3    " \
"N3   NA    " \
"C2   CB    " \
"N2   NB    " \
"C11  CA    " \
"C10  C3    " \
"C12  CA    " \
"N1   NA    " \
"C13  C2    " \
"C14  C2    " \
"C15  CH    " \
"C16  C     " \
"O18  OS    " \
"C19  C3    " \
"O17  O     " \
"N20  NA    " \
"H20  H     " \
"C21  C     " \
"O23  O     " \
"O22  OS    " \
"C24  C3    " \
"C2   CA    " \
"C6   CE    " \
"N2   N2    " \
"H21  H2    " \
"H22  H2    " 
} {
 set data [ split $data " " ]
 set i 0
 foreach value $data {
   if { $value > " " } { incr i; set val($i) $value }
   if { $i == 2 } {
     set class $val(2)
     if { [ info exists mass($class) ] } {
       set mass($val(1)) $mass($class)
     }
   }
 }
}




set natoms 0
set molmass 0.
foreach line $pdb_file {
  if { [ string range $line 0 3 ] == "ATOM" |
       [ string range $line 0 5 ] == "HETATM" } {
    incr natoms
    set at [ string trim [ string range $line 12 15 ] ]
    if { [ info exists mass($at) ] == 0 } { 
      if { [ info exists mass([ string range $at 0 0 ]) ] } {
        set mass($at) $mass([ string range $at 0 0 ])  
        puts -nonewline " Unsure about mass of element $at: $mass($at)." 
        puts -nonewline " Is this correct? (y/n)"
        flush stdout; gets stdin correct
        if { $correct != "y" & $correct != "yes" & $correct != "Y" & $correct != "YES" } {
          puts -nonewline "Type the mass of this atom:"
          flush stdout; gets stdin mass($at)
        }
      } else {
        puts "Could not find mass for atom type: $at"
        puts -nonewline "Type the mass of this atom:"
        flush stdout; gets stdin mass($at)
      }
    }
    set atommass $mass($at) 
    set molmass [ expr $molmass + $atommass ]
  }
}

puts " -------------------------------------------------------"
puts "  Molar mass of structure: $molmass"
puts " -------------------------------------------------------"

# Total volume of the box

if { $density == "none" } { set density 1. }
set volume [ expr $x_length * $y_length * $z_length ]

# We use the approximation that the volume occupied by each type of molecule
# in the system is proportional to its molar mass, therefore:

set water_molar_mass 18.
set n_water [ format %7.0f "[ expr ( $volume*($density*0.602) - $molmass ) / $water_molar_mass ]" ]
puts "  Number of water molecules to be put: $n_water "

# Fraction of the volume occupied by water

set water_vol [ expr $n_water*$water_molar_mass/($n_water*$water_molar_mass + $molmass) ]
set water_vol [ expr $water_vol * $volume ]

# Computing the required number of ions for an approximate concentration
# of 0.16 mol/L (approximate physiological concentration on water)

puts "  Total volume: [ format %8.2f $volume ] A^3"
puts "  Volume occupied by water: [ format %8.2f "$water_vol" ] A^3 "

# Set the number of ions of each type

if { $noions == "true" } {
  set n_chloride 0
  set n_sodium 0
} else {
  if { $charge <= 0 } { 
    set n_sodium [ format %8.0f [ expr 0.16 * 0.000602 * $water_vol ] ]
    if { $n_sodium > [ expr -1.*$charge ] } { 
      set n_chloride [ expr $n_sodium + $charge ] 
    } else {
      set n_sodium [ expr -1.*$charge ]
      set n_chloride 0
    }
  }
  if { $charge > 0 } {
    set n_chloride [ format %8.0f [ expr 0.16 * 0.000602 * $water_vol ] ]
    if { $n_chloride > $charge } {
      set n_sodium [ expr $n_chloride - $charge ]
    } else {
      set n_chloride $charge
      set n_sodium 0
    } 
  }
}

puts "  Number of Sodium ions to be put: [ string trim $n_sodium ]"
puts "  Number of Chloride ions to be put: [ string trim  $n_chloride ]"

# Minimum and maximum coordinates of the box

set xmin [ expr -1.*$x_length / 2. ]
set xmax [ expr     $x_length / 2. ]
set ymin [ expr -1.*$y_length / 2. ]
set ymax [ expr     $y_length / 2. ]
set zmin [ expr -1.*$z_length / 2. ]
set zmax [ expr     $z_length / 2. ]

#
# Write packmol input file
#

set packmol_input_file [ open ./$packmol_input w ]

puts $packmol_input_file "
tolerance 2.0
filetype pdb
output $output

structure $pdb_file_name
  number 1
  center
  fixed 0. 0. 0. 0. 0. 0.
end structure

structure WATER.pdb
  number $n_water
  inside box $xmin $ymin $zmin $xmax $ymax $zmax
end structure
"

if { $n_sodium > 0 } {
  puts $packmol_input_file "
structure SODIUM.pdb
  number $n_sodium
  inside box $xmin $ymin $zmin $xmax $ymax $zmax
end structure
  "
}
  
if { $n_chloride > 0 } {
  puts $packmol_input_file "
structure CHLORIDE.pdb
  number $n_chloride
  inside box $xmin $ymin $zmin $xmax $ymax $zmax
end structure
  "
}
close $packmol_input_file

puts "  Wrote packmol input. "

#
# This are the temporary pdb file that will be temporaritly created
#

set water_file [ open ./WATER.pdb w ]
puts $water_file \
"HETATM    1  H1  TIP3    1       9.626   6.787  12.673                       H"
puts $water_file \
"HETATM    2  H2  TIP3    1       9.626   8.420  12.673                       H"
puts $water_file \
"HETATM    3  OH2 TIP3    1      10.203   7.604  12.673                       O"
close $water_file


set sod_file [ open ./SODIUM.pdb w ]
puts $sod_file  \
"HETATM    1 SOD  SOD    2        0.000   0.000   0.000                      Na"
close $sod_file

set cl_file [ open ./CHLORIDE.pdb w ]
puts $cl_file  \
"HETATM    1 CLA  CLA    2        0.000   0.000   0.000                      Cl"
close $cl_file

# Run packmol 

puts " -------------------------------------------------------"
puts "  Use these lengths for periodic boundary conditions: "
puts "  x: [ expr $x_length + 1. ]"
puts "  y: [ expr $y_length + 1. ]"
puts "  z: [ expr $z_length + 1. ]"
puts " -------------------------------------------------------"
puts " -------------------------------------------------------"
puts "  Now, run packmol with: packmol < $packmol_input       "
puts " -------------------------------------------------------"
puts "  The solvated file will be: $output "
puts " -------------------------------------------------------"


