(****************************************************************************)
(*                           the diy toolsuite                              *)
(*                                                                          *)
(* Jade Alglave, University College London, UK.                             *)
(* Luc Maranget, INRIA Paris-Rocquencourt, France.                          *)
(*                                                                          *)
(* Copyright 2010-present Institut National de Recherche en Informatique et *)
(* en Automatique and the authors. All rights reserved.                     *)
(*                                                                          *)
(* This software is governed by the CeCILL-B license under French law and   *)
(* abiding by the rules of distribution of free software. You can use,      *)
(* modify and/ or redistribute the software under the terms of the CeCILL-B *)
(* license as circulated by CEA, CNRS and INRIA at the following URL        *)
(* "http://www.cecill.info". We also give a copy in LICENSE.txt.            *)
(****************************************************************************)

open Printf

let comment = "#"

module Make(O:Arch_litmus.Config)(V:Constant.S) = struct
  include X86_64Base
  module V = V
  module FaultType = FaultType.No

  let reg_to_string r = match r with
    | RIP -> "%rip"
    | CS -> "%cs"
(* We always want the 64 bits version of the register because the smaller version are created by GCC*)
    | Ireg (r, _) -> "%" ^ X86_64Base.reg64_string r
    | XMM xmm -> "%" ^ X86_64Base.xmm_string xmm
    | Internal i -> sprintf "i%i" i
    | _ -> assert false

  let error _ _ = false
  let warn _ _ = false

  include
    ArchExtra_litmus.Make(O)
      (struct
        module V = V

        type arch_reg = reg
        let arch = `X86_64
        let forbidden_regs = []
        let pp_reg = pp_reg
        let reg_compare = reg_compare
        let reg_to_string = reg_to_string
        let internal_init _ _ = None

        let do_reg_class =
          function
          (* as some instructions have eax as implicit argument,
             we must allocate our EAX to machine %eax
          | Ireg RAX -> "a"
          (* esi and edi implicit for MOVSD *)
          | Ireg RSI -> "S"
          | Ireg RDI -> "D" *)
          | Ireg (AX, _) -> "a"
          | Ireg (_, R8bH) -> "Q"
          | XMM _ -> "x"
          | _ -> "r"

        let reg_class r = "=&" ^ do_reg_class r 

        let reg_class_stable init r =
          (if init then "+" else "=&")
          ^ do_reg_class r
        let comment = comment
      end)

  let features = []
  let nop =  I_NOP

  include HardwareExtra.No

  module GetInstr = GetInstr.No(struct type instr=instruction end)

end
